# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet # pylint: disable-msg=W0611
import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from RoutingIntfUtils import allIpIntfNames
import Tac

# Since "tcp mss ceiling" set's attributes in both the ipv4 and ipv6 config,
# we need both the ipv4 and ipv6 config while reconstructing the original
# command. Hence this is moved to different file, which differs from other
# commands
IntfConfigMode.addCommandSequence( 'Ira.tcpMss', after=[ 'Arnet.intf',
   'Arnet.l2l3barrier', 'Ira.ipIntf', 'Ira.intf6' ] )

#-----------------------------------------------------------------------
# Saver methods
#-----------------------------------------------------------------------
@CliSave.saver( 'Ip::Config', 'ip/config',
                requireMounts=( 'interface/config/all', 'interface/status/all',
                                'ip6/config', ) )
def saveTcpMssCeilingConfig( entity, root, requireMounts, options ):
   # Interface config
   if options.saveAllDetail:
      cfgIntfNames = allIpIntfNames( requireMounts, includeEligible=True )
   elif options.saveAll:
      # IP configuration is allowed on switchport interfaces as well.
      # Display config on all ip interfaces and switchports with non-default
      # config.
      ipIntfNames = allIpIntfNames( requireMounts )
      cfgIntfNames = set( ipIntfNames )
      cfgIntfNames.update( entity.ipIntfConfig )
   else:
      cfgIntfNames = entity.ipIntfConfig

   for intfName in cfgIntfNames:
      if options.intfFilter and intfName not in options.intfFilter:
         continue
      intfConfig = entity.ipIntfConfig.get( intfName )
      ip6IntfConfig = requireMounts[ 'ip6/config' ].intf.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Ip::IpIntfConfig', intfName )
            intfConfig.l3Config = Tac.newInstance( 'L3::Intf::Config', intfName )
         else:
            continue

      if not ip6IntfConfig:
         ip6IntfConfig = Tac.newInstance( 'Ip6::IntfConfig', intfName )

      saveIntfTcpMssConfig( intfConfig, root, options, ip6IntfConfig )

def saveTcpMssConfig( saveAll, config4, config6 ):
   ingress = config4.maxMssIngress or config6.maxMssIngress
   egress = config4.maxMssEgress or config6.maxMssEgress
   bidirectional = ingress and egress

   ipv4MaxMss = max( config4.maxMssIngress, config4.maxMssEgress )
   ipv6MaxMss = max( config6.maxMssIngress, config6.maxMssEgress )

   ipv4MaxMssStr = " ipv4 " + str( ipv4MaxMss ) if ipv4MaxMss else ''
   ipv6MaxMssStr = " ipv6 " + str( ipv6MaxMss ) if ipv6MaxMss else ''

   ingressOrEgressStr = ''
   if bidirectional:
      ingressOrEgressStr = ''
   elif ingress:
      ingressOrEgressStr = ' ingress'
   elif egress:
      ingressOrEgressStr = ' egress'

   cmd = None
   if ipv4MaxMss or ipv6MaxMss:
      cmd = 'tcp mss ceiling' + ipv4MaxMssStr + ipv6MaxMssStr + ingressOrEgressStr
   elif saveAll:
      cmd = 'no tcp mss ceiling'
   return cmd

def saveIntfTcpMssConfig( entity, root, options, ip6Config ):

   saveAll = options.saveAll
   if entity.intfId.startswith( "Internal" ):
      # Internal interface configuration. Abort
      # BUG944 - need a more general way of doing this
      return

   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Ira.tcpMss' ]
   intfId = entity.intfId
   if ( not intfId.startswith( "Loopback" ) and
        not intfId.startswith( "Management" ) and
        not intfId.startswith( "Vxlan" ) ):
      cmd = saveTcpMssConfig( saveAll, entity, ip6Config )
      if cmd is not None:
         cmds.addCommand( cmd )
