# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the CLI save plugin for all CLI defined L1 card profiles.'''

from CliMode.L1CardProfileDefinition import CardProfileDefinitionMode
from Intf import IntfRange
from Intf.CardIntfSlotRange import AllIntfSlotTypes
from Plugins import SkipPluginModule
from Toggles import L1ProfileToggleLib
from TypeFuture import TacLazyType
from collections import defaultdict
import CliSave

if not L1ProfileToggleLib.toggleL1ProfileCustomModuleSupportEnabled():
   raise SkipPluginModule( 'L1 Profiles Custom Module Toggle Not Enabled' )

MountConstants = TacLazyType( 'L1Profile::MountConstants' )

class CardProfileDefinitionConfigMode( CardProfileDefinitionMode, CliSave.Mode ):
   def __init__( self, param ):
      CardProfileDefinitionMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( CardProfileDefinitionConfigMode )

CardProfileDefinitionConfigMode.addCommandSequence( 'CardProfileDef' )

@CliSave.saver( typename='L1Profile::CardProfileDir',
                pathPrefix=MountConstants.cardProfileLibraryCliDirPath(),
                attrName='cardProfile' )
def saveL1ProfileDefinition( cardProfileLibraryCliDir,
                             root,
                             requireMounts,
                             options ):
   for desc, cardProfile in cardProfileLibraryCliDir.cardProfile.items():
      mode = root[ CardProfileDefinitionConfigMode ].getOrCreateModeInstance(
         desc.name )
      cmds = mode[ 'CardProfileDef' ]
      if cardProfile.description:
         cmds.addCommand( f'description {cardProfile.description}' )

      # In order to not bloat the user config, interface slot ranges need to be
      # compressed. For example:
      #  port Ethernet1 profile X -+
      #                            | -> port Ethernet1-2 profile X
      #  port Ethernet2 profile X -+
      #
      # To aid processing, ports are first grouped together by their configured
      # interface slot profiles.
      slotProfileToIntfSlotNames = defaultdict( set )
      for intfSlot, intfSlotProfileDesc in cardProfile.intfSlotProfile.items():
         slotProfileToIntfSlotNames[ intfSlotProfileDesc.name ].add(
            f'{intfSlot.intfSlotPrefix}{intfSlot.intfSlotId}' )

      for intfSlotProfileName, intfSlotNames in sorted(
            slotProfileToIntfSlotNames.items() ):
         # The abuse of the interface range infra continues! This time, the interface
         # range infra is used to compress the interface slot ranges. This only works
         # because presently every interface slot ID can be parsed as a valid
         # representation of an interface ID.
         intfSlotNameRanges = IntfRange.intfListToCanonical(
            intfSlotNames, shortName=False, intfTypes=AllIntfSlotTypes )
         for intfSlotNameRange in intfSlotNameRanges:
            cmds.addCommand(
               f'port {intfSlotNameRange} profile {intfSlotProfileName}' )
