# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements saving the Lag CLI, including Port-Channel interfaces
# and interface-specific configuration.
#-------------------------------------------------------------------------------
import CliSave
from CliSavePlugin import IntfCliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliMode.LagLoadBalanceMode import LBPoliciesMode
import LacpConstants
import re
import Tac
import LacpLib
from LagLib import speedUnitStrFromEnum
from EthIntfLib import ethPhyIntfConfigIter
from Toggles.LagToggleLib import toggleMaxLinksPreemptivePriorityEnabled
from TypeFuture import TacLazyType
import Ethernet

isRecirc = Tac.Type( "Arnet::PortChannelIntfId" ).isRecirc
MinBandwidthTimeoutBase = Tac.Type( "Lag::MinBandwidthTimeoutBase" )
LacpTimeoutMultiplier = Tac.Type( "Lacp::TimeoutMultiplier" )
LagSpeedValue = TacLazyType( "Interface::LagSpeedValue" )
PortChannelIntfId = TacLazyType( "Arnet::PortChannelIntfId" )
FabricChannelIntfId = TacLazyType( "Arnet::FabricChannelIntfId" )

CliSave.GlobalConfigMode.addCommandSequence( 'Lag.global',
                                             before=[ IntfConfigMode ] )

IntfConfigMode.addCommandSequence( 'Lag.lagIntf', after=[ 'Arnet.intf' ] )
IntfConfigMode.addCommandSequence( 'Lag.lagConfig', after=[ 'Ebra.switchport' ] )

# LoadBalance policies mode
class LbPoliciesConfigMode( LBPoliciesMode, CliSave.Mode ):

   def __init__( self, param ):
      LBPoliciesMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( LbPoliciesConfigMode )

@CliSave.saver( 'Interface::EthLagIntfConfig', 'interface/config/eth/lag',
                attrName='intfConfig',
                requireMounts=( 'interface/status/all', 'bridging/hwcapabilities',
                                'routerMac/hwCapability' ) )
def saveEthLagIntfConfig( entity, root, requireMounts, options ):
   # Save the baseclass (Arnet::IntfConfig) attributes.
   IntfCliSave.saveIntfConfig( entity, root, requireMounts, options )

   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Lag.lagIntf' ]
   saveAll = options.saveAll

   if entity.minLinks != entity.minLinksDefault:
      cmds.addCommand( 'port-channel min-links %d' % entity.minLinks )
   elif saveAll:
      cmds.addCommand( 'no port-channel min-links' )

   if toggleMaxLinksPreemptivePriorityEnabled():
      if entity.maxLinks is not None:
         cmds.addCommand( 'port-channel max-links %d' % entity.maxLinks )
      elif saveAll:
         cmds.addCommand( 'no port-channel max-links' )

      if entity.priorityEnforced:
         cmds.addCommand( 'port-channel member priority preemptive' )
      elif saveAll:
         cmds.addCommand( 'no port-channel member priority' )

   if entity.mixedSpeed != False: # pylint: disable=singleton-comparison
      cmds.addCommand( 'port-channel speed mixed' )
   elif saveAll:
      cmds.addCommand( 'no port-channel speed mixed' )

   if entity.minSpeed.speedValue != LagSpeedValue.defaultSpeed:
      speedUnitStr = speedUnitStrFromEnum( entity.minSpeed.speedUnit )
      cmds.addCommand( 'port-channel speed minimum %d %s' %
                       ( entity.minSpeed.speedValue, speedUnitStr ) )
   elif saveAll:
      cmds.addCommand( 'no port-channel speed minimum' )

   if entity.fallback == 'fallbackStatic':
      cmds.addCommand( 'port-channel lacp fallback static' )
   elif entity.fallback == 'fallbackIndividual':
      cmds.addCommand( 'port-channel lacp fallback individual' )
   elif saveAll:
      cmds.addCommand( 'no port-channel lacp fallback' )

   if entity.fallbackTimeout != entity.fallbackTimeoutDefault or saveAll:
      cmds.addCommand( 'port-channel lacp fallback timeout %d' % (
                        entity.fallbackTimeout ) )
      
   if entity.l2Mtu != 0:
      cmds.addCommand( 'l2 mtu %u' % entity.l2Mtu )
   elif saveAll:
      cmds.addCommand( 'no l2 mtu' )

   if entity.l2Mru != 0:
      cmds.addCommand( 'l2 mru %u' % entity.l2Mru )
   elif saveAll:
      cmds.addCommand( 'no l2 mru' )

   if isRecirc( entity.intfId ):
      if len( entity.recircFeature ) > 0:
         vxlanVal = Tac.enumValue(
            "Lag::Recirc::RecircFeature", "vxlan" )
         teleInbVal = Tac.enumValue(
            "Lag::Recirc::RecircFeature", "telemetryinband" )
         openFlowVal = Tac.enumValue(
            "Lag::Recirc::RecircFeature", "openflow" )
         cpumirrorVal = Tac.enumValue(
            "Lag::Recirc::RecircFeature", "cpumirror" )
   
         if vxlanVal in entity.recircFeature:
            cmds.addCommand( 'switchport recirculation features %s' %
                             ( 'vxlan' ) )
         elif teleInbVal in entity.recircFeature:
            cmds.addCommand( 'switchport recirculation features %s' %
                             ( 'telemetry inband' ) )
         elif openFlowVal in entity.recircFeature:
            cmds.addCommand( 'switchport recirculation features %s' %
                             ( 'openflow' ) )
         elif cpumirrorVal in entity.recircFeature:
            cmds.addCommand( 'switchport recirculation features %s' %
                             ( 'cpu-mirror' ) )
      elif saveAll:
         cmds.addCommand( 'no switchport recirculation features' )

   if entity.lacpLoopEnable:
      cmds.addCommand( 'port-channel lacp loopback' )
   elif saveAll:
      cmds.addCommand( 'no port-channel lacp loopback' )

   hwCaps = requireMounts[ 'routerMac/hwCapability' ]
   if hwCaps.intfRouterMacSupported:
      if entity.routerMacConfigured:
         if entity.addr == '00:00:00:00:00:00':
            rmac = 'physical'
         else:
            rmac = Ethernet.convertMacAddrCanonicalToDisplay( entity.addr )
         cmds.addCommand( 'mac-address router %s' % rmac )
      elif saveAll:
         cmds.addCommand( 'no mac-address router' )

   # # Display defaults when this command is supported.
   # if entity.maxBundle != entity.maxBundleDefault:
   #    cmds.addCommand( 'lacp max-bundle %d' % entity.maxBundle )
   # elif saveAll:
   #    cmds.addCommand( 'no lacp max-bundle' )

@CliSave.saver( 'Lag::Input::Config', 'lag/input/config/cli',
                requireMounts = ( 'interface/config/eth/phy/slice',
                                  'lag/lacp/input/config/cli' ) )
def saveLagConfig( entity, root, requireMounts, options ):

   cmds = root[ 'Lag.global' ]
   # Get the list of interfaces based saveAll type
   ethPhyIntfConfigSliceDir = requireMounts[ 'interface/config/eth/phy/slice' ]

   allEthPhyIntfConfigs = ethPhyIntfConfigIter( ethPhyIntfConfigSliceDir )
   
   if entity.memberStatusLogEnabled == False: # pylint: disable=singleton-comparison
      cmds.addCommand( 'no logging event port-channel member-status global' )
   elif options.saveAll:
      cmds.addCommand( 'logging event port-channel member-status global' )

   if entity.minLinksTimeoutBase != MinBandwidthTimeoutBase.timeoutDefault:
      cmds.addCommand( 'port-channel min-links review interval %d' \
                       % entity.minLinksTimeoutBase )
   elif options.saveAll:
      cmds.addCommand( 'no port-channel min-links review interval' )

   if entity.minSpeedTimeoutBase != MinBandwidthTimeoutBase.timeoutDefault:
      cmds.addCommand( 'port-channel speed minimum review interval %d seconds' \
                       % entity.minSpeedTimeoutBase )
   elif options.saveAll:
      cmds.addCommand( 'no port-channel speed minimum review interval' )

   if options.saveAll:
      cfgIntfNames = []

      cfgIntfNames = [ intfName for intfName in allEthPhyIntfConfigs
                       if LacpLib.intfSupportsLag( intfName ) ]
   else:
      cfgIntfNames = entity.phyIntf

   for intfName in cfgIntfNames:
      intfConfig = entity.phyIntf.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Lag::EthPhyIntfLagConfig', intfName )
         else:
            continue
      # save intfConfig
      saveEthPhyIntfLagConfig( intfConfig, root, requireMounts, options )

   if options.saveAll:
      # EosImage/CliModeTests.py needs this
      root[ LbPoliciesConfigMode ].getSingletonInstance()

def saveEthPhyIntfLagConfig( entity, root, requireMounts, options ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Lag.lagConfig' ]
   saveAll = options.saveAll

   lacpCliConfig = requireMounts[ 'lag/lacp/input/config/cli' ]
   # save lacp port-id before channel-group cli command
   if entity.portId != LacpConstants.LacpPortIdDefault:
      cmds.addCommand( 'lacp port-id %d' % entity.portId )
   elif saveAll:
      cmds.addCommand( 'no lacp port-id' )

   if entity.lag:
      isRecircLag = isRecirc( entity.lag.intfId )
      isPoLag = PortChannelIntfId.isPo( entity.lag.intfId )
      isFcLag = FabricChannelIntfId.isFabricChannelIntfId( entity.lag.intfId )
      if isPoLag:
         m = re.match( r'Port-Channel(\d+)', entity.lag.intfId )
         if m:
            modemap = { 'lacpModeOff': 'on',
                        'lacpModeActive': 'active', 'lacpModePassive': 'passive' }
            cmds.addCommand( 'channel-group {} mode {}'.format(
                             m.group( 1 ), modemap[ entity.mode ] ) )
      elif isRecircLag:
         m = re.match( r'Recirc-Channel(\d+)', entity.lag.intfId )
         if m:
            cmds.addCommand( 'channel-group recirculation %s' % m.group( 1 ) )
      elif isFcLag:
         memberId = FabricChannelIntfId.memberId( entity.lag.intfId )
         channelId = FabricChannelIntfId.channelId( entity.lag.intfId )
         cmds.addCommand( f'channel-group fabric {memberId}/{channelId}' )
   elif saveAll:
      cmds.addCommand( 'no channel-group' )

   if entity.timeout == 'lacpShortTimeout':
      cmds.addCommand( 'lacp timer fast' )
   elif saveAll:
      cmds.addCommand( 'lacp timer normal' )
   if entity.timeoutMult != LacpTimeoutMultiplier.defaultValue:
      cmds.addCommand( 'lacp timer multiplier %d' % entity.timeoutMult )
   elif saveAll:
      cmds.addCommand( 'lacp timer multiplier %d'
                       % LacpTimeoutMultiplier.defaultValue )
   if entity.priority != LacpConstants.PortPriorityDefault or saveAll:
      cmds.addCommand( 'lacp port-priority %d' % entity.priority )

   rateLimitIntfEnable = lacpCliConfig.rateLimitIntfEnable
   if entity.intfId in rateLimitIntfEnable:
      if rateLimitIntfEnable[ entity.intfId ]:
         cmds.addCommand( 'lacp rate-limit' )
      else:
         cmds.addCommand( 'no lacp rate-limit' )
   elif saveAll:
      cmds.addCommand( 'default lacp rate-limit' )

   if entity.fwdTxEnabled == False: # pylint: disable=singleton-comparison
      cmds.addCommand( 'forwarding port-channel transmit disabled' )
   elif saveAll:
      cmds.addCommand( 'no forwarding port-channel transmit disabled' )

@CliSave.saver( 'Lacp::CliConfig', 'lag/lacp/input/config/cli' )
def saveLacpConfig( entity, root, requireMounts, options ):
   cmds = root[ 'Lag.global' ]

   if entity.priority != LacpConstants.SystemPriorityDefault or options.saveAll:
      cmds.addCommand( 'lacp system-priority %d' % entity.priority )
   if entity.portIdRange != Tac.Value( 'Lacp::PortIdRange' ):
      cmds.addCommand( 'lacp port-id range %d %d' % \
                ( entity.portIdRange.portIdMin, entity.portIdRange.portIdMax ) )
   elif options.saveAll:
      cmds.addCommand( 'no lacp port-id range' )
   if entity.enableAudit != False:  # pylint: disable=singleton-comparison
      # don't show enableAudit if False (unset) even if saveAll --- it's hidden!
      cmds.addCommand( 'lacp audit' )
   if not entity.rateLimitEnable:
      cmds.addCommand( 'no lacp rate-limit default' )
   elif options.saveAll:
      cmds.addCommand( 'lacp rate-limit default' )

@CliSave.saver( 'Lag::Input::LacpOverrideDir', 'lag/input/lacpoverride/cli' )
def saveLacpOverride( entity, root, requireMounts, options ):
   cmds = root[ 'Lag.global' ]
   systemIds = entity.systemId

   for intfId in systemIds:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
      cmds = mode[ 'Lag.lagIntf' ]
      ethAddr = Tac.Value( 'Arnet::EthAddr', 
                           stringValue=systemIds[ intfId ] )
      cmds.addCommand( 'lacp system-id %s' % ethAddr.displayString )
