#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliSave import escapeFormatString
from CliSavePlugin import Management
from CliSavePlugin.NetworkCliSave import networkConfigCmdSeq
from CliSavePlugin.Security import mgmtSecurityConfigPath
from CliSavePlugin.Security import SecurityConfigMode
import Tac
from CliMode.Ldap import ( ServerConfigModeBase, GroupPolicyConfigModeBase,
                           LdapServerGroupModeBase )
import LdapConstants
from IpLibConsts import DEFAULT_VRF
import ReversibleSecretCli

DEFAULT_ROLE_PRIVLEVEL = 1

class LdapConfigSaveMode( Management.MgmtConfigMode ):
   def __init__( self, param ):
      Management.MgmtConfigMode.__init__( self, "ldap" )

   def skipIfEmpty( self ):
      return True

class ServerConfigSaveMode( ServerConfigModeBase, CliSave.Mode ):
   def __init__( self, param ):
      ServerConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )
      if param == 'defaults':
         self.index_ = -1
      else:
         self.index_ = param.index

   def skipIfEmpty( self ):
      return self.param_ == 'defaults'

   def instanceKey( self ):
      return self.index_

   @classmethod
   def useInsertionOrder( cls ):
      # because `instanceKey` is overridden with the index of the server
      return True

class GroupPolicyConfigSaveMode( GroupPolicyConfigModeBase, CliSave.Mode ):
   def __init__( self, param ):
      GroupPolicyConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class AaaLdapServerGroupConfigSaveMode( LdapServerGroupModeBase, CliSave.Mode ):
   def __init__( self, param ):
      LdapServerGroupModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( LdapConfigSaveMode,
                                       after=[ SecurityConfigMode ] )
LdapConfigSaveMode.addCommandSequence( 'Mgmt.ldap' )

LdapConfigSaveMode.addChildMode( ServerConfigSaveMode )

ServerConfigSaveMode.addCommandSequence( 'ldap.server' )

LdapConfigSaveMode.addChildMode( GroupPolicyConfigSaveMode,
                                 after=[ ServerConfigSaveMode ] )
GroupPolicyConfigSaveMode.addCommandSequence( 'ldap.groupPolicy' )

CliSave.GlobalConfigMode.addChildMode( AaaLdapServerGroupConfigSaveMode,
                                       before=[ 'Aaa.global' ] )
AaaLdapServerGroupConfigSaveMode.addCommandSequence( 'ldap.serverGroup' )

CliSave.GlobalConfigMode.addCommandSequence( 'AaaLdap.global',
                                             before=[
                                                'Aaa.global',
                                                AaaLdapServerGroupConfigSaveMode ],
                                             after=[ networkConfigCmdSeq ] )

def addServerDefaults( serverCmds, ldapConfig, securityConfig, options ):
   if ldapConfig.baseDn:
      baseDn = ldapConfig.baseDn
      baseDnStr = f'"{baseDn}"' if ' ' in baseDn else baseDn
      serverCmds.addCommand( f'base-dn {baseDnStr}' )
   if ldapConfig.userRdnAttribute:
      serverCmds.addCommand( f'rdn attribute user {ldapConfig.userRdnAttribute}' )
   if ldapConfig.sslProfile:
      serverCmds.addCommand( f'ssl-profile {ldapConfig.sslProfile}' )
   if ldapConfig.ldapTimeout != ldapConfig.defaultLdapTimeout or options.saveAll:
      serverCmds.addCommand( f'timeout {int( ldapConfig.ldapTimeout )}' )
   if ldapConfig.activeGroupPolicy:
      serverCmds.addCommand(
         f'authorization group policy {ldapConfig.activeGroupPolicy}' )

   if ldapConfig.searchUsernamePassword != Tac.Value(
         "Ldap::UsernamePassword", "", ReversibleSecretCli.getDefaultSecret() ):
      username = ldapConfig.searchUsernamePassword.username
      pwd = ldapConfig.searchUsernamePassword.password

      userstr = f'"{username}"' if ' ' in username else username
      formatStr = f'search username {escapeFormatString( userstr )} password {{}}'
      serverCmds.addCommand( ReversibleSecretCli.getCliSaveCommand( formatStr,
                                                                    securityConfig,
                                                                    pwd ) )

def addGroupPolicy( groupPolicyCmds, groupPolicy ):
   if groupPolicy.searchFilter != Tac.Value( "Ldap::ObjectClassOptions", "", "" ):
      groupPolicyCmds.addCommand( 'search filter objectclass {} attribute {}'.format(
         groupPolicy.searchFilter.group, groupPolicy.searchFilter.member ) )
   for groupRolePriv in groupPolicy.groupRolePrivilege.values():
      cmd = f'group "{groupRolePriv.group}" role {groupRolePriv.role}'
      if groupRolePriv.privilege != DEFAULT_ROLE_PRIVLEVEL:
         cmd += f' privilege {groupRolePriv.privilege}'
      groupPolicyCmds.addCommand( cmd )

@CliSave.saver( 'Ldap::Config', "security/aaa/ldap/config",
                requireMounts=( mgmtSecurityConfigPath, ) )
def saveLdap( ldapConfig, root, requireMounts, options ):
   securityConfig = requireMounts[ mgmtSecurityConfigPath ]

   ldapMode = root[ LdapConfigSaveMode ].getSingletonInstance()
   # Add 'server host <hostnameOrIp> ...
   for spec in sorted( ldapConfig.host.values(), key=lambda host: host.index ):
      serverMode = ldapMode[ ServerConfigSaveMode ].getOrCreateModeInstance( spec )
      serverCmds = serverMode[ 'ldap.server' ]
      addServerDefaults( serverCmds, ldapConfig.host[ spec.spec ].serverConfig,
                         securityConfig, options )
   # 'defaults' is a special mode, not a singleton (unless we create a separate
   # ServerDefaultsConfigSaveMode).
   serverMode = ldapMode[ ServerConfigSaveMode ].getOrCreateModeInstance(
      'defaults' )
   serverCmds = serverMode[ 'ldap.server' ]
   # Add commands in 'server defaults' mode
   addServerDefaults( serverCmds, ldapConfig.defaultConfig, securityConfig,
                        options )

   # Add commands in 'group policy <policyName>' mode
   for name, groupPolicy in ldapConfig.groupPolicy.items():
      groupPolicyMode = ldapMode[ GroupPolicyConfigSaveMode
      ].getOrCreateModeInstance( name )
      groupPolicyCmds = groupPolicyMode[ 'ldap.groupPolicy' ]
      addGroupPolicy( groupPolicyCmds, groupPolicy )

@CliSave.saver( 'Aaa::HostGroup', 'security/aaa/config' )
def saveHostGroup( entity, root, requireMounts, options ):
   if entity.groupType != 'ldap':
      return
   mode = root[ AaaLdapServerGroupConfigSaveMode ].getOrCreateModeInstance(
      entity.name )
   cmds = mode[ 'ldap.serverGroup' ]
   for m in entity.member.values():
      cmd = "server %s" % ( m.spec.hostname )
      assert m.spec.vrf != ''
      if m.spec.vrf != DEFAULT_VRF:
         cmd += " vrf %s" % ( m.spec.vrf )
      if m.spec.port != LdapConstants.DEFAULT_LDAP_PORT or options.saveAll:
         cmd += " port %d" % ( m.spec.port )
      cmds.addCommand( cmd )
