# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements CliSave for the link-flap commands
#-------------------------------------------------------------------------------

from CliMode.LinkFlapDetection import LinkFlapMode, LinkFlapDampingMode
import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
import Tac

DAMPING_CONF_PATH = 'interface/archer/config/eth/linkFlapDamping/dampingConfig'
DAMPING_INTF_CONFIG_DIR = 'interface/archer/config/eth/linkFlapDamping/slice/'
LINKFLAPCFG_SYSDB_PATH = 'interface/errdisable/linkFlapConfig'

requireIntfMounts = (
   [ DAMPING_INTF_CONFIG_DIR + 'FixedSystem', ] +
   [ ( DAMPING_INTF_CONFIG_DIR + f'Linecard{i}' ) for i in range( 1, 19 ) ]
)

CliSave.GlobalConfigMode.addCommandSequence( 'LinkFlap.config' )

class LinkFlapConfigMode( LinkFlapMode, CliSave.Mode ):
   def __init__( self, param ):
      LinkFlapMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class LinkFlapDampingConfigMode( LinkFlapDampingMode, CliSave.Mode ):
   def __init__( self, param ):
      LinkFlapDampingMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( LinkFlapConfigMode,
                                       before=[ IntfConfigMode ] )
LinkFlapConfigMode.addChildMode( LinkFlapDampingConfigMode )

# Commands defining link-flap configs and damping configs
# ( linkFlap.profileConfig command section ) must preceed the
# command specifying default profiles (linkFlap.defaultConfig
# command section ) in the system config.
# The reason is that default profile CLI command parser would reject
# names of profiles not configured in the system.
LinkFlapConfigMode.addCommandSequence( 'linkFlap.profileConfig',
                                        before=[ 'linkFlap.defaultConfig' ] )

LinkFlapDampingConfigMode.addCommandSequence( 'linkFlap.profileConfig' )
LinkFlapConfigMode.addCommandSequence( 'linkFlap.defaultConfig' )
IntfConfigMode.addCommandSequence( 'linkFlap.intfConfig' )

@CliSave.saver( 'LinkFlap::Config', LINKFLAPCFG_SYSDB_PATH )
def saveLinkFlapConfig( entity, root, requireMounts, options ):
   cmds = root[ 'LinkFlap.config' ]
   saveAll = options.saveAll or options.saveAllDetail
   if entity.maxFlaps != entity.maxFlapsDefault or \
      entity.interval != entity.intervalDefault or saveAll:
      cmds.addCommand(
         f'errdisable flap-setting cause link-flap max-flaps {entity.maxFlaps}'
         f' time {int(entity.interval)}'
      )

@CliSave.saver( 'LinkFlap::Config', LINKFLAPCFG_SYSDB_PATH,
                requireMounts=( DAMPING_CONF_PATH, ) )
def saveProfileConfig( entity, root, requireMounts, options ):
   saveAll = options.saveAll or options.saveAllDetail
   dampingConfig = requireMounts[ DAMPING_CONF_PATH ]
   defaultProfiles = list( entity.defaultProfileList ) + \
                     list( dampingConfig.defaultProfileList )
   if entity.profileConfig or defaultProfiles or saveAll:
      linkFlapMode = root[ LinkFlapConfigMode ].getSingletonInstance()
      profileCmds = linkFlapMode[ 'linkFlap.profileConfig' ]
      if entity.profileConfig or defaultProfiles:
         for profileConfig in sorted( entity.profileConfig.values() ):
            profileCmds.addCommand(
               f'profile {profileConfig.profileName} '
               f'max-flaps {profileConfig.maxFlaps} '
               f'time {int(profileConfig.interval)} '
               f'violations {profileConfig.violations} '
               f'intervals {profileConfig.trackedIntervals}'
            )
         profileCmds = linkFlapMode[ 'linkFlap.defaultConfig' ]
         if defaultProfiles:
            cmd = 'default-profiles ' + ' '.join( sorted( defaultProfiles ) )
            profileCmds.addCommand( cmd )
      elif saveAll:
         profileCmds.addCommand( 'no default-profiles' )

@CliSave.saver( 'LinkFlap::Config', LINKFLAPCFG_SYSDB_PATH,
                requireMounts=tuple( requireIntfMounts ) )
def saveIntfConfig( entity, root, requireMounts, options ):
   dampingIntfConfig = {}
   for mountPath in requireIntfMounts:
      intfConfigDir = requireMounts[ mountPath ]
      dampingIntfConfig.update( intfConfigDir.intfConfig )

   for intfName in set( entity.intfConfig ) | set( dampingIntfConfig ):
      intfMode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      intfCmds = intfMode[ 'linkFlap.intfConfig' ]
      enabled = True
      profiles = []
      if intfName in entity.intfConfig:
         intfConfig = entity.intfConfig[ intfName ]
         enabled &= intfConfig.enabled
         profiles.extend( intfConfig.profileList )
      if intfName in dampingIntfConfig:
         intfConfig = dampingIntfConfig[ intfName ]
         enabled &= intfConfig.enabled
         profiles.extend( intfConfig.profileList )

      if not enabled:
         intfCmds.addCommand( 'no monitor link-flap' )
      else:
         if profiles:
            intfCmds.addCommand(
               'monitor link-flap profiles ' + ' '.join( sorted( profiles ) )
            )
         else:
            intfCmds.addCommand( 'monitor link-flap' )

@CliSave.saver( 'LinkFlap::DampingConfig', DAMPING_CONF_PATH )
def saveDampingProfiles( entity, root, requireMounts, options ):
   saveAll = options.saveAll or options.saveAllDetail
   if entity.profileConfig:
      linkFlapMode = root[ LinkFlapConfigMode ].getSingletonInstance()
      defaultParameters = \
         Tac.newInstance( "LinkFlap::DampingProfileConfig", "" ).parameters

      for profileConfig in sorted( entity.profileConfig.values() ):
         parameters = profileConfig.parameters
         profileMode = \
            linkFlapMode[ LinkFlapDampingConfigMode ].getOrCreateModeInstance(
               profileConfig.profileName
            )
         profileCmds = profileMode[ 'linkFlap.profileConfig' ]
         if parameters.maxThreshold != defaultParameters.maxThreshold or \
            parameters.suppressThreshold != defaultParameters.suppressThreshold or \
            parameters.reuseThreshold != defaultParameters.reuseThreshold or \
            saveAll:
            cmd = f'penalty threshold reuse {parameters.reuseThreshold}' + \
                  f' suppression {parameters.suppressThreshold}' + \
                  f' maximum {parameters.maxThreshold}'
            profileCmds.addCommand( cmd )

         if parameters.penaltyFaultLocal != defaultParameters.penaltyFaultLocal or \
            saveAll:
            cmd = f'penalty mac fault local {parameters.penaltyFaultLocal}'
            profileCmds.addCommand( cmd )

         if parameters.penaltyFaultRemote != defaultParameters.penaltyFaultRemote \
            or saveAll:
            cmd = f'penalty mac fault remote {parameters.penaltyFaultRemote}'
            profileCmds.addCommand( cmd )

         if parameters.decayHalfLife != defaultParameters.decayHalfLife or saveAll:
            halfLifeInt = int( parameters.decayHalfLife )
            cmd = f'penalty decay half-life {halfLifeInt} seconds'
            profileCmds.addCommand( cmd )
