# Copyright (c) 2009-2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliSave
import CliSavePlugin.IntfCliSave
import LldpConstants
from EthIntfUtil import allPhyInterfaceNames
from IpLibConsts import DEFAULT_VRF
from LldpTypes import PortConfig

MAX_REQ_POWER = PortConfig.poeAllocatedPowerFallbackMax
CliSave.GlobalConfigMode.addCommandSequence( 'Lldp.config' )
CliSavePlugin.IntfCliSave.IntfConfigMode.addCommandSequence( 'Lldp.config' )

@CliSave.saver( 'Lldp::CliConfig', 'l2discovery/lldp/cliConfig',
                requireMounts=( 'interface/config/eth/phy/slice', ) )
def saveLldpCliConfig( entity, root, requireMounts, options ):
   # First save the global configuration.
   cmds = root[ 'Lldp.config' ]
   saveAll = options.saveAll

   if entity.messageTxInterval != entity.messageTxIntervalDefault or saveAll:
      cmds.addCommand( 'lldp timer %s' % entity.messageTxInterval )

   if entity.messageTxHoldTime != entity.messageTxHoldTimeDefault or saveAll:
      cmds.addCommand( 'lldp hold-time %s' % entity.messageTxHoldTime )

   if entity.reinitDelay != entity.reinitDelayDefault:
      cmds.addCommand( 'lldp timer reinitialization %s' % entity.reinitDelay )
   elif saveAll:
      cmds.addCommand( 'lldp timer reinitialization %s' % entity.reinitDelay )

   tlvTransmit = "tlv transmit"
   for keyword, attrName in sorted( LldpConstants.tlvKeywordToAttrNameMap.items() ):
      if keyword in LldpConstants.defaultDisabledTlvs:
         if getattr( entity, attrName ):
            cmds.addCommand( f'lldp {tlvTransmit} {keyword}' )
         elif saveAll:
            cmds.addCommand( f'no lldp {tlvTransmit} {keyword}' )
      elif keyword == 'system-name' and entity.sysNameTlv:
         cmds.addCommand( 'lldp {} {} {}'.format( tlvTransmit, keyword,
                          entity.sysNameTlv.decode() ) )
      elif keyword == 'system-description' and entity.sysDescTlv:
         cmds.addCommand( 'lldp {} {} {}'.format( tlvTransmit, keyword,
                          entity.sysDescTlv.decode() ) )
      elif not getattr( entity, attrName ):
         cmds.addCommand( f'no lldp {tlvTransmit} {keyword}' )
      elif saveAll:
         cmds.addCommand( f'lldp {tlvTransmit} {keyword}' )

   for keyword, attr in sorted( LldpConstants.getMedAttrMap().items() ):
      if not getattr( entity, attr ):
         cmds.addCommand( f'no lldp {tlvTransmit} med {keyword}' )
      elif saveAll:
         cmds.addCommand( f'lldp {tlvTransmit} med {keyword}' )

   if not entity.enabled:
      cmds.addCommand( 'no lldp run' )
   elif saveAll:
      cmds.addCommand( 'lldp run' )

   if entity.mgmtAddrTxOption.mgmtAddrTxMode == 'mgmtAddrTxModeAll':
      cmds.addCommand( 'lldp management-address all' )
   elif entity.mgmtAddrTxOption.mgmtAddrTxMode == 'mgmtAddrTxModeIntfName':
      cmds.addCommand( 'lldp management-address %s' % \
         entity.mgmtAddrTxOption.intfName )
   elif saveAll:
      cmds.addCommand( 'no lldp management-address' )

   if entity.mgmtAddrVrf and entity.mgmtAddrVrf != DEFAULT_VRF:
      cmds.addCommand( 'lldp management-address vrf %s' % entity.mgmtAddrVrf )

   if entity.ip6MgmtAddrIntfName:
      cmds.addCommand( 'lldp management-address ipv6 %s' %
                       entity.ip6MgmtAddrIntfName )
   elif saveAll:
      cmds.addCommand( 'no lldp management-address ipv6' )

   if entity.rxTaggedPacketsDrop:
      cmds.addCommand( 'lldp receive packet tagged drop' )
   elif saveAll:
      cmds.addCommand( 'no lldp receive packet tagged drop' )

   # Then save the port-specific configuration.
   if saveAll:
      phyIntfConfigDir = requireMounts[ 'interface/config/eth/phy/slice' ]
      cfgPortNames = allPhyInterfaceNames( phyIntfConfigDir )
   else:
      cfgPortNames = entity.portConfig

   for portName in cfgPortNames:
      cliPortConfig = entity.portConfig.get( portName )
      if not cliPortConfig:
         if saveAll:
            cliPortConfig = Tac.newInstance( 'Lldp::CliPortConfig', portName )
         else:
            continue

      # save Config
      mode = root[ CliSavePlugin.IntfCliSave.IntfConfigMode ]
      mode = mode.getOrCreateModeInstance( portName )
      cmds = mode[ 'Lldp.config' ]

      txEnabled = cliPortConfig.adminStatus in [ 'txOnly', 'txAndRx' ]
      if not txEnabled:
         cmds.addCommand( 'no lldp transmit' )
      elif saveAll:
         cmds.addCommand( 'lldp transmit' )

      rxEnabled = cliPortConfig.adminStatus in [ 'rxOnly', 'txAndRx' ]
      if not rxEnabled:
         cmds.addCommand( 'no lldp receive' )
      elif saveAll:
         cmds.addCommand( 'lldp receive' )

      fallbackCmd = 'lldp tlv transmit power-via-mdi fallback allocated-power '
      if cliPortConfig.poeAllocatedPowerFallback == MAX_REQ_POWER:
         cmds.addCommand( fallbackCmd + 'max-requested-power' )
      elif cliPortConfig.poeAllocatedPowerFallback:
         watts = cliPortConfig.poeAllocatedPowerFallback / 10.0
         cmds.addCommand( fallbackCmd + str( watts ) )
      elif saveAll:
         cmds.addCommand( 'no ' + fallbackCmd )

      bootVlanCmd = 'lldp tlv transmit ztp vlan '
      if cliPortConfig.ztpBootVlan:
         cmds.addCommand( bootVlanCmd + str( cliPortConfig.ztpBootVlan ) )
      elif saveAll:
         cmds.addCommand( 'no ' + bootVlanCmd +
                          str( cliPortConfig.ztpBootVlanMin ) )

      saveLldpCliConfigMedNetworkPolicy( cliPortConfig, cmds, saveAll )

def saveLldpCliConfigMedNetworkPolicy( cliPortConfig, cmds, saveAll ):
   hasNetworkPolicyConfig = False
   for appValue in sorted( cliPortConfig.networkPolicyConfig ):
      networkPolicy = cliPortConfig.networkPolicyConfig.get( appValue )
      if not networkPolicy:
         continue
      cmd = [ 'lldp med network-policy application' ]
      cmd.append( networkPolicy.appName )
      if networkPolicy.tagged:
         cmd.append( 'dot1q-tag %d' % networkPolicy.vlanId )
         if networkPolicy.cosValue != 0 or saveAll:
            cmd.append( 'cos %d' % networkPolicy.cosValue )
      if networkPolicy.dscpValue != 0 or saveAll:
         cmd.append( 'dscp %d' % networkPolicy.dscpValue )
      cmds.addCommand( ' '.join( cmd ) )
      hasNetworkPolicyConfig = True
   if saveAll and not hasNetworkPolicyConfig:
      cmds.addCommand( 'no lldp med network-policy' )

