# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import CliSave
import LoggingLib

CliSave.GlobalConfigMode.addCommandSequence( 'Log.config' )
CliSave.GlobalConfigMode.addCommandSequence( 'Log.facility' )

@CliSave.saver( 'LogMgr::LogConfig', 'sys/logging/config',
                requireMounts=( 'logging/config', ) )
def saveSysLogConfig( entity, root, requireMounts, options ):
   defaultVrf = Tac.newInstance( 'L3::VrfName', 'temp-status' ).defaultVrf

   cmds = root[ 'Log.config' ]
   saveAll = options.saveAll

   # 'logging on' is True by default
   if entity.loggingOn != entity.loggingOnDefault:
      cmds.addCommand( 'no logging on' )
   elif saveAll:
      cmds.addCommand( 'logging on' )

   # 'logging relogging-interval <repeatInterval>'
   if entity.repeatInterval:
      cmds.addCommand( 'logging relogging-interval %d' % entity.repeatInterval )
   elif saveAll:
      cmds.addCommand( 'no logging relogging-interval' )

   # 'repeat-messages' is off by default
   if entity.repeatMsgs != entity.repeatMsgsDefault:
      cmds.addCommand( 'logging repeat-messages' )
   elif saveAll:
      cmds.addCommand( 'no logging repeat-messages' )

   # 'logging event login root' is off by default
   if entity.loggingRootLogin != entity.loggingRootLoginDefault:
      cmds.addCommand( 'logging event login root' )
   elif saveAll:
      cmds.addCommand( 'no logging event login root' )

   # Buffered logging may be enabled, or it may not. If it is
   # disabled, the other configurable attributes (buffer size,
   # severity threshold) have been reset to their default values.
   if not entity.buffered:
      # The industry standard CLI resets these values to their
      # defaults when the feature is disabled. If someone diddles with
      # the TAC objects under the covers, those changes will not be
      # saved by our CLI.
      assert entity.bufferedSeverity == entity.bufferedSeverityDefault
      assert entity.bufferMessageCount == entity.bufferMessageCountDefault
      cmds.addCommand( 'no logging buffered' )
   else:
      nonDefaultValuesConfigured = False
      cmd = 'logging buffered'

      # The buffer size and the severity threshold may be at their
      # default values, or they may vnot.
      if entity.bufferMessageCount != entity.bufferMessageCountDefault or saveAll:
         cmd += ' %s' % entity.bufferMessageCount
         nonDefaultValuesConfigured = True

      if entity.bufferedSeverity != entity.bufferedSeverityDefault or saveAll:
         cmd += ' %s' % LoggingLib.severityTaccToCli( entity.bufferedSeverity )
         nonDefaultValuesConfigured = True

      if nonDefaultValuesConfigured:
         cmds.addCommand( cmd )

   # 'logging trap' is enabled by default
   if not entity.loggingTrapEnabled:
      # The industry standard CLI resets this value to its default
      # when the feature is disabled. If someone diddles with the TAC
      # objects under the covers, those changes will not be saved by
      # our CLI.
      assert entity.loggingTrapSeverity == entity.loggingTrapSeverityDefault
      cmds.addCommand( 'no logging trap' )
   else:
      # 'logging trap' is enabled by default, so only generate the
      # command if the severity threshold is not at its default.
      if entity.loggingTrapSeverity != entity.loggingTrapSeverityDefault or saveAll:
         severity = LoggingLib.severityTaccToCli( entity.loggingTrapSeverity )
         cmds.addCommand( 'logging trap %s' % severity )

   # Treat 'logging trap system' separately from 'logging trap'
   # i.e., we still print it out even for 'no logging trap'
   tokenAndLoggingSystem = ( ( 'trap', entity.loggingTrapSystem ),
                             ( 'buffered', entity.loggingBufferedSystem ) )
   for token, system in tokenAndLoggingSystem:
      for config in system:
         cmd = f'logging {token} system'
         if config.tag:
            cmd += ' tag ' + config.tag
         if config.flag.facility:
            cmd += ' facility ' + LoggingLib.facilityTaccToCli( config.facility )
         if config.flag.severity:
            cmd += ' severity ' + LoggingLib.severityTaccToCli( config.severity )
         if config.contain:
            cmd += ' contain ' + config.contain
         cmds.addCommand( cmd )

   # 'logging console' is enabled by default
   if not entity.loggingConsoleEnabled:
      # The industry standard CLI resets this value to its default
      # when the feature is disabled. If someone diddles with the TAC
      # objects under the covers, those changes will not be saved by
      # our CLI.
      assert entity.loggingConsoleSeverity == \
             entity.loggingConsoleSeverityDefault
      cmds.addCommand( 'no logging console' )
   else:
      # 'logging console' is enabled by default, so only generate the
      # command if the severity threshold is not at its default.
      if ( entity.loggingConsoleSeverity != entity.loggingConsoleSeverityDefault
           or saveAll ):
         severity = LoggingLib.severityTaccToCli( entity.loggingConsoleSeverity )
         cmds.addCommand( 'logging console %s' % severity )

   # 'logging terminal' is enabled by default
   if not entity.loggingMonitorUseConsole:
      if entity.loggingMonitorEnabled:
         severity = LoggingLib.severityTaccToCli(
            entity.loggingMonitorSeverity )
         cmds.addCommand( 'logging monitor %s' % severity )
      else:
         cmds.addCommand( 'no logging monitor' )
   # Should we print 'logging monitor xxx' for saveAll? I am not sure.
   # Because it actually doesn't do the right thing as to restore the
   # configuration to default.

   # 'logging synchronous' is disabled by default
   if entity.loggingSynchronousEnabled:
      severity = 'all' if entity.loggingSynchronousAll else \
                 LoggingLib.severityTaccToCli( entity.loggingSynchronousSeverity )
      cmds.addCommand( 'logging synchronous level %s' % severity )
   elif saveAll:
      assert entity.loggingSynchronousSeverity == \
             entity.loggingSynchronousSeverityDefault
      assert entity.loggingSynchronousAll == \
             entity.loggingSynchronousAllDefault
      cmds.addCommand( 'no logging synchronous' )

   # 'logging host
   for vrf in sorted( entity.vrfLoggingHost ):
      vrfString = ''
      if vrf != defaultVrf:
         vrfString = ' vrf %s' % vrf
      loggingHost = entity.vrfLoggingHost[ vrf ].loggingHost
      for host in sorted( loggingHost ):
         protocol = loggingHost[ host ].protocol
         cmd = f'logging{vrfString} host {host}'
         if not ( len( loggingHost[ host ].ports ) == 1 and
                  next( iter( loggingHost[ host ].ports ) ) ==
                  loggingHost[ host ].portDefault ) or saveAll:
            for port in loggingHost[ host ].ports:
               cmd += ' %d' % ( port )
         if protocol != loggingHost[ host ].protocolDefault or saveAll:
            cmd += ' protocol %s' % ( protocol )
         sslProfile = loggingHost[ host ].sslProfile
         if sslProfile:
            cmd += ' ssl-profile %s' % sslProfile
         cmds.addCommand( cmd )

   # 'logging timestamp format traditional [ year | timezone ].
   if entity.timestampFormat == entity.timestampFormatDefault and (
      entity.timestampYear or entity.timestampTimezone ):
      fmt = ''
      if entity.timestampYear:
         fmt += ' year'
      if entity.timestampTimezone:
         fmt += ' timezone'
      cmds.addCommand( 'logging format timestamp traditional%s' % fmt )
  # 'logging timestamp format high-resolution and saveAll.
   elif entity.timestampFormat != entity.timestampFormatDefault or saveAll:
      fmt = LoggingLib.timestampFormatToCli( entity.timestampFormat )
      cmds.addCommand( 'logging format timestamp %s' % fmt )

   # 'logging format rfc5424'
   if entity.syslogFormatRFC5424:
      cmds.addCommand( 'logging format rfc5424' )
   elif saveAll:
      cmds.addCommand( 'no logging format rfc5424' )

   # 'logging format hostname { fqdn|ipv4 }'
   if entity.hostnameFormat == 'fqdn':
      cmds.addCommand( 'logging format hostname fqdn' )
   elif entity.hostnameFormat == 'ipv4':
      cmds.addCommand( 'logging format hostname ipv4' )
   elif saveAll:
      cmds.addCommand( 'no logging format hostname' )

   # 'logging format sequence-numbers'
   if entity.sequenceNumbersEnabled != entity.sequenceNumbersEnabledDefault:
      cmds.addCommand( 'logging format sequence-numbers' )
   elif saveAll:
      # the default case should always have the new syntax
      cmds.addCommand( 'no logging format sequence-numbers' )

   # 'logging facility
   if entity.loggingFacility != entity.loggingFacilityDefault or saveAll:
      facility = LoggingLib.facilityTaccToCli( entity.loggingFacility )
      cmds.addCommand( 'logging facility %s' % facility )

   # 'logging source-address
   for vrf in sorted( set( entity.srcIpAddress ) |
                      set( entity.srcIp6Address ) ):
      vrfString = ''
      if vrf != defaultVrf:
         vrfString = ' vrf %s' % vrf
      if entity.srcIpAddress.get( vrf ):
         cmds.addCommand( 'logging%s source-address %s' %
                          ( vrfString, entity.srcIpAddress[ vrf ] ) )
      if entity.srcIp6Address.get( vrf ):
         cmds.addCommand( 'logging%s source-address %s' %
                          ( vrfString, entity.srcIp6Address[ vrf ] ) )

   # 'logging source-interface
   for vrf in sorted( entity.srcIntfName ):
      vrfString = ''
      if vrf != defaultVrf:
         vrfString = ' vrf %s' % vrf
      cmds.addCommand( 'logging%s source-interface %s' %
                       ( vrfString, entity.srcIntfName[ vrf ] ) )

   if len( entity.srcIntfName ) == 0 and saveAll:
      cmds.addCommand( 'no logging source-interface' )

   # logging policy match [invert-result] match-list <name> discard
   if entity.loggingMatchList:
      if entity.loggingMatchList.inverseResult:
         cmds.addCommand( 'logging policy match inverse-result match-list \
                          %s discard' % entity.loggingMatchList.matchListName )
      else:
         cmds.addCommand( 'logging policy match match-list %s discard'
                          % entity.loggingMatchList.matchListName )
   elif saveAll:
      cmds.addCommand( 'no logging policy' )

   if not entity.msgSeverityLevel and saveAll:
      cmds.addCommand( 'no logging message all severity' )
   else:
      for name, msgSeverity in sorted( entity.msgSeverityLevel.items() ):
         severity = LoggingLib.severityTaccToCli( msgSeverity )
         cmds.addCommand( f'logging message {name} severity {severity}' )

   if entity.facilityLogLevelDefault != entity.tacLogLevelDefault or saveAll:
      severity = LoggingLib.severityTaccToCli( entity.facilityLogLevelDefault )
      cmds.addCommand( f'logging level default {severity}' )

   facilityCmds = root[ 'Log.facility' ]
   # Note logging/config has the entire list of registered facilities
   # so we use it for saveAll case.
   for name in sorted( requireMounts[ 'logging/config' ].facilityConfig ):
      facilityLevel = entity.facilityLogLevel.get(
         name, entity.facilityLogLevelDefault )
      if facilityLevel != entity.facilityLogLevelDefault or saveAll:
         severity = LoggingLib.severityTaccToCli( facilityLevel )
         facilityCmds.addCommand( f'logging level {name} {severity}' )

   if entity.dscpValue != entity.dscpValueDefault:
      cmds.addCommand( 'logging qos dscp %d' % entity.dscpValue )
   elif saveAll:
      cmds.addCommand( 'no logging qos dscp' )

   # 'logging persistent' is disabled by default
   if entity.persistentLog:
      cmd = 'logging persistent'
      if entity.persistentLogSize != entity.persistentLogSizeDefault:
         cmd += ' %s' % entity.persistentLogSize
      cmds.addCommand( cmd )
   elif saveAll:
      assert entity.persistentLogSize == entity.persistentLogSizeDefault
      cmds.addCommand( 'no logging persistent' )
