# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliSave import escapeFormatString
from CliSaveBlock import SensitiveCommand
import Tac
from CliMode.Macsec import MacsecProfileBaseMode, MacsecBaseMode
from CliMode.Macsec import MacsecProfileStaticSakMode 
from CliMode.Macsec import MacsecProfileSecureChannelMode
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.Security import mgmtSecurityConfigPath
from CliSavePlugin.Security import SecurityConfigMode
import Toggles.MacsecCommonToggleLib as macsecToggle
from MacsecCommon import tacCipherSuiteToCli
import ReversibleSecretCli
from TypeFuture import TacLazyType

PtpBypass = TacLazyType( "Macsec::PtpBypass" )
IntfConfigMode.addCommandSequence( 'Macsec.intf' )

class MacsecConfigMode( MacsecBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      MacsecBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( MacsecConfigMode,
                                       after=[ SecurityConfigMode ],
                                       before=[ IntfConfigMode ] ) 
MacsecConfigMode.addCommandSequence( 'Macsec.config' )

class MacsecProfileConfigMode( MacsecProfileBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      MacsecProfileBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

MacsecConfigMode.addChildMode( MacsecProfileConfigMode )
MacsecProfileConfigMode.addCommandSequence( 'Macsec.config.profile' )

class MacsecStaticSakConfigMode( MacsecProfileStaticSakMode, CliSave.Mode ):
   def __init__( self, param ):
      MacsecProfileStaticSakMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

MacsecProfileConfigMode.addChildMode( MacsecStaticSakConfigMode )
MacsecStaticSakConfigMode.addCommandSequence( 'Macsec.config.profile.ss' )

class MacsecSecureChannelConfigMode( MacsecProfileSecureChannelMode, CliSave.Mode ):
   def __init__( self, param ):
      MacsecProfileSecureChannelMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

MacsecStaticSakConfigMode.addChildMode( MacsecSecureChannelConfigMode )
MacsecSecureChannelConfigMode.addCommandSequence( 'Macsec.config.profile.ss.sc' )

def saveProfileConfig( profile, root, config, options, requireMounts ):
   secretConfig = requireMounts[ mgmtSecurityConfigPath ]
   secMode = root[ MacsecConfigMode ].getSingletonInstance()
   mode = secMode[ MacsecProfileConfigMode ].getOrCreateModeInstance( profile.name )
   cmds = mode[ 'Macsec.config.profile' ]

   emptyKey = Tac.Value( "Macsec::Cak" )
   if profile.cipherSuite != profile.cipherSuiteDefault:
      cmds.addCommand('cipher %s' % tacCipherSuiteToCli[ profile.cipherSuite ] )
   elif options.saveAll:
      cmds.addCommand('cipher %s' %
                      tacCipherSuiteToCli[ profile.cipherSuiteDefault ] )
     
   if profile.key != emptyKey:
      cmd = ReversibleSecretCli.getCliSaveCommand(
         f'key {escapeFormatString( profile.key.ckn )} {{}}',
         secretConfig,
         profile.key.cakSecret )
      cmds.addCommand( cmd )

   if profile.defaultKey != emptyKey:
      cmd = ReversibleSecretCli.getCliSaveCommand(
         f'key {escapeFormatString( profile.defaultKey.ckn )} {{}} fallback',
         secretConfig,
         profile.defaultKey.cakSecret )
      cmds.addCommand( cmd )
  
   if profile.keyServerPriority != profile.keyServerPriorityDefault:
      cmds.addCommand( 'mka key-server priority %d' % profile.keyServerPriority )
   elif options.saveAll:
      cmds.addCommand( 'no mka key-server priority' )

   if profile.sessionReKeyPeriod != profile.sessionReKeyPeriodDefault:
      cmds.addCommand( 'mka session rekey-period %d' % profile.sessionReKeyPeriod )
   elif options.saveAll:
      cmds.addCommand( 'no mka session rekey-period' )

   if profile.oldKeyRetirementPeriod != profile.oldKeyRetirementPeriodDefault:
      cmds.addCommand( 'sak rx retire-delay %d seconds' %
            profile.oldKeyRetirementPeriod )
   elif options.saveAll:
      cmds.addCommand( 'no sak rx retire-delay' )

   if profile.latestKeyEnablePeriod != profile.latestKeyEnablePeriodDefault:
      cmds.addCommand( 'sak tx transmit-delay %d seconds' %
            profile.latestKeyEnablePeriod )
   elif options.saveAll:
      cmds.addCommand( 'no sak tx transmit-delay' )

   if profile.trafficPolicyOnNoMka == 'unprotected':
      cmds.addCommand( 'traffic unprotected allow' )
   elif profile.trafficPolicyOnNoMka == 'blocked':
      cmds.addCommand( 'traffic unprotected drop' )
   elif options.saveAll:
      cmds.addCommand( 'traffic unprotected allow active-sak' )

   if profile.mkaLifeTime != profile.mkaLifeTimeDefault:
      cmds.addCommand( 'mka session lifetime %d seconds' % profile.mkaLifeTime )
   elif options.saveAll:
      cmds.addCommand( 'no mka session lifetime' )

   # We do not show the default in "show running-config all". The default value will
   # not be meaningful here as default lpnValueZeroNonXpn is False and for
   # lpnValueZeroXpn is True.
   if profile.lpnValueZeroNonXpn and profile.lpnValueZeroXpn:
      cmds.addCommand( 'mka session lpn advertisement disabled' )
   elif not profile.lpnValueZeroNonXpn and not profile.lpnValueZeroXpn:
      cmds.addCommand( 'mka session lpn advertisement' )

   if profile.only1BitAnMode != profile.only1BitAnModeDefault:
      cmds.addCommand( 'sak an maximum 1' )
   elif options.saveAll:
      cmds.addCommand( 'sak an maximum 3' )

   if profile.dot1xEnabled != profile.dot1xEnabledDefault:
      cmds.addCommand( 'key source dot1x' )
   elif options.saveAll:
      cmds.addCommand( 'no key source dot1x' )
 
   if profile.secretProfileName:
      cmds.addCommand( 'key source shared-secret profile %s' %
                       profile.secretProfileName )
   elif options.saveAll:
      cmds.addCommand( 'no key source shared-secret profile' )

   if profile.defaultSecretProfileName:
      cmds.addCommand( 'key source shared-secret profile %s fallback' %
                       profile.defaultSecretProfileName )
   elif options.saveAll:
      cmds.addCommand( 'no key source shared-secret profile fallback' )

   if profile.groupCak.enabled != profile.groupCak.enabledDefault:
      cmd = 'key source group-cak '
      if profile.groupCak.lifetime != profile.groupCak.lifetimeDefault:
         cmd += str( profile.groupCak.lifetime )
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no key source group-cak' )
     
   if profile.keyDerivationPadding == \
      Tac.Type( "Macsec::KeyDerivationPadding" ).append:
      cmds.addCommand( 'key derivation padding append' )
   elif options.saveAll:
      cmds.addCommand( 'key derivation padding prepend' )

   if profile.keyRetire:
      cmds.addCommand( 'key retirement immediate' )
   elif options.saveAll:
      cmds.addCommand( 'no key retirement immediate' )

   if profile.includeSci != profile.includeSciDefault:
      cmds.addCommand( 'sci' )
   elif options.saveAll:
      cmds.addCommand( 'no sci' )

   if profile.bypassLldp != profile.bypassLldpDefault:
      cmd = 'l2-protocol lldp bypass '
      if profile.bypassLldpUnauth != profile.bypassLldpDefault:
         cmd += 'unauthorized'
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no l2-protocol lldp bypass' )

   if profile.ethFlowControl == Tac.Type( "Macsec::EthFlowControl" ).encrypt:
      cmds.addCommand( 'l2-protocol ethernet-flow-control encrypt' )
   elif profile.ethFlowControl == Tac.Type( "Macsec::EthFlowControl" ).bypass:
      cmds.addCommand( 'l2-protocol ethernet-flow-control bypass' )
   elif options.saveAll:
      cmds.addCommand( 'no l2-protocol ethernet-flow-control' )

   if profile.bypassPtp == PtpBypass.ptpBypass:
      cmds.addCommand( 'ptp bypass' )
   elif profile.bypassPtp == PtpBypass.ptpEncrypt:
      cmds.addCommand( 'no ptp bypass' )
   elif options.saveAll:
      cmds.addCommand( 'default ptp bypass' )

   if profile.replayProtection != profile.replayProtectionDefault:
      cmds.addCommand( 'replay protection disabled' )
   elif options.saveAll:
      cmds.addCommand( 'no replay protection disabled' )
   if profile.replayProtectionWindow != profile.replayProtectionWindowDefault:
      cmds.addCommand( 'replay protection window ' +
         str( profile.replayProtectionWindow ) )
   elif options.saveAll:
      cmds.addCommand( 'replay protection window 0 ' )
      
   if macsecToggle.toggleMacsecFipsPostFailureForcedEnabled():
      if profile.fipsPostFailureForced:
         cmds.addCommand( 'fips post failure forced' )
      elif options.saveAll:
         cmds.addCommand( 'no fips post failure forced' )

   # Save static sak config
   # Save secure channel tx config
   staticMode = mode[ MacsecStaticSakConfigMode ].getOrCreateModeInstance(
      profile.name )
   txScMode = staticMode[ MacsecSecureChannelConfigMode ].getOrCreateModeInstance(
                                                          ( profile.name, 'tx' ) )
   cmds = txScMode[ 'Macsec.config.profile.ss.sc' ]
   txSci = profile.txStaticSci
   txStaticSak = profile.txStaticSak
   if txSci:
      cmds.addCommand( f'identifier {txSci.addr}::{txSci.portNum}' )
   if txStaticSak:
      formatStr = f'an {txStaticSak.an} key {{}}'
      if macsecToggle.toggleMacsecStaticSakNonXpnEnabled() and txStaticSak.salt:
         formatStr += f' salt {escapeFormatString( txStaticSak.salt )}'
      cmd = ReversibleSecretCli.getCliSaveCommand( formatStr,
                                                   secretConfig,
                                                   txStaticSak.keySecret )
      cmds.addCommand( cmd )

   # Save secure channel rx config
   rxScMode = staticMode[ MacsecSecureChannelConfigMode ].getOrCreateModeInstance( 
                                                          ( profile.name, 'rx' ) )
   cmds = rxScMode[ 'Macsec.config.profile.ss.sc' ]
   rxSci = profile.rxStaticSci
   if rxSci:
      cmds.addCommand( f'identifier {rxSci.addr}::{rxSci.portNum}' )
   if profile.rxStaticSak:
      for i in sorted( profile.rxStaticSak ):
         rxSak = profile.rxStaticSak[ i ]
         formatStr = f'an {rxSak.an} key {{}}'
         if macsecToggle.toggleMacsecStaticSakNonXpnEnabled() and rxSak.salt:
            formatStr += f' salt {escapeFormatString( rxSak.salt )}'
         cmd = ReversibleSecretCli.getCliSaveCommand( formatStr,
                                                      secretConfig,
                                                      rxSak.keySecret )
         cmds.addCommand( cmd )

def saveIntfConfig( intfConfig, root, config, saveAll ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfConfig.intfId )
   cmds = mode[ 'Macsec.intf' ]

   # When a profile is deleted from the open-config end, the profileName in
   # intfConfig is set to an empty string. To prevent treating this empty string
   # as a new profile, ensure that the profileName is not empty before executing
   # the save command.
   if intfConfig.profileName:
      cmds.addCommand( 'mac security profile %s' % intfConfig.profileName )

def saveLicenseConfig( root, config, options ):
   if config.licenseConfig != Tac.Value( "Macsec::LicenseConfig" ) or \
         options.saveAll:
      mode = root[ MacsecConfigMode ].getSingletonInstance()
      cmds = mode[ 'Macsec.config' ]
      licensee = config.licenseConfig.licenseeName
      authKey = config.licenseConfig.authKey

      if config.licenseConfig != Tac.Value( "Macsec::LicenseConfig" ):
         cmd = SensitiveCommand(
            f'license {CliSave.escapeFormatString( licensee )} {{}}',
            f'{authKey:x}' )
         cmds.addCommand( cmd )
      elif options.saveAll:
         cmds.addCommand( 'no license' )

def saveDelayProtectionConfig( root, config, options ):
   if config.delayProtection != config.delayProtectionDefault or \
         options.saveAll:
      mode = root[ MacsecConfigMode ].getSingletonInstance()
      cmds = mode[ 'Macsec.config' ]
      if config.delayProtection:
         cmds.addCommand( 'delay protection' )
      elif options.saveAll:
         cmds.addCommand( 'no delay protection' )

def saveFipsRestrictionsConfig( root, config, options ):
   if config.fipsRestrictions != config.fipsRestrictionsDefault or \
         options.saveAll:
      mode = root[ MacsecConfigMode ].getSingletonInstance()
      cmds = mode[ 'Macsec.config' ]
      if config.fipsRestrictions:
         cmds.addCommand( 'fips restrictions' )
      elif options.saveAll:
         cmds.addCommand('no fips restrictions' )

def saveEthFlowControlEncrypt( root, config, options ):
   if config.ethFlowControl != \
      Tac.Type( "Macsec::EthFlowControl" ).platformDefault or \
      options.saveAll:
      mode = root[ MacsecConfigMode ].getSingletonInstance()
      cmds = mode[ 'Macsec.config' ]
      if config.ethFlowControl == Tac.Type( "Macsec::EthFlowControl" ).encrypt:
         cmds.addCommand( 'l2-protocol ethernet-flow-control encrypt' )
      elif config.ethFlowControl == Tac.Type( "Macsec::EthFlowControl" ).bypass:
         cmds.addCommand( 'l2-protocol ethernet-flow-control bypass' )
      elif options.saveAll:
         cmds.addCommand( 'no l2-protocol ethernet-flow-control' )

def saveShutdownConfig( root, config, options ):
   if config.shutdown or options.saveAll:
      mode = root[ MacsecConfigMode ].getSingletonInstance()
      cmds = mode[ 'Macsec.config' ]

      if config.shutdown:
         cmds.addCommand( 'shutdown' )
      elif options.saveAll:
         cmds.addCommand( 'no shutdown' )

def saveEapolAttributes( root, config, options ):
   if config.eapolAttr.destinationMac != config.eapolAttr.destinationMacDefault or\
      config.eapolAttr.etherType != config.eapolAttr.etherTypeDefault or\
      options.saveAll:
      mode = root[ MacsecConfigMode ].getSingletonInstance()
      cmds = mode[ 'Macsec.config' ]

      if config.eapolAttr.destinationMac != \
            config.eapolAttr.destinationMacDefault:
         cmds.addCommand( 'eapol mac destination %s' %
               config.eapolAttr.destinationMac )
      elif options.saveAll:
         cmds.addCommand( 'no eapol mac destination' )

      if macsecToggle.toggleMacsecConfigurableEapolEtherTypeEnabled():
         if config.eapolAttr.etherType != config.eapolAttr.etherTypeDefault:
            cmds.addCommand( 'eapol ethertype %d' %
                          config.eapolAttr.etherType )
         elif options.saveAll:
            cmds.addCommand( 'no eapol ethertype' )

def savePtpBypass( root, config, options ):
   if config.bypassPtp != PtpBypass.ptpPlatformDefault or options.saveAll:
      mode = root[ MacsecConfigMode ].getSingletonInstance()
      cmds = mode[ 'Macsec.config' ]
      if config.bypassPtp == PtpBypass.ptpBypass:
         cmds.addCommand( 'ptp bypass' )
      elif config.bypassPtp == PtpBypass.ptpEncrypt:
         cmds.addCommand( 'no ptp bypass' )
      elif options.saveAll:
         cmds.addCommand( 'default ptp bypass' )

@CliSave.saver( 'Macsec::Config', 'macsec/input/cli',
                requireMounts = ( mgmtSecurityConfigPath, ) )
def saveConfig( config, root, requireMounts, options ):
   # Save profile configs.
   for profileName in sorted( config.profile ):
      saveProfileConfig( config.profile[ profileName ], root, config, 
                         options, requireMounts )

   # Save the interface configs.
   for intfId in config.intfConfig:
      saveIntfConfig( config.intfConfig[ intfId ], root, config, options.saveAll )
   
   # Save the license config.
   saveLicenseConfig( root, config, options )

   # Save the delay protection config
   saveDelayProtectionConfig( root, config, options )

   # Save the FIPS config
   saveFipsRestrictionsConfig( root, config, options )

   # Save flow control encryption config
   saveEthFlowControlEncrypt( root, config, options )
   
   # Save Ptp bypass config
   savePtpBypass( root, config, options )

   # Save the shutdown config
   saveShutdownConfig( root, config, options )

   # Save the EAPoL attributes config
   saveEapolAttributes( root, config, options )

