#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac, re
import CliSave
from CliMode.BgpGroup import BgpGroupMode
from CliMode.BgpMaintenanceProfile import BgpMaintenanceProfileMode
from CliMode.MaintenanceModeCliSaveLib import MaintenanceConfigMode
from CliSavePlugin.IraVrfCliSave import VrfDefinitionCliSaveMode
# pylint: disable-next=ungrouped-imports
from CliMode.BgpMaintenanceMode import BgpMaintenanceMode
from Toggles import RcfLibToggleLib

defaultProfileName = \
   Tac.Type( 'Maintenance::Profile::DefaultProfile' ).systemDefaultProfileName
bgpGroupType = Tac.Type( "Group::GroupType" ).groupTypeBgp
unitType = Tac.Type( "Maintenance::Unit::UnitType" )

#-----------------------------------------------------------------     
# CliSave Plugin for Bgp Submode under Maintenance Config Mode   
#-----------------------------------------------------------------   
class BgpMaintenanceCliSaveMode( BgpMaintenanceMode, CliSave.Mode ):

   def __init__( self, param ):
      BgpMaintenanceMode.__init__( self, param )                 
      CliSave.Mode.__init__( self, param ) 
                                             
CliSave.GlobalConfigMode.addChildMode( MaintenanceConfigMode )
MaintenanceConfigMode.addChildMode( BgpMaintenanceCliSaveMode )
BgpMaintenanceCliSaveMode.addCommandSequence( 'Maintenance.bgpMode' )         

@CliSave.saver( 'Maintenance::Unit::ConfigDir', 'maintenance/unit/config',
                requireMounts = ( 'maintenance/unit/input/cli', ) )
def saveMaintenanceBgpConfig( entity, root, requireMounts, options ):
   unitInputCli = requireMounts[ 'maintenance/unit/input/cli' ]
   for unit in entity.config.values():
      if unit.unitType == unitType.dynamic:
         isBgpUnit = False
         for group in unit.group:
            if group.type == bgpGroupType:
               isBgpUnit = True
         if isBgpUnit:
            bgpVrfRe = "<Dynamic.+><.+><vrf-(.+)>"
            if re.search( bgpVrfRe, unit.unitName ):
               mode = root[ MaintenanceConfigMode ].getSingletonInstance()
               bgpMode = mode[ BgpMaintenanceCliSaveMode ].getOrCreateModeInstance(
                  unit.unitName )
            cmds = bgpMode[ 'Maintenance.bgpMode' ]
            if unit.unitName in unitInputCli.unit:
               cmds.addCommand( 'quiesce' )

#-----------------------------------------------------------------
# CliSave Plugin for Bgp Maintenance Groups
#-----------------------------------------------------------------
class BgpGroupConfigMode( BgpGroupMode, CliSave.Mode ):

   def __init__( self, param ):
      BgpGroupMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( BgpGroupConfigMode,
                                       after=[ VrfDefinitionCliSaveMode ] )
BgpGroupConfigMode.addCommandSequence( 'BgpGroup.configDir' )

@CliSave.saver( 'BgpGroup::ConfigDir', 'group/config/bgp',
                requireMounts = ( 'maintenance/group/config', ) )
def saveBgpGroupConfig( entity, root, requireMounts, options ):
   maintenanceGroupDir = requireMounts[ 'maintenance/group/config' ]
   groupOrigin = Tac.Type( "Group::GroupOrigin" )
   for group, groupConfig in entity.config.items():
      if groupConfig.origin == groupOrigin.dynamic:
         continue
      mode = root[ BgpGroupConfigMode ].getOrCreateModeInstance( group )
      cmds = mode[ 'BgpGroup.configDir' ]
      if groupConfig.origin == groupOrigin.userConfigured:
         if groupConfig.vrfName != 'default':
            cmds.addCommand( 'vrf %s' % ( groupConfig.vrfName ) )
         for neighbor in groupConfig.neighbor:
            cmds.addCommand( 'neighbor %s' % ( neighbor.stringValue ) )
      maintenanceGroup = maintenanceGroupDir.maintenanceGroup.get( groupConfig.key )
      if maintenanceGroup:
         for profile in maintenanceGroup.profile:
            cmds.addCommand( 'maintenance profile bgp %s' %( profile.name ) )

      #XXX Temporary hack to avoid 'maintenance' config mode following this from 
      # cribbing for IncompleteCommandError
      cmds.addCommand( 'exit' )

#-----------------------------------------------------------------
# CliSave Plugin for Bgp Maintenance Profiles
#-----------------------------------------------------------------
class BgpMaintenanceProfileConfigMode( BgpMaintenanceProfileMode, CliSave.Mode ):
   
   def __init__( self, param ):
      BgpMaintenanceProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

MaintenanceConfigMode.addChildMode( BgpMaintenanceProfileConfigMode )
BgpMaintenanceProfileConfigMode.addCommandSequence( 'BgpProfile.configDir' )
MaintenanceConfigMode.addCommandSequence( 'DefaultProfile' )

def saveBgpProfileHelper( cmds, policy, dirStr ):
   if policy and policy.isValid():
      if policy.isRcfFunction():
         if RcfLibToggleLib.toggleRcfMaintenancePoaEnabled():
            cmds.addCommand(
               f'initiator rcf {dirStr} {policy.policyName}()' )
      else:
         cmds.addCommand(
            f'initiator route-map {policy.policyName} {dirStr}' )

@CliSave.saver( 'BgpMaintenanceProfile::ConfigDir',
                'maintenance/profile/config/bgp' )
def saveBgpProfileConfig( entity, root, requireMounts, options ):
   for profile, member in entity.config.items():
      if profile != defaultProfileName:
         mode = root[ MaintenanceConfigMode ].getSingletonInstance()
         profileMode = mode[
            BgpMaintenanceProfileConfigMode ].getOrCreateModeInstance( profile )
         cmds = profileMode[ 'BgpProfile.configDir' ]
         saveBgpProfileHelper( cmds, member.policyInOut, "inout" )
         saveBgpProfileHelper( cmds, member.policyIn, "in" )
         saveBgpProfileHelper( cmds, member.policyOut, "out" )

@CliSave.saver( 'Maintenance::Profile::DefaultProfile',
                'maintenance/profile/config/default/bgp' )
def saveDefaultBgpProfile( entity, root, requireMounts, options ):
   if entity.profileName != defaultProfileName:
      mode = root[ MaintenanceConfigMode ].getSingletonInstance()
      cmds = mode[ 'DefaultProfile' ]
      cmds.addCommand( 'profile bgp %s default' % entity.profileName )
