#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import CliSave
from CliMode.IntfMaintenanceProfile import IntfMaintenanceProfileMode
from CliMode.MaintenanceModeCliSaveLib import MaintenanceConfigMode
from CliMode.IntfGroup import IntfGroupMode
from Intf.IntfRange import intfListToCanonical
from CliSavePlugin.IntfCliSave import IntfConfigMode
   
#-----------------------------------------------------------------
# CliSave Plugin for Interface Maintenance Groups
#-----------------------------------------------------------------
class IntfGroupConfigMode( IntfGroupMode, CliSave.Mode ):

   def __init__( self, param ):
      IntfGroupMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( IntfGroupConfigMode,
                                       after=[ IntfConfigMode ] )
IntfGroupConfigMode.addCommandSequence( 'IntfGroup.configDir' )

@CliSave.saver( 'IntfGroup::ConfigDir', 'group/config/interface',
                requireMounts = ( 'interface/status/all',
                                  'maintenance/group/config' ) )
def saveIntfGroupConfig( entity, root, requireMounts, options ):
   maintenanceGroupDir = requireMounts[ 'maintenance/group/config' ]
   groupOrigin = Tac.Type( "Group::GroupOrigin" )
   for group, groupConfig in entity.config.items():
      if groupConfig.origin == groupOrigin.dynamic:
         continue
      mode = root[ IntfGroupConfigMode ].getOrCreateModeInstance( group )
      cmds = mode[ 'IntfGroup.configDir' ]
      if groupConfig.origin == groupOrigin.userConfigured:
         ethIntfDir = requireMounts[ 'interface/status/all' ]
         allIntf = list( groupConfig.intf )
         nonExistingIntfs = []
         for intf in groupConfig.intf:
            if ( intf.startswith( 'Ethernet' ) and
                 intf not in ethIntfDir.intfStatus ):
               nonExistingIntfs.append( intf )
               allIntf.remove( intf )
         intfs = intfListToCanonical( allIntf )
         for intf in intfs:
            cmds.addCommand( 'interface %s' % intf )
         for intf in nonExistingIntfs:
            cmds.addCommand( 'interface %s' % intf )
      intfProfileType = 'profileTypeInterface'
      bgpProfileType = 'profileTypeBgp'
      maintenanceGroup = maintenanceGroupDir.maintenanceGroup.get( groupConfig.key )
      if maintenanceGroup:
         for profile in maintenanceGroup.profile:
            # pylint: disable-next=superfluous-parens
            if ( profile.type == intfProfileType ):
               cmds.addCommand( 
                  'maintenance profile interface %s' %( profile.name ) )
            # pylint: disable-next=superfluous-parens
            elif ( profile.type == bgpProfileType ):
               cmds.addCommand( 'maintenance profile bgp %s' %( profile.name ) )

      #XXX Temporary hack to avoid 'maintenance' config mode following this from 
      # cribbing for IncompleteCommandError
      cmds.addCommand( 'exit' )

#--------------------------------------------------------------------
# CliSave Plugin for Interface Maintenance Profile
#--------------------------------------------------------------------
class IntfMaintenanceProfileConfigMode( IntfMaintenanceProfileMode, CliSave.Mode ):
   
   def __init__( self, param ):
      IntfMaintenanceProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( MaintenanceConfigMode,
                                       after=[ IntfConfigMode ] )
MaintenanceConfigMode.addCommandSequence( 'DefaultProfile' )

defaultProfileName = \
   Tac.Type( 'Maintenance::Profile::DefaultProfile' ).systemDefaultProfileName

@CliSave.saver( 'Maintenance::Profile::DefaultProfile',
                'maintenance/profile/config/default/interface' )
def saveDefaultInterfaceProfile( entity, root, sysdb, options ):
   if entity.profileName != defaultProfileName:
      mode = root[ MaintenanceConfigMode ].getSingletonInstance()
      cmds = mode[ 'DefaultProfile' ]
      cmds.addCommand( 'profile interface %s default' % entity.profileName )

MaintenanceConfigMode.addChildMode( IntfMaintenanceProfileConfigMode )
IntfMaintenanceProfileConfigMode.addCommandSequence( 'IntfProfile.configDir' )

@CliSave.saver( 'IntfMaintenanceProfile::ConfigDir',
                'maintenance/profile/config/interface' )
def saveIntfProfileConfig( entity, root, sysdb, options ):
   default = options.saveAll or options.saveAllDetail 
   for profile, member in entity.config.items():
      if profile != defaultProfileName:
         mode = root[ MaintenanceConfigMode ].getSingletonInstance()
         profileMode = mode[
            IntfMaintenanceProfileConfigMode ].getOrCreateModeInstance( profile )
         cmds = profileMode[ 'IntfProfile.configDir' ]
         if member.rateMonLoadInterval != member.defaultLoadInterval:
            cmds.addCommand( 'rate-monitoring load-interval %s' % ( 
                  int ( member.rateMonLoadInterval ) ) )
         elif default:
            cmds.addCommand( 'rate-monitoring load-interval %s' % (
                  int ( member.defaultLoadInterval ) ) )
         if member.rateMonThresholdInKbps != member.defaultRateMonThreshold:
            cmds.addCommand( 'rate-monitoring threshold %s' % ( 
                  int (  member.rateMonThresholdInKbps ) ) )
         elif default:
            cmds.addCommand( 'rate-monitoring threshold %s' % ( 
                  int (  member.defaultRateMonThreshold ) ) )
         # pylint: disable-next=superfluous-parens,singleton-comparison
         if ( member.shutdown == True ):
            cmds.addCommand( 'shutdown max-delay %s' % int ( 
                  member.shutdownMaxDelay ) )
         elif default:
            cmds.addCommand( 'no shutdown' )
