#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-----------------------------------------------------------------
# This module implements saving the CLI for Maintenance Units
#-----------------------------------------------------------------
import Tac, re
import CliSave
from CliMode.Intf import IntfMode
from CliMode.MaintenanceUnitMode import MaintenanceUnitMode
from CliMode.MaintenanceModeCliSaveLib import MaintenanceConfigMode
from CliMode.MaintenanceUnitProfileMode import MaintenanceUnitProfileMode
from CliMode.IntfMaintenanceMode import IntfMaintenanceMode

intfGroupType = Tac.Type( "Group::GroupType" ).groupTypeInterface
bgpGroupType = Tac.Type( "Group::GroupType" ).groupTypeBgp
defaultProfileName = \
    Tac.Type( 'Maintenance::Profile::DefaultProfile' ).systemDefaultProfileName
unitType = Tac.Type( "Maintenance::Unit::UnitType" )

class IntfMaintenanceCliSaveMode( IntfMaintenanceMode, CliSave.Mode ):
   
   def __init__( self, param ):
      intfShortName = IntfMode.getShortname( param )
      IntfMaintenanceMode.__init__( self, param, intfShortName )
      CliSave.Mode.__init__( self, param )

class MaintenanceUnitConfigMode( MaintenanceUnitMode, CliSave.Mode ):
   def __init__( self, param ):
      MaintenanceUnitMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class MaintenanceProfileUnitConfigMode( MaintenanceUnitProfileMode, CliSave.Mode ):
   def __init__( self, param ):
      MaintenanceUnitProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( MaintenanceConfigMode )
MaintenanceConfigMode.addChildMode( MaintenanceUnitConfigMode )
MaintenanceConfigMode.addChildMode( IntfMaintenanceCliSaveMode )
MaintenanceConfigMode.addChildMode( MaintenanceProfileUnitConfigMode )
MaintenanceProfileUnitConfigMode.addCommandSequence( 
   'MaintenanceUnitProfile.configDir' )
MaintenanceUnitConfigMode.addCommandSequence( 'MaintenanceUnit.configDir' )
MaintenanceConfigMode.addCommandSequence( 'DefaultProfile' )
IntfMaintenanceCliSaveMode.addCommandSequence( 'Maintenance.intfMode' )

@CliSave.saver( 'Maintenance::Unit::ConfigDir', 'maintenance/unit/config',
                requireMounts = ( 'maintenance/unit/input/cli', ) )
def saveMaintenanceIntfConfig( entity, root, requireMounts, options ):
   unitInputCli = requireMounts[ 'maintenance/unit/input/cli' ]
   for unit in entity.config.values():
      if unit.unitType == unitType.dynamic:
         isIntfUnit = False
         for group in unit.group:
            if group.type == intfGroupType:
               isIntfUnit = True
         if isIntfUnit:
            dynamicComponentRe = "<Dynamic.+><(.+)><.*>"
            vrfRe = "<Dynamic.+><.+><vrf-(.+)>"
            if not re.search( vrfRe, unit.unitName ):
               intf = re.search( dynamicComponentRe, unit.unitName ).group( 1 )
               mode = root[ MaintenanceConfigMode ].getSingletonInstance()
               intfMode = mode[
                  IntfMaintenanceCliSaveMode ].getOrCreateModeInstance( intf )
               cmds = intfMode[ 'Maintenance.intfMode' ]
               if unit.unitName in unitInputCli.unit:
                  cmds.addCommand( 'quiesce' )

@CliSave.saver( 'Maintenance::Profile::DefaultProfile',
                'maintenance/profile/config/default/unit' )
def saveDefaultUnitProfile( entity, root, requireMounts, options ):
   if entity.profileName != defaultProfileName:
      mode = root[ MaintenanceConfigMode ].getSingletonInstance()
      cmds = mode[ 'DefaultProfile' ]
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'profile unit %s default' % entity.profileName )

@CliSave.saver( 'MaintenanceUnitProfile::ConfigDir',
                'maintenance/profile/config/unit' )
def saveUnitProfileConfig( entity, root, requireMounts, options ):
   for profile in entity.config:
      if profile != defaultProfileName:
         mode = root[ MaintenanceConfigMode ].getSingletonInstance()
         profileMode = mode[
            MaintenanceProfileUnitConfigMode ].getOrCreateModeInstance( profile )
         cmds = profileMode[ 'MaintenanceUnitProfile.configDir' ]
         unitProfile = entity.config.get( profile )
         if not unitProfile:
            continue
         if unitProfile.onBootEnabled:
            # pylint: disable-next=consider-using-f-string
            cmds.addCommand( 'on-boot duration %i' % unitProfile.onBootDuration )
         elif ( options.saveAll or options.saveAllDetail ):
            cmds.addCommand( 'no on-boot' )
      
@CliSave.saver( 'Maintenance::Unit::ConfigDir', 'maintenance/unit/config',
                requireMounts = ( 'maintenance/unit/input/cli', ) )
def saveMaintenanceUnitConfig( entity, root, requireMounts, options ):
   unitCliInput = requireMounts[ 'maintenance/unit/input/cli' ]
   for unit in entity.config.values():
      if unit.unitType == unitType.dynamic:
         continue
      mode = root[ MaintenanceConfigMode ].getSingletonInstance()
      unitMode = mode[ MaintenanceUnitConfigMode ].getOrCreateModeInstance (
         unit.unitName )
      cmds = unitMode[ 'MaintenanceUnit.configDir' ]
      if unit.unitType == unitType.userConfigured:
         for group in unit.group:
            if group.type == bgpGroupType:
               # pylint: disable-next=consider-using-f-string
               cmds.addCommand( 'group bgp %s' % group.name )
            elif group.type == intfGroupType:
               # pylint: disable-next=consider-using-f-string
               cmds.addCommand( 'group interface %s' % group.name )
      if unit.profileName:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'profile unit %s' % unit.profileName )
      if unit.unitName in unitCliInput.unit:
         cmds.addCommand( 'quiesce' )
      elif options.saveAll or options.saveAllDetail:
         cmds.addCommand( 'no quiesce' )
         
