#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliSavePlugin import IntfCliSave
import Tac
from RoutingIntfUtils import allRoutingProtocolIntfNames

AddressFamily = Tac.Type( "Arnet::AddressFamily" )

IntfCliSave.IntfConfigMode.addCommandSequence( \
      'McastBoundary.intf', after=[ 'Ira.ipIntf' ] )

def mcastRoutingSupported( sysdbRoot, routingHardwareStatus=None ):
   if routingHardwareStatus is None:
      routingHardwareStatus = sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   return routingHardwareStatus.multicastRoutingSupported

def McastBoundaryConfig( mcastBoundaryConfig, root, requireMounts, options, af ):
   #Save defaults only if platform supports multicast routing
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   if af == AddressFamily.ipv4:
      hwStatus = requireMounts[ 'routing/hardware/status' ]
   else:
      hwStatus = requireMounts[ 'routing6/hardware/status' ]

   if not mcastRoutingSupported(
         None,
         hwStatus ):
      saveAll = False
      saveAllDetail = False

   if saveAllDetail:
      cfgIntfNames = allRoutingProtocolIntfNames( requireMounts,
                                                  includeEligible=True )
   elif saveAll:
      # Routing configuration is allowed on switchports as well.
      # Save configuration on all routing protocol interfaces and switchports
      # with non-default config.
      cfgIntfNames = set( allRoutingProtocolIntfNames( requireMounts ) )
      cfgIntfNames.update( mcastBoundaryConfig.intfConfig )
   else:
      cfgIntfNames = mcastBoundaryConfig.intfConfig

   for intfName in cfgIntfNames:
      intfConfig = mcastBoundaryConfig.intfConfig.get( intfName )

      if not intfConfig:
         if saveAll:
            intfConfig = Tac.newInstance( "Routing::McastBoundary::IntfConfig",
                  intfName )
         else:
            continue

      saveIntfConfigConverted( intfConfig, root, saveAll,
                               saveAllDetail, af )

@CliSave.saver( "Routing::McastBoundary::Config", 'routing/mcastboundary/config',
                requireMounts=( 'routing/hardware/status',
                                'interface/config/all', 'interface/status/all' ) )
def McastBoundaryIpv4Config( mcastBoundaryConfig, root, requireMounts, options ):
   McastBoundaryConfig( mcastBoundaryConfig, root, requireMounts, options,
                        AddressFamily.ipv4 )

@CliSave.saver( "Routing::McastBoundary::Config", 'routing6/mcastboundary/config',
                requireMounts=( 'routing6/hardware/status',
                                'interface/config/all', 'interface/status/all' ) )
def McastBoundaryIpv6Config( mcastBoundaryConfig, root, requireMounts, options ):
   McastBoundaryConfig( mcastBoundaryConfig, root, requireMounts, options,
                        AddressFamily.ipv6 )

def saveIntfConfigConverted( mcastBoundaryIntfConfig, root, saveAll,
                             saveAllDetail, af ):
   cmdPrefix = 'multicast %s boundary' % af

   intf = mcastBoundaryIntfConfig
   mode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance( intf.intfId )
   cmds = mode[ 'McastBoundary.intf' ]

   for k in intf.boundary:
      v = intf.boundary[ k ]
      prefix = k.v4Prefix.stringValue if af==AddressFamily.ipv4 \
                                      else k.v6Prefix.stringValue
      out = " out" if v.out else ""
      cmds.addCommand( '%s %s%s' % \
                       ( cmdPrefix, prefix, out ) )

   for key in intf.boundaryAcl:
      value = intf.boundaryAcl[ key ]
      out = " out" if value.out else ""
      cmds.addCommand( '%s %s%s' % ( cmdPrefix, value.acl, out ) )
