# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import CliSave
import ReversibleSecretCli
import Tac
from CliMode.MgmtGnmiClient import (
   MgmtGnmiClientBaseMode,
   GnmiServerGroupBaseMode,
   GnmiServerBaseMode
)
from CliSavePlugin.Security import mgmtSecurityConfigPath
from TypeFuture import TacLazyType

GnmiConstants = TacLazyType( 'MgmtGnmiClient::GnmiConstants' )

class MgmtGnmiClientSaveMode( MgmtGnmiClientBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      MgmtGnmiClientBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class GnmiServerSaveMode( GnmiServerBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      GnmiServerBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class GnmiServerGroupSaveMode( GnmiServerGroupBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      GnmiServerGroupBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( MgmtGnmiClientSaveMode )
MgmtGnmiClientSaveMode.addCommandSequence( 'Mgmt.gnmiClient' )

MgmtGnmiClientSaveMode.addChildMode( GnmiServerSaveMode )
GnmiServerSaveMode.addCommandSequence( 'Mgmt.gnmiClient.server' )

MgmtGnmiClientSaveMode.addChildMode( GnmiServerGroupSaveMode )
GnmiServerGroupSaveMode.addCommandSequence( 'Mgmt.gnmiClient.serverGroup' )

def isAddressDefault( server ):
   return ( server.hostname == '' and # Default Hostname
            server.port == GnmiConstants.gnmiServerPortDefault and
            server.vrf == GnmiConstants.gnmiServerVrfDefault )

def isUsernamePwdDefault( server ):
   return ( server.username == '' and # Default Username
            server.password == GnmiConstants.gnmiServerPasswordDefault )

def saveGnmiServer( server, secretConfig, mode, options ):
   serverName = server.name
   serverMode = mode[ GnmiServerSaveMode ].getOrCreateModeInstance( serverName )
   cmds = serverMode[ 'Mgmt.gnmiClient.server' ]
   if not isAddressDefault( server ):
      cmds.addCommand( f'address {server.hostname} port {server.port}' )
   elif options.saveAll:
      cmds.addCommand( 'no address' )
   if not isUsernamePwdDefault( server ):
      cmd = f'username {server.username} password {{}}'
      cmds.addCommand( ReversibleSecretCli.getCliSaveCommand( cmd, secretConfig,
                                                              server.password ) )
   elif options.saveAll:
      cmds.addCommand( 'no username' )

def saveGnmiServerGroup( serverGroup, mode, options ):
   serverGroupName = serverGroup.name
   serverGroupMode = mode[ GnmiServerGroupSaveMode ].getOrCreateModeInstance(
                        serverGroupName )
   cmds = serverGroupMode[ 'Mgmt.gnmiClient.serverGroup' ]
   for serverName in sorted( serverGroup.memberGnmiServer ):
      cmds.addCommand( f'member {serverName}' )

@CliSave.saver( 'MgmtGnmiClient::Config', 'mgmtGnmiClient/config',
                requireMounts=( mgmtSecurityConfigPath, ) )
def saveGnmiClient( config, root, requireMounts, options ):
   secretConfig = requireMounts[ mgmtSecurityConfigPath ]
   mode = root[ MgmtGnmiClientSaveMode ].getOrCreateModeInstance(
             'client-api-gnmi' )
   for gnmiServer in config.gnmiServer.values():
      saveGnmiServer( gnmiServer, secretConfig, mode, options )
   for gnmiServerGroup in config.gnmiServerGroup.values():
      saveGnmiServerGroup( gnmiServerGroup, mode, options )
