# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from Intf.IntfRange import intfListToCanonical

import CliSave
from CliMode.MldSnooping import MldSnoopingMode, MldSnoopingVlanMode
from CliSavePlugin.IntfCliSave import IntfConfigMode
from MultiRangeRule import multiRangeToCanonicalString
import Toggles.GmpToggleLib

#-------------------------------------------------------------------------------
# Object used for saving commands in "config-mld-snooping-vlan" mode.
#-------------------------------------------------------------------------------
class MldSnoopingVlanConfigSaveMode( MldSnoopingVlanMode, CliSave.Mode ):
   mergeRange = True

   def __init__( self, param ):
      MldSnoopingVlanMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def instanceKey( self ):
      return self.param_

   def canMergeRange( self ):
      return True

   @staticmethod
   def enterRangeCmd( modes ):
      return "vlan " + multiRangeToCanonicalString(
         x.instanceKey() for x in modes )

MldSnoopingVlanConfigSaveMode.addCommandSequence( 'MldSnooping.vlanConfig' )

#-------------------------------------------------------------------------------
# Object used for saving commands in "config-mld-snooping" mode.
#-------------------------------------------------------------------------------
class MldSnoopingConfigSaveMode( MldSnoopingMode, CliSave.Mode ):
   def __init__( self, param ):
      MldSnoopingMode.__init__( self )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addCommandSequence( 'MldSnooping.global',
                                             after=[ IntfConfigMode ],
                                             before=[ 'Ira.routes',
                                                      'Ira.routes6' ] )
CliSave.GlobalConfigMode.addChildMode( MldSnoopingConfigSaveMode,
                                       after=[ IntfConfigMode ],
                                       before=[ 'Ira.routes', 'Ira.routes6' ] )
MldSnoopingConfigSaveMode.addCommandSequence( 'MldSnooping.config' )
MldSnoopingConfigSaveMode.addChildMode( MldSnoopingVlanConfigSaveMode,
                                    after=[ 'MldSnooping.config' ] )

@CliSave.saver( 'Bridging::IgmpSnooping::Config', 'bridging/mldsnooping/config',
                requireMounts=( 'bridging/hwcapabilities', ) )
def saveConfig( entity, root, requireMounts, options ):
   hwCapability = requireMounts[ 'bridging/hwcapabilities' ]
   if hwCapability.mldSnoopingSupported is False:
      return

   saveAll = options.saveAll
   if entity.mldSnoopingConfigured:
      mode = \
         root[ MldSnoopingConfigSaveMode ].getOrCreateModeInstance( entity.name )
      cmds = mode[ 'MldSnooping.config' ]
      if not entity.vlanDefaultEnabled:
         cmds.addCommand( 'disabled' )
      elif saveAll:
         cmds.addCommand( 'no disabled' )
   elif saveAll:
      cmds = root[ 'MldSnooping.global' ]
      cmds.addCommand( 'no mld snooping' )

def getVlanConfig( entity, saveAll ):
   cmds = []
   if entity.enabled == 'vlanStateDisabled':
      cmds.append( 'disabled' )
   elif saveAll:
      cmds.append( 'no disabled' )

   def appendIntfList( cmdPrefix, intfs ):
      for combinedStr in intfListToCanonical( intfs ):
         combinedStr = combinedStr.replace( "Et", "Ethernet" )
         combinedStr = combinedStr.replace( "Po", "Port-Channel" )
         cmds.append( cmdPrefix % combinedStr )

   def vlanConfigHelper( intfList, cmdPrefix ):
      # We don't expect any other intfs except for Ethernet and
      # Port-Channel but not asserting here incase if we ever have to add
      # non Ethernet or Port-Channel intf directly from Sysdb
      intfs = [ intf for intf in intfList
                if ( 'Ethernet' in intf ) or ( 'Port-Channel' in intf ) ]

      if not intfs:
         return

      SubIntfId = Tac.Type( 'Arnet::SubIntfId' )
      subIntfs = list( filter( SubIntfId.isSubIntfId, intfs ) )
      intfs = list( set( intfs ) - set( subIntfs ) )
      appendIntfList( cmdPrefix, subIntfs )
      appendIntfList( cmdPrefix, intfs )

   groupAddrs = list( entity.ipGroup )
   for groupAddr in sorted( groupAddrs ):
      ipGroup = entity.ipGroup[ groupAddr ]
      cmdPrefix = "member %s interface " % groupAddr
      cmdPrefix += "%s"
      vlanConfigHelper( ipGroup.intf, cmdPrefix )

   querierConfig = entity.vlanQuerierConfig
   if querierConfig:
      # The default value of these fields is 0 for
      # a default
      if querierConfig.queryIntervalConfigured:
         cmds.append( "querier interval %d seconds"
            % ( querierConfig.queryIntervalConfigured ) )
      elif saveAll:
         cmds.append( "querier interval 125 seconds" )
      if querierConfig.queryResponseIntervalConfigured:
         cmds.append( "querier max response interval %d seconds"
            % querierConfig.queryResponseIntervalConfigured )
      elif saveAll:
         cmds.append( "querier max response interval 10 seconds" )
      if querierConfig.lastMemberQueryCountConfigured:
         cmds.append( "querier last member query count %d"
            % querierConfig.lastMemberQueryCountConfigured )
      elif saveAll:
         cmds.append( "querier last member query count 2" )
      if querierConfig.lastMemberQueryIntervalConfigured:
         cmds.append( "querier last member query interval %d seconds"
            % querierConfig.lastMemberQueryIntervalConfigured )
      elif saveAll:
         cmds.append( "querier last member query interval 1 seconds" )
      if querierConfig.enabledConfigured != 'vlanQuerierDefault':
         cmds.append( "querier" )
      elif saveAll:
         cmds.append( "no querier" )

   vlanConfigHelper( entity.routerIntf, 'multicast-router interface %s' )

   if Toggles.GmpToggleLib.toggleMldReportFloodingEnabled():
      vlanConfigHelper( entity.switchPort, 'report-flooding switch-port %s' )
      if entity.reportFloodingAllSwitchPorts:
         cmds.append( 'report-flooding switch-port all' )
      elif saveAll:
         cmds.append( 'no report-flooding switch-port all' )

   return cmds

@CliSave.saver( 'Bridging::IgmpSnooping::Config', 'bridging/mldsnooping/config',
                requireMounts=( 'bridging/hwcapabilities', ) )
def saveVlanConfig( entity, root, requireMounts, options ):
   hwCapability = requireMounts[ 'bridging/hwcapabilities' ]
   if hwCapability.mldSnoopingSupported is False:
      return

   if not entity.mldSnoopingConfigured:
      return
   for v in sorted( entity.vlanConfig, key=int ):
      vlanEntity = entity.vlanConfig[ v ]
      cmdList = getVlanConfig( vlanEntity, options.saveAll )
      mode = \
         root[ MldSnoopingConfigSaveMode ].getOrCreateModeInstance( entity.name )
      mode = mode[ MldSnoopingVlanConfigSaveMode ].getOrCreateModeInstance( v )
      cmds = mode[ 'MldSnooping.vlanConfig' ]
      for cmd in cmdList:
         cmds.addCommand( cmd )
