# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# pylint: disable-msg=ungrouped-imports
import CliSave
import MplsLib
import Tac
import Toggles.MplsToggleLib
from CliMode.Mpls import ( StaticMulticastModeBase,
                           TunnelStaticModeBase,
                           TunnelTerminationModeBase,
                           TunnelTerminationVrfModeBase,
                           LfibStitchingPreferencesModeBase )
from CliSavePlugin.IntfCliSave import IntfConfigMode
from MplsLib import tunTypeEnumDict
from MplsTypeLib import tunnelTypeRevXlate
from RoutingIntfUtils import allRoutingProtocolIntfNames
from TypeFuture import TacLazyType

CliSave.GlobalConfigMode.addCommandSequence( 'Mpls.labelRange',
                                             after=[ 'Ira.routes' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Mpls.routes',
                                             after=[ 'Mpls.labelRange' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Mpls.config', after=[ 'Ira.routing' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Tunnel.static.config',
                                             after=[ 'Ira.routing' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Mpls.staticL3Vpn',
                                             after=[ 'Tunnel.static.config' ] )
IntfConfigMode.addCommandSequence( 'Mpls.intf', after=[ 'Ira.ipIntf' ] )

FecIdIntfId = TacLazyType( 'Arnet::FecIdIntfId' )
MplsVia = Tac.Type( 'Tunnel::TunnelTable::MplsVia' )
TacDyTunIntfId = Tac.Type( 'Arnet::DynamicTunnelIntfId' )
LabelRangeInfo = TacLazyType( 'Mpls::LabelRangeInfo' )
QosMap = TacLazyType( 'QosLib::QosMap' )

#mpls static top-label 14460 00fe:11c7::1245:0000 swap-label 12902 metric 1
#mpls static top-label 14500 00fd::1234:0001 pop payload-type ipv4
#mpls label dynamic-range 1000 2000

@CliSave.saver( 'Mpls::RouteConfigInput', 'routing/mpls/route/input/cli' )
def saveRoutingTable( entity, root, requireMounts, options ):
   metricDefaultValue = Tac.Type( 'Mpls::Constants' ).metricDefault

   routeCmd = 'mpls static top-label'

   for rk in sorted( entity.route, key=lambda rm: ( rm.topLabel, rm.metric ) ):
      route = entity.route[ rk ]
      if route.dynamic:
         continue

      def renderVia( via, isBackup=False ):
         # This variable tracks in which cases we don't want to show LSP name since
         # there are few cases like "mpls static top-label <inLabel> debug
         # tunnel-type ... " for which we don't support LSP name configuration but
         # since the entity is shared, it can be displayed in this configuration
         # during ConfigSave.
         displayLspName = True
         # pylint: disable-msg=W0640, cell-var-from-loop
         cmd = "%s %s" % ( routeCmd, rk.routeKey.labelStack.cliString() )
         addPayload = True
         if via in route.viaToIndex:
            viaIndex = route.viaToIndex[ via ]
            if viaIndex.indexConfigured or options.saveAll:
               cmd += f" index {viaIndex.index}"

         if via.nexthopGroup:
            cmd += " nexthop-group %s" % via.nexthopGroup
            addPayload = False
         else:
            if via.intf:
               if TacDyTunIntfId.isDynamicTunnelIntfId( via.intf ):
                  tunnelId = TacDyTunIntfId.tunnelId( via.intf )
                  tunnelId = Tac.Value( 'Tunnel::TunnelTable::TunnelId', tunnelId )
                  tunTypeEnum = tunnelId.tunnelType()
                  tunType = tunTypeEnumDict[ tunTypeEnum ]
                  tunInd = tunnelId.tunnelIndex()
                  cmd += ' debug' + ( ' backup' if isBackup else '' )
                  cmd += ' tunnel-type %s tunnel-index %d' \
                         % ( tunType, tunInd )
                  addPayload = False
                  # Do not support LSP name config for this CLI cmd.
                  displayLspName = False
               elif isBackup:
                  cmd += ' debug backup' if isBackup else ''
                  cmd += " %s" % via.intf
                  cmd += ' ' + via.nextHop.stringValue
                  addPayload = False
                  # Do not support LSP name config for this CLI cmd.
                  displayLspName = False
               else:
                  cmd += " %s" % via.intf
                  cmd += ' ' + via.nextHop.stringValue
                  # Reset to displaying the LSP name
                  displayLspName = True
            else:
               # Reset to displaying the LSP name
               displayLspName = True
               cmd += ' ' + via.nextHop.stringValue

         if via.isBgpPeerMonitored:
            cmd += " bgp peer"
            assert via.bgpPeer or via.nextHop
            if via.bgpPeer:
               cmd += " %s" % via.bgpPeer.stringValue

         if via.labelAction == 'swap':
            cmd += ' swap-label ' + str( via.outLabelStack.topLabel() )
         if via.labelAction == 'pop':
            cmd += ' pop'
            if via.payloadType != 'autoDecide':
               cmd += ' payload-type ' + via.payloadType
            elif addPayload:
               cmd += ' payload-type auto'
            if via.skipEgressAcl:
               cmd += ' access-list bypass'
         # If CLI supports LSP name configuration then only present in 'show run'
         # pylint: disable-msg=W0640, cell-var-from-loop
         if displayLspName and route.lspName:
            if route.lspNameConfigured or options.saveAll:
               cmd += f" name {route.lspName}"
         if rk.metric != metricDefaultValue or options.saveAll:
            cmd += ' metric ' + str( rk.metric )
         if via.weight != 1:
            cmd += ' weight ' + str( via.weight )

         root[ 'Mpls.routes' ].addCommand( cmd )
      for via in sorted( route.via ):
         renderVia( via )
      for via in sorted( route.backupVia ):
         renderVia( via, isBackup=True )

@CliSave.saver( 'Mpls::Config', 'routing/mpls/config',
                requireMounts=( 'interface/config/all', 'interface/status/all',
                                  'routing/hardware/mpls/capability' ) )
def saveConfig( entity, root, requireMounts, options ):
   if entity.mplsRouting:
      root[ 'Mpls.config' ].addCommand( "mpls ip" )
   elif options.saveAll:
      root[ 'Mpls.config' ].addCommand( "no mpls ip" )

   cmd = "mpls next-hop resolution allow default-route"
   # pylint: disable-next=singleton-comparison
   if entity.nexthopResolutionAllowDefaultRoute == True:
      root[ 'Mpls.config' ].addCommand( cmd )
   elif options.saveAll:
      root[ 'Mpls.config' ].addCommand( "no %s" % cmd )

   cmdStem = "mpls tunnel termination model"
   cmd = cmdStem
   hwCapability = requireMounts[ 'routing/hardware/mpls/capability' ]
   ttlMode = entity.labelTerminationTtlMode
   dscpMode = entity.labelTerminationDscpMode
   # ttl=pipe, dscp=uniform is not supported in Jericho
   # dscp=uniform is not supported in J2 for now
   ret = isTtlDscpModelSupported( hwCapability, ttlMode, dscpMode )
   # pylint: disable-next=singleton-comparison,consider-using-in
   if ret == None or ret == False:
      pass
   elif ttlMode != "platformDefault" and dscpMode != "platformDefault":
      cmd += ' ttl %s' % ttlMode
      cmd += ' dscp %s' % dscpMode
   if cmd != cmdStem:
      root[ 'Mpls.config' ].addCommand( cmd )
   elif options.saveAll:
      root[ 'Mpls.config' ].addCommand( "no " + cmd )

   cmd = "mpls tunnel encapsulation model"
   ttlMode = entity.labelEncapsulationTtlMode
   expMode = entity.labelEncapsulationExpMode
   ttlValue = entity.labelEncapsulationTtlValue
   expValue = entity.labelEncapsulationExpValue
   if ttlMode == "pipe" or expMode == "pipe":
      cmd += ' ttl %s' % ttlMode
      if ttlMode == "pipe":
         cmd += ' %s' % ttlValue
      cmd += ' exp %s' % expMode
      if expMode == "pipe":
         cmd += ' %s' % expValue
      root[ 'Mpls.config' ].addCommand( cmd )
   elif options.saveAll:
      root[ 'Mpls.config' ].addCommand( cmd + " ttl uniform exp uniform" )

   # The meaning of the default "undefined" differs between platforms.
   # Therefore only output the cmd if it has been explicity set to
   # "pipe" or "uniform"
   cmdStem = 'mpls tunnel termination php model'
   cmd = cmdStem
   ttlMode = entity.labelPhpTtlMode
   dscpMode = entity.labelPhpDscpMode
   if ttlMode != 'undefinedTtlMode':
      cmd += ' ttl %s' % ttlMode
   if dscpMode != 'undefinedTtlMode':
      cmd += ' dscp %s' % dscpMode
   if cmd != cmdStem:
      root[ 'Mpls.config' ].addCommand( cmd )
   elif options.saveAll:
      root[ 'Mpls.config' ].addCommand( "no " + cmd )

   cmd = "mpls fec ip sharing disabled"
   if not entity.optimizeStaticRoutes:
      root[ 'Mpls.config' ].addCommand( cmd )
   elif options.saveAll:
      root[ 'Mpls.config' ].addCommand( "no " + cmd )

   cmd = "mpls segment-routing is-is fec ip sharing disabled"
   if not entity.optimizeIsisSrLfibRoutes:
      root[ 'Mpls.config' ].addCommand( cmd )
   elif options.saveAll:
      root[ 'Mpls.config' ].addCommand( "no " + cmd )

   if Toggles.MplsToggleLib.toggleMplsEntropyLabelSupportEnabled():
      cmd = "mpls entropy-label pop"
      if entity.entropyLabelPop:
         root[ 'Mpls.config' ].addCommand( cmd )
      elif options.saveAll:
         root[ 'Mpls.config' ].addCommand( 'no ' + cmd )

   if ( hwCapability.mplsModifyParsingSupported and
       # pylint: disable-next=singleton-comparison
       ( entity.speculativeParsingConfig != None or options.saveAll ) ):
      defaultParsingCfg = hwCapability.mplsParsingDefault
      curParsingCfg = entity.speculativeParsingConfig
      if not curParsingCfg:
         curParsingCfg = defaultParsingCfg
      parsedTypes = [
         ( 'ipv4', curParsingCfg.parseIpv4, defaultParsingCfg.parseIpv4 ),
         ( 'ipv6', curParsingCfg.parseIpv6, defaultParsingCfg.parseIpv6 ),
         ( 'ethernet', curParsingCfg.parseEth, defaultParsingCfg.parseEth ),
         ( 'control-word', curParsingCfg.parsePwCw, defaultParsingCfg.parsePwCw ), ]
      for packetType, curParsing, defaultParsing in parsedTypes:
         if options.saveAll or curParsing != defaultParsing:
            cmd = 'mpls parsing speculative %s' % packetType
            if not curParsing:
               cmd = 'no ' + cmd
            root[ 'Mpls.config' ].addCommand( cmd )

   if Toggles.MplsToggleLib.toggleLfibStitchingConfigurablePreferencesEnabled():
      cmd = "mpls lfib stitching preferences"
      if entity.lfibStitchingPrefConfigEnabled != \
            entity.lfibStitchingPrefConfigEnabledDefault:
         root[ LfibStitchingPreferencesSaveMode ].getOrCreateModeInstance( None )
      elif options.saveAll:
         root[ 'Mpls.config' ].addCommand( 'no ' + cmd )

   # Save Interface level config
   saveIntfConfig( entity, root, requireMounts, options )

def saveIntfConfig( entity, root, requireMounts, options ):

   if options.saveAll:
      intfs = allRoutingProtocolIntfNames( requireMounts )
   else:
      intfs = entity.mplsRoutingDisabledIntf

   for intf in intfs:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmdSequence = mode[ 'Mpls.intf' ]
      if intf not in entity.mplsRoutingDisabledIntf and options.saveAll:
         cmdSequence.addCommand( 'mpls ip' )
      elif ( intf in entity.mplsRoutingDisabledIntf and
             entity.mplsRoutingDisabledIntf[ intf ] ):
         cmdSequence.addCommand( 'no mpls ip' )

CliSave.GlobalConfigMode.addCommandSequence( 'Mpls.tunnel.config' )

def isTtlDscpModelSupported( hwCapability, ttlMode, dscpMode ):
   if ttlMode == "uniform" and dscpMode == "uniform":
      return hwCapability.mplsTtlUniformDscpUniformModelSupported
   elif ttlMode == "uniform" and dscpMode == "pipe":
      return hwCapability.mplsTtlUniformDscpPipeModelSupported
   elif ttlMode == "pipe" and dscpMode == "uniform":
      return hwCapability.mplsTtlPipeDscpUniformModelSupported
   elif ttlMode == "pipe" and dscpMode == "pipe":
      return hwCapability.mplsTtlPipeDscpPipeModelSupported
   else:
      return None

@CliSave.saver( 'Tunnel::MplsTunnelConfig', 'routing/mpls/tunnel/config' )
def saveMplsTunnelConfig( entity, root, requireMounts, options ):
   if Toggles.MplsToggleLib.toggleMplsEntropyLabelSupportEnabled():
      cmd = "mpls tunnel entropy-label push"
      if entity.entropyLabelPush:
         root[ 'Mpls.tunnel.config' ].addCommand( cmd )
      elif options.saveAll:
         root[ 'Mpls.tunnel.config' ].addCommand( 'no ' + cmd )

@CliSave.saver( 'Mpls::Config', 'routing/mpls/config' )
def saveLabelRanges( entity, root, requireMounts, options ):
   for rangeType in sorted( entity.labelRange ):
      saveLabelRange( entity, root, options, rangeType )

def saveLabelRange( entity, root, options, rangeType ):
   defaultRange = MplsLib.labelRangeDefault( entity, rangeType )
   value = MplsLib.labelRange( entity, rangeType )
   cliRangeType = rangeType if rangeType != LabelRangeInfo.rangeTypeL2evpnSharedEs \
                  else 'l2evpn ethernet-segment'
   if options.saveAll or value != defaultRange:
      base = value.base
      size = value.size
      baseCmd = 'mpls label range ' + cliRangeType
      root[ 'Mpls.labelRange' ].addCommand( baseCmd + ' %d %d' % ( base, size ) )

@CliSave.saver( 'Mpls::Config', 'routing/mpls/config' )
def saveLookupLabelCount( entity, root, requireMounts, options ):
   if entity.mplsLookupLabelCount != 1 or options.saveAll:
      root[ 'Mpls.config' ].addCommand( 'mpls lookup label count %d'
                                        % entity.mplsLookupLabelCount )

@CliSave.saver( 'Mpls::Config', 'routing/mpls/config' )
def saveLookupFallbackIp( entity, root, requireMounts, options ):
   if entity.mplsLookupFallbackIp:
      root[ 'Mpls.config' ].addCommand( 'mpls lookup fallback ip' )

class TunnelStaticSaveMode( TunnelStaticModeBase, CliSave.Mode ):
   def __init__( self, param ):
      tunName, tep = param
      TunnelStaticModeBase.__init__( self, tunName, tep )
      CliSave.Mode.__init__( self, tunName )

   def instanceKey( self ):
      return self.tunName

CliSave.GlobalConfigMode.addCommandSequence( 'Mpls.staticTunnelConfig',
                                             after=[ 'Tunnel.static.config' ] )
CliSave.GlobalConfigMode.addChildMode( TunnelStaticSaveMode,
                                       after=[ 'Mpls.staticTunnelConfig' ] )
TunnelStaticSaveMode.addCommandSequence( 'Mpls.staticTunnelConfigVias' )

@CliSave.saver( 'Tunnel::Static::Config', 'tunnel/static/config' )
def saveStaticTunnelConfig( entity, root, requireMounts, options ):
   for name in entity.entry:
      staticTunnelConfigEntry = entity.entry[ name ]
      if len( staticTunnelConfigEntry.via ) > 1:
         assert staticTunnelConfigEntry.inStaticTunnelMode
      if staticTunnelConfigEntry.inStaticTunnelMode:
         mode = root[ TunnelStaticSaveMode ].getOrCreateModeInstance(
            ( staticTunnelConfigEntry.name, staticTunnelConfigEntry.tep ) )
         cmds = mode[ 'Mpls.staticTunnelConfigVias' ]
         for via in staticTunnelConfigEntry.via:
            cmd = []
            cmd.append( 'via' )
            cmd.append( via.nexthop.stringValue )
            cmd.append( via.intfId )
            labelStack = []
            labelsObj = via.labels
            for idx in reversed( list( range( labelsObj.stackSize ) ) ):
               labelStack.append( str( labelsObj.labelStack( idx ) ) )
            if labelStack == [ "3" ]:
               cmd.append( 'imp-null-tunnel' )
            else:
               cmd.append( 'label-stack' )
               cmd += labelStack
            cmds.addCommand( ' '.join( cmd ) )
      else:
         for via in staticTunnelConfigEntry.via:
            saveStaticTunnelConfigEntryVia( root,
                                            staticTunnelConfigEntry,
                                            via,
                                            False )
            break # This case will have only one via at most
      if staticTunnelConfigEntry.backupVia != MplsVia():
         saveStaticTunnelConfigEntryVia( root,
                                         staticTunnelConfigEntry,
                                         staticTunnelConfigEntry.backupVia,
                                         True )

def saveStaticTunnelConfigEntryVia( root, configEntry, via, backup ):
   cmd = []
   cmd.append( 'mpls' )
   resolving = FecIdIntfId.isFecIdIntfId( via.intfId )
   resolvingTunnel = TacDyTunIntfId.isDynamicTunnelIntfId( via.intfId )
   if backup or resolving or resolvingTunnel:
      cmd.append( 'debug' )
   if backup:
      cmd.append( 'backup' )
   cmd.append( 'tunnel static' )
   cmd.append( configEntry.name )
   if resolving:
      cmd.append( 'resolving' )
      cmd.append( configEntry.tep.stringValue )
   elif resolvingTunnel:
      cmd.append( 'resolving-tunnel' )
      cmd.append( configEntry.tep.stringValue )
      tunnelId = TacDyTunIntfId.tunnelId( via.intfId )
      tunnelId = Tac.Value( 'Tunnel::TunnelTable::TunnelId', tunnelId )
      tunType = tunnelTypeRevXlate[ tunnelId.tunnelType() ]
      cmd.append( 'tunnel-type %s' % tunType )
      tunIndex = tunnelId.tunnelIndex()
      cmd.append( 'tunnel-index %d' % tunIndex )
   else:
      cmd.append( configEntry.tep.stringValue )
      if via.intfId != 'Null0':
         cmd.append( via.nexthop.stringValue )
      cmd.append( via.intfId )
   labelStack = []
   labelsObj = via.labels
   for idx in reversed( list( range( labelsObj.stackSize ) ) ):
      labelStack.append( str( labelsObj.labelStack( idx ) ) )
   if labelStack == [ "3" ]:
      cmd.append( 'imp-null-tunnel' )
   elif labelStack:
      cmd.append( 'label-stack' )
      cmd += labelStack
   root[ 'Tunnel.static.config' ].addCommand( ' '.join( cmd ) )

@CliSave.saver( 'Mpls::VrfLabelConfigInput', 'routing/mpls/vrfLabel/input/cli' )
def saveVrfLabel( entity, root, requireMounts, options ):
   staticVrfLabelCmd = 'mpls static vrf-label {} vrf {}'

   mplsStatic = root[ 'Mpls.staticL3Vpn' ]
   for vrfLabel in entity.vrfLabel.values():
      cmd = staticVrfLabelCmd.format( vrfLabel.label, vrfLabel.vrfName )
      mplsStatic.addCommand( cmd )

class MplsTunnelTerminationSaveMode( TunnelTerminationModeBase, CliSave.Mode ):
   def __init__( self, param ):
      TunnelTerminationModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class MplsTunnelTerminationVrfSaveMode( TunnelTerminationVrfModeBase, CliSave.Mode ):
   def __init__( self, vrfName ):
      self.vrfName = vrfName
      TunnelTerminationVrfModeBase.__init__( self, vrfName )
      CliSave.Mode.__init__( self, vrfName )

CliSave.GlobalConfigMode.addChildMode( MplsTunnelTerminationSaveMode,
                                       after=[ 'Mpls.config' ] )
MplsTunnelTerminationSaveMode.addCommandSequence( 'Mpls.tunnelTerm',
      before=[ MplsTunnelTerminationVrfSaveMode ] )

MplsTunnelTerminationSaveMode.addChildMode( MplsTunnelTerminationVrfSaveMode,
                                            after=[ 'Mpls.tunnelTerm' ] )
MplsTunnelTerminationVrfSaveMode.addCommandSequence( 'Mpls.tunnelTermVrf' )

@CliSave.saver( 'Mpls::VrfQosMapConfig', 'routing/mpls/vrfLabel/qosMap/input/cli' )
def saveVrfQosMapName( entity, root, requireMounts, options ):
   mplsVrfQosCmd = 'qos map dscp to traffic-class {}'

   mplsTunnelTermSaveMode = root[
         MplsTunnelTerminationSaveMode ].getOrCreateModeInstance( None )
   for vrfQosMap in entity.vrfQosMap.values():
      vrfMode = mplsTunnelTermSaveMode[
            MplsTunnelTerminationVrfSaveMode ].getOrCreateModeInstance(
                  vrfQosMap.vrfName )
      cmds = vrfMode[ 'Mpls.tunnelTermVrf' ]
      if vrfQosMap.qosMap != QosMap():
         cmd = mplsVrfQosCmd.format( vrfQosMap.qosMap.mapName )
         cmds.addCommand( cmd )

class StaticMulticastSaveMode( StaticMulticastModeBase, CliSave.Mode ):

   def __init__( self, inLabel ):
      StaticMulticastModeBase.__init__( self, inLabel )
      CliSave.Mode.__init__( self, inLabel )

CliSave.GlobalConfigMode.addCommandSequence( 'Mpls.staticMulticast',
                                             after=[ 'Mpls.config' ] )
CliSave.GlobalConfigMode.addChildMode( StaticMulticastSaveMode,
                                       after=[ 'Mpls.staticMulticast' ] )
StaticMulticastSaveMode.addCommandSequence( 'Mpls.routeConfig' )

@CliSave.saver( 'Mpls::LfibSysdbStatus', 'mpls/staticMcast/lfib' )
def saveStasticMcastLfib( entity, root, requireMounts, options ):
   for viaSet in entity.mplsViaSet.values():
      mode = root[ StaticMulticastSaveMode ].getOrCreateModeInstance( viaSet.label )
      cmds = mode[ 'Mpls.routeConfig' ]
      # sort it manually as TACC does not support ordered sets
      for via in sorted( viaSet.mplsVia ):
         nhAddr = via.nextHop.stringValue
         outLabel = via.outLabel.topLabel()
         cmd = "next-hop %s swap-label %s" % ( nhAddr, outLabel )
         cmds.addCommand( cmd )

class LfibStitchingPreferencesSaveMode( LfibStitchingPreferencesModeBase,
                                        CliSave.Mode ):
   def __init__( self, param ):
      LfibStitchingPreferencesModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( LfibStitchingPreferencesSaveMode,
                                       after=[ 'Mpls.config' ] )
LfibStitchingPreferencesSaveMode.addCommandSequence(
                                          'Mpls.lfibStitchingPreferences' )

def Plugin( entityManager ):
   pass
