# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliMode.MplsUtilsMode import MplsUtilsMode
from TypeFuture import TacLazyType
import Toggles.MplsUtilsToggleLib as Toggle

MplsOamStandard = TacLazyType( 'MplsUtils::MplsOamStandard' )

#-------------------------------------------------------------------------------
# Object used for saving commands in "mpls-ping" mode.
#-------------------------------------------------------------------------------
class MplsUtilsModeConfigMode( MplsUtilsMode, CliSave.Mode ):
   def __init__( self, param ):
      MplsUtilsMode.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( MplsUtilsModeConfigMode,
                                       after=[ 'Mpls.config' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Mpls.MplsUtilsConfig',
                                             after=[ 'Mpls.config' ] )

MplsUtilsModeConfigMode.addCommandSequence( 'Mpls.MplsUtils' )

def saveLspPingServiceAcls( config, root, requireMounts, options ):

   mode = root[ MplsUtilsModeConfigMode ].getSingletonInstance()
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   for aclType in [ 'ip', 'ipv6' ]:
      serviceAclVrfConfig = aclCpConfig.cpConfig[ aclType ].serviceAcl
      for vrf, serviceAclConfig in sorted( serviceAclVrfConfig.items() ):
         if ( 'lspPing' in serviceAclConfig.service and
              serviceAclConfig.service[ 'lspPing' ].aclName ):
            # pylint: disable-next=consider-using-f-string
            mode[ 'Mpls.MplsUtils' ].addCommand( '%s access-group %s vrf %s '
                  % ( aclType, serviceAclConfig.service[ 'lspPing' ].aclName, vrf ) )

def saveIcmpTunnel( config, root, requireMounts, options ):
   cmd = '{no}mpls icmp {token} tunneling'

   tokens = []
   tokens.append( ( 'ttl-exceeded', 'icmpTtlTunneling' ) )

   mplsHwCapability = requireMounts[ 'routing/hardware/mpls/capability' ]
   if mplsHwCapability.mplsFragNeededIcmpTunnelingSupported:
      tokens.append( ( 'fragmentation-needed', 'fragNeededTunneling' ) )

   for token, configAttr in sorted( tokens ):
      if getattr( config, configAttr ):
         addCmd = cmd.format( no='', token=token )
      elif options.saveAll:
         addCmd = cmd.format( no='no ', token=token )
      else:
         addCmd = None
      if addCmd:
         root[ 'Mpls.MplsUtilsConfig' ].addCommand( addCmd )

def saveIcmpSourceInterface( config, root, requireMounts, options ):
   cmd = '{no}mpls icmp {af} source-interface {interface}'
   addCmd = None
   if Toggle.toggleMplsIcmpSrcIpCliEnabled():
      if config.mplsIcmpIpv6SrcIntfId:
         addCmd = cmd.format( no='', af='ipv6',
                              interface=config.mplsIcmpIpv6SrcIntfId )
         root[ 'Mpls.MplsUtilsConfig' ].addCommand( addCmd )
      elif options.saveAll:
         addCmd = cmd.format( no='no ', af='ipv6', interface="" )
         root[ 'Mpls.MplsUtilsConfig' ].addCommand( addCmd )

   if Toggle.toggleMplsIcmpSrcIpCliV4Enabled():
      if config.mplsIcmpIpv4SrcIntfId:
         addCmd = cmd.format( no='', af='ip',
                              interface=config.mplsIcmpIpv4SrcIntfId )
         root[ 'Mpls.MplsUtilsConfig' ].addCommand( addCmd )
      elif options.saveAll:
         addCmd = cmd.format( no='no ', af='ip', interface="" )
         root[ 'Mpls.MplsUtilsConfig' ].addCommand( addCmd )

def saveMplsOamStandard( config, root, requireMounts, options ):
   baseCmd = 'mpls oam standard {standard}'
   if config.oamStandard == MplsOamStandard.ietf:
      cmd = baseCmd.format( standard=MplsOamStandard.ietf )
      root[ 'Mpls.MplsUtilsConfig' ].addCommand( cmd )
   elif options.saveAll:
      cmd = baseCmd.format( standard=MplsOamStandard.arista )
      root[ 'Mpls.MplsUtilsConfig' ].addCommand( cmd )

def saveMplsOamDownstreamValidation( config, root, requireMounts, options ):
   baseCmd = '{no}mpls oam downstream validation {toggle}'
   if config.dsMapValidation:
      cmd = baseCmd.format( no='', toggle='enabled' )
      root[ 'Mpls.MplsUtilsConfig' ].addCommand( cmd )
   elif options.saveAll:
      cmd = baseCmd.format( no='no ', toggle='enabled' )
      root[ 'Mpls.MplsUtilsConfig' ].addCommand( cmd )

@CliSave.saver( 'MplsUtils::Config', 'mplsutils/config',
                requireMounts=( 'routing/hardware/mpls/capability',
                                'acl/cpconfig/cli' ) )
def saveMplsUtilsConfig( config, root, requireMounts, options ):
   saveLspPingServiceAcls( config, root, requireMounts, options )
   saveIcmpTunnel( config, root, requireMounts, options )
   saveMplsOamStandard( config, root, requireMounts, options )
   saveMplsOamDownstreamValidation( config, root,
                                    requireMounts, options )
   saveIcmpSourceInterface( config, root, requireMounts, options )
