#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliSavePlugin.RoutingBgpCliSave import RouterBgpBaseAfConfigMode
from CliSavePlugin.RoutingBgpCliSave import RouterBgpBaseConfigMode
from CliSavePlugin.RoutingBgpCliSave import saveAfConfigCallbackDict
from Toggles.RoutingLibToggleLib import toggleSRv6MplsVpnEnabled
from Toggles.TunnelToggleLib import toggleDynamicGueTunnelsEnabled
from RouteMapLib import isAsdotConfigured
from BgpLib import bgpConfigAttrAfList, peerConfigAttrsAfMap, vpnAfTypeMapInv

RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.vpn-ipv4' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.vpn-ipv6' )

def saveMplsVpnPeerAfConfig( cmds, bgpConfig, peerKey, addrFamily ):
   if addrFamily not in [ 'vpn-ipv4', 'vpn-ipv6' ]:
      return
   peer = peerKey.stringValue
   peerConfig = bgpConfig.neighborConfig.get( peerKey )

   if peerConfig:
      attrName = peerConfigAttrsAfMap[ 'nexthopSelfSrcIntf' ][ addrFamily ]
      nhIntf = getattr( peerConfig, attrName )
      if nhIntf != getattr( peerConfig, attrName + "Default" ):
         cmds.append( 'neighbor %s encapsulation mpls next-hop-self '
                      'source-interface %s' % ( peer, nhIntf ) )

def saveMplsVpnConfig( bgpConfig, addrFamily, saveAllOrDetail=False ):
   cmds = []
   afiSafi = vpnAfTypeMapInv[ addrFamily ] if addrFamily in vpnAfTypeMapInv else None
   if addrFamily == 'vpn-ipv4':
      if toggleSRv6MplsVpnEnabled() and bgpConfig.afVpnV4Encap == 'encapSrv6':
         cmds.append( 'neighbor default encapsulation segment-routing ipv6 '
                      'locator %s' % bgpConfig.afVpnV4Srv6Locator )
      elif( bgpConfig.nexthopSelfSrcIntfVpnV4 !=
          bgpConfig.nexthopSelfSrcIntfVpnV4Default ):
         cmds.append( 'neighbor default encapsulation mpls '
                      'next-hop-self source-interface %s' %
                      bgpConfig.nexthopSelfSrcIntfVpnV4 )
      if bgpConfig.nexthopResolutionVrfUniRibAfVpnV4 == 'isTrue':
         # Since this is a triState bool, the command needs to be printed only when
         # the value of this attribute is 'isTrue'. For the other two values, we
         # dont want this command to be in the output.
         cmds.append( 'next-hop resolution ribs vrf-unicast-rib' )
      elif saveAllOrDetail:
         cmds.append( 'no next-hop resolution ribs vrf-unicast-rib' )

      if toggleDynamicGueTunnelsEnabled():
         if bgpConfig.nexthopResolutionUdpTunnelAfVpnV4 == 'isTrue':
            cmds.append( 'next-hop resolution ribs udp-tunnel' )
         elif saveAllOrDetail:
            cmds.append( 'no next-hop resolution ribs udp-tunnel' )

         if bgpConfig.nhOrigTunnelProfileEncapTypeVpnV4 == 'udp':
            policy = bgpConfig.nhOrigTunnelProfilePolicyNameVpnV4
            cmds.append( f'next-hop originate tunnel type udp policy {policy}' )
         elif saveAllOrDetail:
            cmds.append( 'no next-hop originate tunnel type udp' )

      if bgpConfig.labelAllocationDisabledAfVpnV4 == 'isTrue':
         cmds.append( 'mpls label allocation disabled' )
      elif saveAllOrDetail:
         cmds.append( 'no mpls label allocation disabled' )

      if bgpConfig.domainIdVpnV4 != bgpConfig.domainIdVpnV4Default:
         cmds.append( 'domain identifier %s'
                      % bgpConfig.domainIdVpnV4.toStrepCli() )
      elif saveAllOrDetail:
         cmds.append( 'no domain identifier' )

      if ( vpnAfTypeMapInv[ addrFamily ] in
           bgpConfig.nexthopSelfLocalLabelAlloc ):
         cmds.append( 'neighbor default encapsulation mpls next-hop-self '
                      'received-vpnv4-routes' )
      elif saveAllOrDetail:
         cmds.append( 'no neighbor default encapsulation mpls next-hop-self '
                      'received-vpnv4-routes' )

      if bgpConfig.afVpnV4NexthopResolutionDisabled:
         cmds.append( 'next-hop resolution disabled' )
      elif saveAllOrDetail:
         cmds.append( 'no next-hop resolution disabled' )

      if ( bgpConfig.vpnPruningEnabledVpnV4 !=
           bgpConfig.vpnPruningEnabledVpnV4Default ):
         cmds.append( 'route import match-failure action discard' )
      elif saveAllOrDetail:
         cmds.append( 'no route import match-failure action discard' )

      if afiSafi in bgpConfig.vpnClientIbgpAttributeTunneling and \
            bgpConfig.vpnClientIbgpAttributeTunneling[ afiSafi ]:
         cmds.append( 'vpn client ibgp attribute tunneling' )
      elif saveAllOrDetail:
         cmds.append( 'no vpn client ibgp attribute tunneling' )

   if addrFamily == 'vpn-ipv6':
      if toggleSRv6MplsVpnEnabled() and bgpConfig.afVpnV6Encap == 'encapSrv6':
         cmds.append( 'neighbor default encapsulation segment-routing ipv6 '
                      'locator %s' % bgpConfig.afVpnV6Srv6Locator )
      elif( bgpConfig.nexthopSelfSrcIntfVpnV6 !=
          bgpConfig.nexthopSelfSrcIntfVpnV6Default ):
         cmds.append( 'neighbor default encapsulation mpls '
                      'next-hop-self source-interface %s' %
                      bgpConfig.nexthopSelfSrcIntfVpnV6 )
      if bgpConfig.nexthopResolutionVrfUniRibAfVpnV6 == 'isTrue':
         # Since this is a triState bool, the command needs to be printed only when
         # the value of this attribute is 'isTrue'. For the other two values, we
         # dont want this command to be in the output.
         cmds.append( 'next-hop resolution ribs vrf-unicast-rib' )
      elif saveAllOrDetail:
         cmds.append( 'no next-hop resolution ribs vrf-unicast-rib' )

      if toggleDynamicGueTunnelsEnabled():
         if bgpConfig.nexthopResolutionUdpTunnelAfVpnV6 == 'isTrue':
            cmds.append( 'next-hop resolution ribs udp-tunnel' )
         elif saveAllOrDetail:
            cmds.append( 'no next-hop resolution ribs udp-tunnel' )

         if bgpConfig.nhOrigTunnelProfileEncapTypeVpnV6 == 'udp':
            policy = bgpConfig.nhOrigTunnelProfilePolicyNameVpnV6
            cmds.append( f'next-hop originate tunnel type udp policy {policy}' )
         elif saveAllOrDetail:
            cmds.append( 'no next-hop originate tunnel type udp' )

      if bgpConfig.labelAllocationDisabledAfVpnV6 == 'isTrue':
         cmds.append( 'mpls label allocation disabled' )
      elif saveAllOrDetail:
         cmds.append( 'no mpls label allocation disabled' )

      if bgpConfig.domainIdVpnV6 != bgpConfig.domainIdVpnV6Default:
         cmds.append( 'domain identifier %s'
                      % bgpConfig.domainIdVpnV6.toStrepCli() )
      elif saveAllOrDetail:
         cmds.append( 'no domain identifier' )

      if ( vpnAfTypeMapInv[ addrFamily ] in
           bgpConfig.nexthopSelfLocalLabelAlloc ):
         cmds.append( 'neighbor default encapsulation mpls next-hop-self '
                      'received-vpnv6-routes' )
      elif saveAllOrDetail:
         cmds.append( 'no neighbor default encapsulation mpls next-hop-self '
                      'received-vpnv6-routes' )

      if bgpConfig.afVpnV6NexthopResolutionDisabled:
         cmds.append( 'next-hop resolution disabled' )
      elif saveAllOrDetail:
         cmds.append( 'no next-hop resolution disabled' )

      if ( bgpConfig.vpnPruningEnabledVpnV6 !=
           bgpConfig.vpnPruningEnabledVpnV6Default ):
         cmds.append( 'route import match-failure action discard' )
      elif saveAllOrDetail:
         cmds.append( 'no route import match-failure action discard' )

      if afiSafi in bgpConfig.vpnClientIbgpAttributeTunneling and \
            bgpConfig.vpnClientIbgpAttributeTunneling[ afiSafi ]:
         cmds.append( 'vpn client ibgp attribute tunneling' )
      elif saveAllOrDetail:
         cmds.append( 'no vpn client ibgp attribute tunneling' )

   for peerKey in bgpConfig.neighborConfig:
      saveMplsVpnPeerAfConfig( cmds, bgpConfig, peerKey, addrFamily )

   return cmds

# saveAfConfigCallbackDict would be used to display MplsVpn specific config in
# "show bgp config active" output.
# For  "show running" the config is displayed through saveMplsVpnAfConfig with
# CliSave.saver decorator.
saveAfConfigCallbackDict[ 'vpn-ipv4' ] = saveMplsVpnConfig
saveAfConfigCallbackDict[ 'vpn-ipv6' ] = saveMplsVpnConfig

@CliSave.saver( 'Routing::Bgp::Config', 'routing/bgp/config',
                requireMounts=( 'routing/bgp/asn/config', ) )
def saveMplsVpnAfConfig( bgpConfig, root, requireMounts, options ):
   if bgpConfig.asNumber == 0:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   for af in bgpConfigAttrAfList():
      saveAllOrDetail = options.saveAll or options.saveAllDetail
      cmds = saveMplsVpnConfig( bgpConfig, af, saveAllOrDetail=saveAllOrDetail )
      if cmds:
         bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
               ].getOrCreateModeInstance( af )
         for cmd in cmds:
            bgpAfMode[ 'Bgp.afConfig.%s' % af ].addCommand( cmd )
