#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave, Tac, Tracing
import McastCommonCliLib
from RoutingConsts import defaultStaticRoutePreference
from CliSavePlugin.MrouteCliSave import RouterMulticastBaseConfigMode
from CliSavePlugin.MrouteCliSave import RouterMulticastVrfConfigMode
from CliSavePlugin.MrouteCliSave import getCmdRoot
__defaultTraceHandle__ = Tracing.Handle( "MribCliSave" )
t1 = Tracing.trace1

CliSave.GlobalConfigMode.addCommandSequence( 'Ip.McastStatic',
                                             after=[ "Ip.Multicast" ] )
LegacyConfig = Tac.Type( "Routing::Multicast::MulticastLegacyConfig" )
RouterMulticastBaseConfigMode.addCommandSequence( 'McastStatic.config',
                                                  after=[ 'Multicast.config' ] )
RouterMulticastVrfConfigMode.addCommandSequence( 'McastStatic.vrf.config',
                                               after=[ 'Multicast.vrf.config' ] )

AddressFamily = Tac.Type( "Arnet::AddressFamily" )
LegacyConfig = Tac.Type( "Routing::Multicast::MulticastLegacyConfig" )
def _rpfKeyTuple( rk ):
   # vk is a Routing::Multicast::Static::Rpf
   return ( rk.intfId, rk.address )

def _routeKeyPrefixTuple( rk ):
   # rk is a Routing::Multicast::Static::RouteKeyPrefix
   TupleIpAddr = Tac.newInstance( 'Arnet::IpAddr', 0 )
   TupleIpAddr.stringValue = rk.v4Prefix.address
   IpAddrValue = TupleIpAddr.value
   return ( IpAddrValue, rk.v4Prefix.len )

def saveConfigCommon( mribConfigColl, root, requireMounts, options, af ):
   saveAll = options.saveAll
   # Save the default config only if the platform supports multicast routing
   if not McastCommonCliLib.mcastRoutingSupported(
         None,
         # pylint: disable-next=consider-using-f-string
         requireMounts[ 'routing%s/hardware/status' % ( 
            '6' if af == AddressFamily.ipv6 else '' ) ] ):
      saveAll = False

   for vrf in mribConfigColl.vrfConfig:
      t1( "Processing Vrf %s" % vrf ) # pylint: disable=consider-using-f-string
      mribConfig = mribConfigColl.vrfConfig[ vrf ]

      if not mribConfig.route and not saveAll:
         continue

      cmds = getCmdRoot( root, vrf, af, LegacyConfig.ipMode )

      # Save the configured static mroutes
      for prefix in sorted( mribConfig.route, key=_routeKeyPrefixTuple ):
         rkPrefix = mribConfig.route[ prefix ]
         for preference in rkPrefix.pref:
            for rpfKey in sorted( rkPrefix.pref[ preference ].rpf, \
                  key=_rpfKeyTuple ):
               command = "rpf route " + prefix.stringValue
               if rpfKey.rpfType == 'intf':
                  command += " " + rpfKey.intfId
               else:
                  command += " " + rpfKey.address.stringValue
               if preference != defaultStaticRoutePreference or saveAll:
                  # pylint: disable-next=consider-using-f-string
                  command += " %d" % preference
               cmds.addCommand( command )


@CliSave.saver( 'Routing::Multicast::Rib::ConfigColl',
                'routing/multicast/rib/config',
                requireMounts = ( 'routing/multicast/legacyconfig',
                                  'routing/hardware/status', ) )
def saveConfigIpv4( mribConfigColl, root, requireMounts, options ):
   saveConfigCommon( mribConfigColl, root, requireMounts, options,
                     AddressFamily.ipv4 )

@CliSave.saver( 'Routing::Multicast::Rib::ConfigColl',
                'routing6/multicast/rib/config',
                requireMounts = ( 'routing/multicast/legacyconfig',
                                  'routing6/hardware/status', ) )
def saveConfigIpv6( mribConfigColl, root, requireMounts, options ):
   saveConfigCommon( mribConfigColl, root, requireMounts, options,
                     AddressFamily.ipv6 )
