# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from BgpLib import bgpConfigAttrAfList
from CliMode.Mvpn import (
   MvpnIpv4VrfMode,
   MvpnIpv4VrfSpmsiMode,
   MvpnIpv4VrfIntraAsIPmsiMode
)
from CliSavePlugin.RoutingBgpCliSave import (
   RouterBgpBaseAfConfigMode,
   RouterBgpBaseConfigMode,
   RouterBgpVrfConfigMode,
   RouterBgpVrfAfConfigMode,
   saveAfConfigCallbackDict
)
from RouteMapLib import isAsdotConfigured
from Toggles import (
   MvpnLibToggleLib,
   TunnelToggleLib
)

class MvpnIpv4VrfConfigMode( MvpnIpv4VrfMode, CliSave.Mode ):
   def __init__( self, param ):
      MvpnIpv4VrfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class MvpnIpv4VrfSpmsiConfigMode( MvpnIpv4VrfSpmsiMode, CliSave.Mode ):
   def __init__( self, param ):
      MvpnIpv4VrfSpmsiMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class MvpnIpv4VrfIntraAsIPmsiConfigMode( MvpnIpv4VrfIntraAsIPmsiMode, CliSave.Mode ):
   def __init__( self, param ):
      MvpnIpv4VrfIntraAsIPmsiMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

if MvpnLibToggleLib.toggleMvpnSelectiveTunnelsEnabled():
   RouterBgpVrfConfigMode.addChildMode( MvpnIpv4VrfConfigMode,
                                        after=[ RouterBgpVrfAfConfigMode ] )
   MvpnIpv4VrfConfigMode.addCommandSequence( 'Mvpn.ipv4' )
   MvpnIpv4VrfConfigMode.addChildMode( MvpnIpv4VrfSpmsiConfigMode )
   MvpnIpv4VrfSpmsiConfigMode.addCommandSequence( 'Mvpn.ipv4.spmsi' )
   MvpnIpv4VrfConfigMode.addChildMode( MvpnIpv4VrfIntraAsIPmsiConfigMode )
   MvpnIpv4VrfIntraAsIPmsiConfigMode.addCommandSequence( 'Mvpn.ipv4.ipmsi' )


RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.mvpn-ipv4' )

def saveMvpnConfig( bgpConfig, addrFamily, options=None ):
   cmds = []
   if addrFamily == 'mvpn-ipv4':
      if bgpConfig.afMvpnV4Encap == 'encapMpls':
         cmd = 'neighbor default encapsulation mpls protocol {0} p2mp'
         if bgpConfig.afMvpnV4PmsiTunnelType == 'mLdpP2mpLsp':
            cmds.append( cmd.format( 'mldp' ) )
         elif bgpConfig.afMvpnV4PmsiTunnelType == 'rsvpTeP2mpLsp':
            cmds.append( cmd.format( 'rsvp' ) )
         if ( bgpConfig.nexthopSelfSrcIntfMvpnV4 !=
              bgpConfig.nexthopSelfSrcIntfMvpnV4Default ):
            cmds.append( 'neighbor default encapsulation mpls next-hop-self '
                         f'source-interface {bgpConfig.nexthopSelfSrcIntfMvpnV4}' )
      if bgpConfig.afMvpnV4RsvpP2mpTunnelProfile != '':
         cmds.append( 'rsvp p2mp tunnel profile '
                      f'{bgpConfig.afMvpnV4RsvpP2mpTunnelProfile}' )
      elif TunnelToggleLib.toggleMvpnRsvpP2mpEnabled() and options and (
           options.saveAll or options.saveAllDetail ):
         cmds.append( 'no rsvp p2mp tunnel profile' )
   return cmds

# saveAfConfigCallbackDict will be used to display Mvpn specific config in
# 'show bgp config active' output.
# For 'show running', the config is displayed through saveMvpnAfConfig with
# CliSave.saver decorator.
saveAfConfigCallbackDict[ 'mvpn-ipv4' ] = saveMvpnConfig

@CliSave.saver( 'Routing::Bgp::Config', 'routing/bgp/config',
                requireMounts=( 'routing/bgp/asn/config', ) )
def saveMvpnAfConfig( bgpConfig, root, requireMounts, options ):

   if bgpConfig.asNumber == 0:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   for af in bgpConfigAttrAfList():
      cmds = saveMvpnConfig( bgpConfig, af, options )
      if cmds:
         bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
               ].getOrCreateModeInstance( af )
         for cmd in cmds:
            # pylint: disable-next=consider-using-f-string
            bgpAfMode[ 'Bgp.afConfig.%s' % af ].addCommand( cmd )

def saveMvpnVrfIpv4RsvpP2mpTunnelProfile( vrfName, vrfBgpConfig, mvpnMode, options ):
   if vrfBgpConfig.afMvpnV4RsvpP2mpTunnelProfile != '':
      mvpnMode[ 'Mvpn.ipv4' ].addCommand( 'rsvp p2mp tunnel profile '
                              f'{vrfBgpConfig.afMvpnV4RsvpP2mpTunnelProfile}' )
   elif TunnelToggleLib.toggleMvpnRsvpP2mpEnabled() and ( options.saveAll or
        options.saveAllDetail ):
      mvpnMode[ 'Mvpn.ipv4' ].addCommand( 'no rsvp p2mp tunnel profile' )

def saveMvpnVrfIpv4PmsiTunnelProtocol( vrfBgpConfig, mvpnMode ):
   if vrfBgpConfig.afMvpnV4Encap == 'encapMpls':
      cmd = 'pmsi-tunnel protocol {0} p2mp'
      if vrfBgpConfig.afMvpnV4PmsiTunnelType == 'mLdpP2mpLsp':
         mvpnMode[ 'Mvpn.ipv4' ].addCommand( cmd.format( 'mldp' ) )
      elif vrfBgpConfig.afMvpnV4PmsiTunnelType == 'rsvpTeP2mpLsp':
         mvpnMode[ 'Mvpn.ipv4' ].addCommand( cmd.format( 'rsvp' ) )

def saveMvpnVrfIpv4SpmsiModeConfig( vrfName, vrfBgpConfig, mvpnMode, options ):
   cmds = []
   if vrfBgpConfig.afMvpnV4VrfSpmsiTunnelSourceGroup:
      cmds.append( 'p-tunnel source-group' )
   else:
      if options.saveAll or options.saveAllDetail:
         cmds.append( 'no p-tunnel source-group' )

   if vrfBgpConfig.afMvpnV4VrfSpmsiTunnelStarStar:
      cmds.append( 'p-tunnel star-star' )
   else:
      if options.saveAll or options.saveAllDetail:
         cmds.append( 'no p-tunnel star-star' )

   for cmd in cmds:
      mvpnSpmsiMode = mvpnMode[ MvpnIpv4VrfSpmsiConfigMode ].\
                     getOrCreateModeInstance( vrfName )
      mvpnSpmsiMode[ 'Mvpn.ipv4.spmsi' ].addCommand( cmd )

def saveMvpnVrfIpv4IntraAsIpmsiModeConfig( vrfName, vrfBgpConfig,
                                           mvpnMode, options ):
   cmds = []
   if vrfBgpConfig.afMvpnV4VrfIpmsiTunnelDisable:
      cmds.append( 'route attribute pmsi-tunnel no-advertise' )
   else:
      if options.saveAll or options.saveAllDetail:
         cmds.append( 'route attribute pmsi-tunnel advertise' )

   for cmd in cmds:
      mvpnIpmsiMode = mvpnMode[ MvpnIpv4VrfIntraAsIPmsiConfigMode ].\
                     getOrCreateModeInstance( vrfName )
      mvpnIpmsiMode[ 'Mvpn.ipv4.ipmsi' ].addCommand( cmd )


def saveMvpnVrfIpv4ModeConfig( vrfName, vrfBgpConfig, bgpVrfMode, options ):
   if vrfBgpConfig.afMvpnV4Vrf:
      mvpnMode = bgpVrfMode[ MvpnIpv4VrfConfigMode ].\
                  getOrCreateModeInstance( vrfName )
      saveMvpnVrfIpv4RsvpP2mpTunnelProfile( vrfName, vrfBgpConfig, mvpnMode,
            options )
      saveMvpnVrfIpv4SpmsiModeConfig( vrfName, vrfBgpConfig, mvpnMode,
      options )
      saveMvpnVrfIpv4IntraAsIpmsiModeConfig( vrfName, vrfBgpConfig, mvpnMode,
          options )
      saveMvpnVrfIpv4PmsiTunnelProtocol( vrfBgpConfig, mvpnMode )
   elif options.saveAll or options.saveAllDetail:
      bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( 'no mvpn-ipv4' )

@CliSave.saver( 'Routing::Bgp::VrfConfigDir', 'routing/bgp/vrf/config',
                requireMounts=( 'routing/bgp/config', 'routing/bgp/asn/config', ) )
def saveMvpnVrfIpv4Config( vrfConfigDir, root, requireMounts, options ):

   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )

   for vrfName in vrfConfigDir.vrfConfig:

      vrfBgpConfig = vrfConfigDir.vrfConfig[ vrfName ]
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode ]
      bgpVrfMode = bgpVrfMode.getOrCreateModeInstance( vrfName )

      if MvpnLibToggleLib.toggleMvpnSelectiveTunnelsEnabled():
         saveMvpnVrfIpv4ModeConfig( vrfName, vrfBgpConfig, bgpVrfMode, options )
      else:
         cmd = ''
         if vrfBgpConfig.afMvpnV4Vrf:
            cmd = 'mvpn-ipv4'
         elif options.saveAll or options.saveAllDetail:
            cmd = 'no mvpn-ipv4'

         if cmd:
            bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( cmd )
