# Copyright (c) 2007, 2008, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliMode.VrfConfig import NameServerGroupMode
from IpLibConsts import DEFAULT_VRF
import itertools

networkConfigCmdSeq = 'Network.config'
CliSave.GlobalConfigMode.addCommandSequence( networkConfigCmdSeq )

# -------------------------------------------------------------------------------
# Object used for saving commands in "ip nameserver-group" mode.
# -------------------------------------------------------------------------------
class NameServerGroupSaveMode( NameServerGroupMode, CliSave.Mode ):
   def __init__( self, groupName ):
      self.groupName_ = groupName
      NameServerGroupMode.__init__( self, groupName )
      CliSave.Mode.__init__( self, groupName )

CliSave.GlobalConfigMode.addChildMode( NameServerGroupSaveMode )
NameServerGroupSaveMode.addCommandSequence( 'Network.groupConfigDir' )

@CliSave.saver( 'Tac::Dir', 'sys/net/groupConfigDir' )
def saveConfigDir( netConfigDir, root, requireMounts, options ):
   saveAll = options.saveAll
   for groupName, groupConfig in netConfigDir.items():
      mode = root[ NameServerGroupSaveMode ].getOrCreateModeInstance( groupName )
      cmds = mode[ 'Network.groupConfigDir' ]

      # As/when more cmds are supported in this mode, make below code into
      # separate funcs so as to not replicate code at 2 places in saveConfigDir
      # and saveHostname
      allNameServers = itertools.chain( groupConfig.nameServer.values(),
                                        groupConfig.v6NameServer.values() )
      for nameServer in sorted( allNameServers ):
         ipAddrStr = nameServer.ipAddr.stringValue
         vrfString = "vrf %s " % nameServer.vrfName
         if nameServer.priority != 0:
            priorityStr = " priority %d" % nameServer.priority
         else:
            priorityStr = ""
         cmds.addCommand( "name-server %s%s%s" % ( vrfString, ipAddrStr,
                                                   priorityStr ) )
      if saveAll and ( len( groupConfig.nameServer ) == 0 ) and (
         len( groupConfig.v6NameServer ) == 0 ):
         # no name server configured
         cmds.addCommand( "no name-server" )

      cmd = 'dns domain'
      if groupConfig.domainName:
         cmds.addCommand( "%s %s" % ( cmd, groupConfig.domainName ) )
      elif saveAll:
         cmds.addCommand( "no %s" % cmd )

      for domain in groupConfig.domainList.values():
         cmds.addCommand( "ip domain-list %s" % domain )

@CliSave.saver( 'System::NetConfig', 'sys/net/config' )
def saveHostname( netConfig, root, requireMounts, options ):

   saveAll = options.saveAll

   cmds = root[ 'Network.config' ]
   if netConfig.hostname:
      cmds.addCommand( "hostname %s" % netConfig.hostname )
   elif saveAll:
      cmds.addCommand( "no hostname" )

   if len( netConfig.sourceIntf ) != 0:
      for ( vrfName, srcIntf ) in netConfig.sourceIntf.items():
         siVrf = ''
         if vrfName != DEFAULT_VRF:
            siVrf = 'vrf %s ' % vrfName
         cmds.addCommand( 'ip domain lookup %ssource-interface %s' %
                          ( siVrf, srcIntf ) )
   elif saveAll:
      cmds.addCommand( "no ip domain lookup source-interface" )

   allNameServers = itertools.chain( netConfig.nameServer.values(),
                                     netConfig.v6NameServer.values() )
   for nameServer in sorted( allNameServers ):
      ipAddrStr = nameServer.ipAddr.stringValue
      vrfString = "vrf %s " % nameServer.vrfName
      if nameServer.priority != 0:
         priorityStr = " priority %d" % nameServer.priority
      else:
         priorityStr = ""
      cmds.addCommand( "ip name-server %s%s%s" % ( vrfString, ipAddrStr,
                                                   priorityStr ) )

   if saveAll and ( len( netConfig.nameServer ) == 0 ) and (
      len( netConfig.v6NameServer ) == 0 ):
      # no name server configured
      cmds.addCommand( "no ip name-server" )

   if netConfig.dscpValue:
      cmds.addCommand( "dns qos dscp %d" % netConfig.dscpValue )
   elif saveAll:
      cmds.addCommand( "no dns qos dscp" )

   cmd = 'dns domain'
   if netConfig.domainName:
      cmds.addCommand( "%s %s" % ( cmd, netConfig.domainName ) )
   elif saveAll:
      cmds.addCommand( "no %s" % cmd )

   for domain in netConfig.domainList.values():
      cmds.addCommand( "ip domain-list %s" % domain )
   for hostname, hostEntry in netConfig.hostAddr.items():
      addressString = " ".join( hostEntry.ipAddr )
      if not addressString == "": # pylint: disable=unneeded-not
         cmds.addCommand( "ip host %s %s" % ( hostname, addressString ) )
      addressString = " ".join(
         [ ip6Address.stringValue for ip6Address in hostEntry.ip6Addr ] )
      if not addressString == "": # pylint: disable=unneeded-not
         cmds.addCommand( "ipv6 host %s %s" % ( hostname, addressString ) )
   if saveAll and ( len( netConfig.hostAddr ) == 0 ):
      # no mapping exists
      cmds.addCommand( "no ip host" )
      cmds.addCommand( "no ipv6 host" )

   if netConfig.externalDnsProxy:
      cmds.addCommand( 'ip domain proxy' )
   elif saveAll:
      cmds.addCommand( 'no ip domain proxy' )

   if netConfig.dnsCacheSize != netConfig.defaultDnsCacheSize:
      cmds.addCommand( 'dns cache size %d entries' % netConfig.dnsCacheSize )
   elif saveAll:
      cmds.addCommand( 'default dns cache size' )
