#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode.OcspProfile import OcspProfileMode
import CliSave
from CliSavePlugin.Security import SecurityConfigMode
import Tac
from Toggles import MgmtSecurityToggleLib

Constants = Tac.Type( "Mgmt::Security::Ssl::Constants" )
OcspCertRequirement = Tac.Type( "Mgmt::Security::Ssl::OcspCertRequirement" )
OcspNonceSetting = Tac.Type( "Mgmt::Security::Ssl::OcspNonceSetting" )

class OcspProfileConfigMode( OcspProfileMode, CliSave.Mode ):
   def __init__( self, param ):
      OcspProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

SecurityConfigMode.addChildMode( OcspProfileConfigMode,
                                 after=[ "Mgmt.security" ] )
OcspProfileConfigMode.addCommandSequence( "Mgmt.security.ocsp.profile" )

@CliSave.saver( "Mgmt::Security::Ssl::Config", "mgmt/security/ssl/config" )
def saveOcspProfileConfig( entity, root, requireMounts, options ):
   secMode = root[ SecurityConfigMode ].getSingletonInstance()
   for ocspProfileName, ocspProfile in entity.ocspProfileConfig.items():
      mode = secMode[ OcspProfileConfigMode ].getOrCreateModeInstance(
         ocspProfileName )
      cmds = mode[ "Mgmt.security.ocsp.profile" ]

      if ocspProfile.certRequirement != OcspCertRequirement.all:
         cmds.addCommand( "chain certificate requirement responder"
                          f" {ocspProfile.certRequirement}" )
      elif options.saveAll:
         cmds.addCommand( "chain certificate requirement responder all" )

      if ocspProfile.nonce != OcspNonceSetting.request:
         cmds.addCommand( f"extension nonce request {ocspProfile.nonce}" )
      elif options.saveAll:
         cmds.addCommand( "extension nonce request" )

      if ocspProfile.timeout != Constants.defaultOcspTimeout or options.saveAll:
         cmds.addCommand( f"timeout {ocspProfile.timeout}" )

      if ocspProfile.url:
         cmds.addCommand( f"url {ocspProfile.url}" )
      elif options.saveAll:
         cmds.addCommand( "no url" )

      if MgmtSecurityToggleLib.toggleOcspProfileVrfEnabled():
         if ocspProfile.vrfName != "default":
            cmds.addCommand( f"vrf {ocspProfile.vrfName}" )
         elif options.saveAll:
            cmds.addCommand( "default vrf" )
