# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliSave import escapeFormatString
from CliMode.OpenStack import (
      OpenStackConfigModeBase,
      RegionConfigModeBase,
   )
from CliSavePlugin.Controllerdb import ( CvxConfigMode,
                                         controllerConfigPath,
                                         getClusterName )
from CliSavePlugin.Security import mgmtSecurityConfigPath
from OpenStackLib import coalesceList
from IpLibConsts import DEFAULT_VRF
import Tac
import ReversibleSecretCli

class OpenStackConfigSaveMode( OpenStackConfigModeBase, CliSave.Mode ):
   def __init__( self, param ):
      OpenStackConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class OpenStackRegionConfigSaveMode( RegionConfigModeBase, CliSave.Mode ):
   def __init__( self, param ):
      RegionConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CvxConfigMode.addChildMode( OpenStackConfigSaveMode,
                            after=[ 'cvx' ] )
OpenStackConfigSaveMode.addCommandSequence(
   'openstack.config',
   before=[ OpenStackRegionConfigSaveMode ]
)

OpenStackConfigSaveMode.addChildMode( OpenStackRegionConfigSaveMode )

OpenStackRegionConfigSaveMode.addCommandSequence( 'openstack.vlanToVniConfig' )
OpenStackRegionConfigSaveMode.addCommandSequence( 'openstack.credentials' )
OpenStackRegionConfigSaveMode.addCommandSequence( 'openstack.vlanAllocation' )
OpenStackRegionConfigSaveMode.addCommandSequence( 'openstack.keystoneAuthUrl' )
OpenStackRegionConfigSaveMode.addCommandSequence( 'openstack.mandatoryRegion' )

def getOpenStackMode( root, cvxClusterName ):
   mode = root[ CvxConfigMode ].getOrCreateModeInstance(
      CvxConfigMode.modeName( cvxClusterName ) )
   return mode[ OpenStackConfigSaveMode ].getSingletonInstance( )

def addCommandsToOpenStackRegionMode( osMode, root, cmdSeq, regionName, cmds,
                                      cvxClusterName ):
   osRegionMode = osMode[ OpenStackRegionConfigSaveMode ].getOrCreateModeInstance(
      regionName )
   osRegionMode = osRegionMode[ cmdSeq ]

   for cmd in cmds:
      osRegionMode.addCommand( cmd )

# pylint: disable-msg=C0322 # pylint: disable=bad-option-value
@CliSave.saver( 'OpenStack::Config', 'mgmt/openstack/config',
                requireMounts=( controllerConfigPath, mgmtSecurityConfigPath ) )
def saveConfig( config, root, requireMounts, options ):
   cvxClusterName = getClusterName( requireMounts[ controllerConfigPath ] )
   if not cvxClusterName:
      return
   securityConfig = requireMounts[ mgmtSecurityConfigPath ]
   osMode = getOpenStackMode( root, cvxClusterName )
   osCmds = osMode[ 'openstack.config' ]

   if config.enabled or options.saveAll:
      osCmds.addCommand( "%sshutdown" % ( config.enabled and "no " or "" ) )

   if config.gracePeriod != config.defaultGracePeriod or options.saveAll:
      osCmds.addCommand( "grace-period %i" % config.gracePeriod )

   if config.vlanTypeDriver != config.defaultVlanTypeDriver or options.saveAll:
      typeDriverToCli = { 'defaultVlan' : 'default',
                          'aristaVlan' : 'arista' }
      osCmds.addCommand( "network type-driver vlan %s" %
                   typeDriverToCli.get( config.vlanTypeDriver, 'default' ) )

   if config.authRole != "":
      osCmds.addCommand( "authentication role %s" % config.authRole )

   for credentials in config.credentials.values():
      formatStr = ( f'username {escapeFormatString( credentials.username )} '
                    f'tenant {escapeFormatString( credentials.tenantName )} '
                    'password {}' )
      cmds = [ ReversibleSecretCli.getCliSaveCommand( formatStr,
                                                      securityConfig,
                                                      credentials.password ) ]
      addCommandsToOpenStackRegionMode( osMode, root, 'openstack.credentials',
                                        credentials.name, cmds,
                                        cvxClusterName )

   for region in config.vlanAssignment.values():
      vlanAssignment = list( region.vlan )
      vlanAssignment.sort()
      assignedVlanIds = coalesceList( vlanAssignment )
      cmds = [ 'resource-pool vlan %s' % ( assignedVlanIds ) ]
      addCommandsToOpenStackRegionMode( osMode, root, 'openstack.vlanAllocation',
                                        region.regionName, cmds, cvxClusterName )

   for region in config.keystoneAuthUrl.values():
      # port is necessary for backwards compatibility
      cmds = [ 'keystone auth-url %s' % region.authUrl ]
      addCommandsToOpenStackRegionMode( osMode, root, 'openstack.keystoneAuthUrl',
                                        region.name, cmds,
                                        cvxClusterName )

   for region in config.mandatoryRegion:
      addCommandsToOpenStackRegionMode( osMode, root, 'openstack.mandatoryRegion',
                                        region, [ 'provision sync mandatory' ],
                                        cvxClusterName )

@CliSave.saver( 'VirtualNetwork::Config', 'mgmt/virtualnetwork/config',
                requireMounts=( controllerConfigPath, ) )
def saveVirtualNetworkConfig( config, root, requireMounts, options ):
   cvxClusterName = getClusterName( requireMounts[ controllerConfigPath ] )
   if not cvxClusterName:
      return
   osMode = getOpenStackMode( root, cvxClusterName )

   for vlanMapVal in config.vlanToVniMap.values():
      cmds = []
      for vlanRange, vniRange in vlanMapVal.map.items():
         if vlanRange.start != vlanRange.end:
            cmd = 'networks map vlan %d - %d vni %d - %d' % (
                     vlanRange.start,
                     vlanRange.end,
                     vniRange.start,
                     vniRange.end )
         else:
            cmd = 'networks map vlan %d vni %d' % (
                     vlanRange.start,
                     vniRange.start )
         cmds.append( cmd )

      addCommandsToOpenStackRegionMode( osMode, root, 'openstack.vlanToVniConfig',
                                        vlanMapVal.domainName.domain, cmds,
                                        cvxClusterName )

@CliSave.saver( 'OpenStack::NameResolution::Config',
                'mgmt/openstack/nameResolutionConfig',
                requireMounts=( controllerConfigPath, ) )
def saveNameResolutionConfig( config, root, requireMounts, options ):
   cvxClusterName = getClusterName( requireMounts[ controllerConfigPath ] )
   if not cvxClusterName:
      return
   osMode = getOpenStackMode( root, cvxClusterName )
   osCmds = osMode[ 'openstack.config' ]

   if config.refreshPeriod != config.defaultRefreshPeriod or options.saveAll:
      refresh = int( config.refreshPeriod != Tac.endOfTime and config.refreshPeriod )
      osCmds.addCommand( 'name-resolution interval %s' % refresh )

@CliSave.saver( 'OpenStack::Config',
                'mgmt/openstack/config',
                requireMounts=( controllerConfigPath, ) )
def saveServiceAclConfig( config, root, requireMounts, options ):
   cvxClusterName = getClusterName( requireMounts[ controllerConfigPath ] )
   if not cvxClusterName:
      return
   osMode = getOpenStackMode( root, cvxClusterName )
   osCmds = osMode[ 'openstack.config' ]
   # Only default vrf supported
   if config.serviceAclTypeVrfMap:
      for aclType in [ 'ip', 'ipv6' ]:
         key = Tac.Value( "Acl::AclTypeAndVrfName", aclType, DEFAULT_VRF )
         aclName = config.serviceAclTypeVrfMap.aclName.get( key )
         if aclName:
            osCmds.addCommand( f'{aclType} access-group {aclName}' )
