#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from __future__ import absolute_import, division, print_function
import AuthnUserPriorityCli as aupCli
import Tac
import Arnet
import CliSave
import MultiRangeRule
import Intf.IntfRange
from IpLibConsts import DEFAULT_VRF
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliMode.P4RuntimeMode import P4RuntimeMode, P4RuntimeTransportMode
from Toggles.P4RuntimeConfigToggleLib import (
   toggleP4RuntimeAccountingRequestsEnabled,
   toggleP4RuntimeAUPEnabled,
   toggleP4RuntimeDscpEnabled )

EthIntfIdType = Tac.Type( "Arnet::EthIntfId" )
ChassisId = Tac.Type( 'P4Runtime::ChassisId' )

class P4RuntimeConfigMode( P4RuntimeMode, CliSave.Mode ):
   def __init__( self, param ):
      P4RuntimeMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( P4RuntimeConfigMode,
                                       after=[ IntfConfigMode ] )
P4RuntimeConfigMode.addCommandSequence( 'P4Runtime.global' )
P4RuntimeConfigMode.addCommandSequence( 'P4Runtime.deviceConfig' )
P4RuntimeConfigMode.addCommandSequence( 'P4Runtime.portMapping' )

class P4RuntimeTransportSaveMode( P4RuntimeTransportMode, CliSave.Mode ):
   def __init__( self, param ):
      P4RuntimeTransportMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

P4RuntimeConfigMode.addChildMode( P4RuntimeTransportSaveMode )
P4RuntimeTransportSaveMode.addCommandSequence( 'P4Runtime.transport' )

def portMappingCfgCmd( p4PortIdList, intfIdList ):
   if not p4PortIdList:
      return ''
   p4PortIdRangeStr = ','.join(
      MultiRangeRule.multiRangeToCanonicalString( p4PortIdList ).split( "," ) )
   intfNameList = [ intfId.stringValue for intfId in intfIdList ]
   intfNamesStr = ','.join( Intf.IntfRange.intfListToCanonical( intfNameList ) )
   intfNamesStr = intfNamesStr.replace( 'Po', 'Port-Channel' )
   intfNamesStr = intfNamesStr.replace( 'Et', 'Ethernet' )
   intfNamesStr = intfNamesStr.replace( 'Sw', 'Switch' )
   cfgCmd = "port id %s interface %s" % ( p4PortIdRangeStr, intfNamesStr )
   return cfgCmd

def groupPortIntfIdPairsByParts( cfg ):
   '''
   returns a dictionary indexed by "parts" in the interface, where a part is the
   numerical indices in interface name. e.g. Et1 has 1 part, Et2/1 has 2 parts etc.
   This is so that we can generate configuration by sorting interfaces with same
   number of parts
   Index 0 is used for "cpu"
   Index 99 is for LAGs so that they are printed/saved at the end
   <parts> : [ ( p4PortId1, intfId1 ), (p4PortId2, intfId2), ... ]
   '''
   portIntfIdByParts = {}
   for p4PortId in sorted( cfg.p4PortIdToIntfId.keys() ):
      intfId = Arnet.IntfId( cfg.p4PortIdToIntfId[ p4PortId ] )
      if intfId == Arnet.IntfId( 'Cpu' ):
         index = -1
      elif intfId.stringValue.startswith( 'Po' ):
         # We want to save the LAGs at the end
         index = 99
      else:
         index = EthIntfIdType.parts( intfId )
      onePortIntfIdGroup = portIntfIdByParts.setdefault( index, list() )
      onePortIntfIdGroup.append( ( p4PortId, intfId ) )
   return portIntfIdByParts

def groupContiguousPortIntfId( portIntfIdByParts ):
   '''
   returns a groups of ports that have sorted interfaces so that canonical
   form may be generated in the CliSave configuration
   '''
   portIntfIdGroups = {}
   for index in sorted( portIntfIdByParts.keys() ):
      prevIntfId = None
      for portIntfIdMapping in portIntfIdByParts[ index ]:
         p4PortId = portIntfIdMapping[ 0 ]
         intfId = portIntfIdMapping[ 1 ]
         if not prevIntfId or intfId < prevIntfId:
            intfGroupingIndex = len( portIntfIdGroups )
         else:
            intfGroupingIndex = len( portIntfIdGroups ) - 1
         prevIntfId = intfId
         portIntfIdMapping = portIntfIdGroups.setdefault( intfGroupingIndex,
                                                          tuple( ( [], [] ) ) )
         portIntfIdMapping[ 0 ].append( p4PortId )
         portIntfIdMapping[ 1 ].append( intfId )
   return portIntfIdGroups

@CliSave.saver( 'P4RuntimeConfig::PortMappingConfig',
                'p4runtime/config/port-mapping/cli',
                requireMounts=( 'p4runtime/status/hardware/status', ) )
def saveP4RuntimePortMapping( cfg, root, requireMounts, options ):
   # Display config only if P4Runtime is supported by the hardware
   p4RtHwStatus = requireMounts[ 'p4runtime/status/hardware/status' ]
   if not p4RtHwStatus.p4RuntimeSupported:
      return
   # Create a tuple consisting of ( [ p4PortIds ], [ intfNames ] ) such that the
   # intfNames are in sorted order so that the canonical string representation is
   # correct. For example,
   #   1 -> Po2, 2 -> Po1, 7 -> Po3
   #   3 -> Et2, 4 -> Et1, 5 -> Et3
   # the cofnfiguration would look like
   #   port id 3 interface Ethernet 2
   #   port id 4-5 interface Ethernet 1,3
   #   port id 1 interface Port-channel 2
   #   port id 2,7 interface Port-channel 1,3
   #
   # We do this by grouping the interfaces by "parts" since sort across different
   # interface types and parts does not work
   #   i.e. Ethernet<n> (1 part) vs Ethernetx/y/z (3 parts)
   # Then, for each group of ports in each part, we look for contiguous set of
   # interfaces that we can "rangify" using a canonical/range-rule
   mode = root[ P4RuntimeConfigMode ].getSingletonInstance()
   cmds = mode[ 'P4Runtime.portMapping' ]
   portIntfIdByParts = groupPortIntfIdPairsByParts( cfg )
   portIntfIdGroups = groupContiguousPortIntfId( portIntfIdByParts )
   for portIntfIdMapping in portIntfIdGroups.values():
      cmd = portMappingCfgCmd( portIntfIdMapping[ 0 ], portIntfIdMapping[ 1 ] )
      cmds.addCommand( cmd )

@CliSave.saver( 'P4RuntimeConfig::P4RtDeviceConfig',
                'p4runtime/config/deviceConfig',
                requireMounts=( 'p4runtime/status/hardware/status', ) )
def saveP4RuntimeDeviceConfig( cfg, root, requireMounts, options ):
   p4RtHwStatus = requireMounts[ 'p4runtime/status/hardware/status' ]
   if not p4RtHwStatus.p4RuntimeSupported:
      return
   if not cfg.fapNameToDeviceId:
      return
   mode = root[ P4RuntimeConfigMode ].getOrCreateModeInstance( None )
   cmds = mode[ 'P4Runtime.deviceConfig' ]
   for fapName, deviceId in cfg.fapNameToDeviceId.items():
      cmds.addCommand( 'forwarding chip {} device id {}'.format(
         fapName, deviceId ) )

@CliSave.saver( 'P4RuntimeConfig::P4RtSystemConfig', 'p4runtime/config/system',
                requireMounts=( 'p4runtime/status/hardware/status', ) )
def saveP4RuntimeConfig( cfg, root, requireMounts, options ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   # Display config only if P4Runtime is supported by the hardware
   p4RtHwStatus = requireMounts[ 'p4runtime/status/hardware/status' ]
   if not p4RtHwStatus.p4RuntimeSupported:
      saveAll = False
      saveAllDetail = False
   saveDefault = saveAllDetail or saveAll

   mode = root[ P4RuntimeConfigMode ].getSingletonInstance()
   cmds = mode[ 'P4Runtime.global' ]
   if cfg.enabled:
      cmds.addCommand( "no shutdown" )
   elif saveDefault:
      cmds.addCommand( "shutdown" )

   if cfg.chassisId != ChassisId.null:
      cmds.addCommand( "chassis id {}".format( cfg.chassisId ) )
   elif saveDefault:
      cmds.addCommand( "no chassis id" )
   if cfg.transportName:
      transportMode = mode[ P4RuntimeTransportSaveMode ].getOrCreateModeInstance(
                        cfg.transportName )
      transportCmds = transportMode[ 'P4Runtime.transport' ]
      p4RtSystemConfig = Tac.Value( "P4RuntimeConfig::ServerParams" )
      if cfg.grpcServerPort != p4RtSystemConfig.defaultPiServerPort or saveDefault:
         transportCmds.addCommand( 'port {}'.format( cfg.grpcServerPort ) )
      if cfg.grpcServerVrfName != DEFAULT_VRF or saveDefault:
         transportCmds.addCommand( 'vrf {}'.format( cfg.grpcServerVrfName ) )

      if cfg.sslProfileName:
         transportCmds.addCommand( 'ssl profile {}'.format( cfg.sslProfileName ) )
      elif saveDefault:
         transportCmds.addCommand( 'no ssl profile' )

      if toggleP4RuntimeAUPEnabled():
         if ( cfg.authnUsernamePriority.values() !=
               aupCli.defaultAuthnUserPriority ) or options.saveAll:
            tokens = [ aupCli.authnUsernamePriorityToCLI[ v.source ]
                  for v in cfg.authnUsernamePriority.values() ]
            transportCmds.addCommand(
                  f"authentication username priority {' '.join( tokens )}" )
      else:
         if cfg.certUsernameAuthn:
            transportCmds.addCommand( "certificate username authentication" )
         elif saveDefault:
            transportCmds.addCommand( "no certificate username authentication" )

      if toggleP4RuntimeAccountingRequestsEnabled():
         if cfg.accountingRequests:
            transportCmds.addCommand( "accounting requests" )
         elif saveDefault:
            transportCmds.addCommand( "no accounting requests" )

      if toggleP4RuntimeDscpEnabled():
         if cfg.dscp:
            transportCmds.addCommand( 'qos dscp {}'.format( cfg.dscp ) )
         elif saveDefault:
            transportCmds.addCommand( 'no qos dscp' )

