# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from PhyFeatureCliLib import getPhyFeatureStatusNested
from PhyFecHistogramLib import getTh4FecHistoFeatureStatus
import Tac
from Toggles.PhyEeeToggleLib import (
   toggleParallelDetectionCliEnabled,
   togglePhyLinkProfileSupportEnabled,
   togglePhyPrecodingCmdEnabled,
   toggleStrataFecHistogramsEnabled
)
from TypeFuture import TacLazyType

linkDetectionRf = TacLazyType( "Hardware::Phy::LinkDetectionRF" )
LineRateEnum = TacLazyType( "Phy::Coherent::LineRate" )
baseTPairPolarity = TacLazyType( "Hardware::Phy::BaseTPairPolarity" )

IntfConfigMode.addCommandSequence( 'PhyEee.intf' )

# Helper function to return the CLI command string based on enum value
def modulationEnumToCmd( modulation ):
   modulationEnum = Tac.Type( "Phy::Coherent::Modulation" )
   return { modulationEnum.modulation16Qam : '16qam',
            modulationEnum.modulation8Qam : '8qam',
            modulationEnum.modulationDpQpsk : 'dp-qpsk' }.get( modulation )

def lineRateEnumToCmd( lineRate ):
   return { LineRateEnum.lineRate100g : '100g',
            LineRateEnum.lineRate200g : '200g',
            LineRateEnum.lineRate300g : '300g',
            LineRateEnum.lineRate400g : '400g' }.get( lineRate )

def coherentFecEnumToCmd( encoding ):
   fecEnum = Tac.Type( "Phy::Coherent::CoherentFecEncodingConfig" )
   return { fecEnum.coherentFecEncodingCfgSd15 :'soft-decision overhead 15 percent',
         fecEnum.coherentFecEncodingCfgSd25 : 'soft-decision overhead 25 percent',
         fecEnum.coherentFecEncodingCfgSd25Bch : \
               'soft-decision overhead 25 percent bch',
         fecEnum.coherentFecEncodingCfgSd20 : 'soft-decision overhead 20 percent',
         fecEnum.coherentFecEncodingCfgHd7 : 'staircase overhead 7 percent',
         fecEnum.coherentFecEncodingCfgG709 : 'reed-solomon g709',
         fecEnum.coherentFecEncodingCfgCfec : 'concatenated',
         fecEnum.coherentFecEncodingCfgOfec : 'open' }.get( encoding )

def polarityEnumToCmd( polarity ):
   return { baseTPairPolarity.baseTPairPolarityStandard : 'standard',
            baseTPairPolarity.baseTPairPolarityReversed : 'reversed'
          }[ polarity ]

@CliSave.saver( 'Hardware::Phy::PhyCliConfigDir', 'hardware/phy/config/cli' )
def savePhyConfig( phyCliConfigDir, root, requireMounts, options ):

   for entity in phyCliConfigDir.phyCliConfig.values():
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.name )
      cmds = mode[ 'PhyEee.intf' ]
   
      # NOTE: we ignore "options.saveAll" and "options.saveAllDetail"
      #       here, because the "tx-preemphasis" and "phy ll-tx-path"
      #       commands are hidden, so we avoid exposing them via "show
      #       running-config all [detail]".
      tpo = entity.txPreemphOverride
      if tpo.overrideMethod == 'absolute':
         cmds.addCommand( 'tx-preemphasis %f' % tpo.attn )
      elif tpo.overrideMethod == 'internalTraceOnly':
         cmds.addCommand( 'tx-preemphasis standard' )
   
      if entity.llTxPath != entity.defaultLlTxPath:
         ld = { 'five' : '5',
                'ten' : '10',
                'eleven' : '11' }
         cmds.addCommand( 'phy ll-tx-path %s' % ld[ entity.llTxPath ] )
   
      if entity.tn8044BFwVersion == 'tn8044BFwVer111':
         cmds.addCommand( 'phy 1g-autoneg resilient' )
   
      if entity.stabilizationDelay != entity.defaultStabilizationDelay:
         cmds.addCommand( 'hardware phy rx-signal stabilization-delay %d' %
                          entity.stabilizationDelay )
      elif options.saveAll:
         cmds.addCommand( 'no hardware phy rx-signal stabilization-delay' )

      testPattern = Tac.Type( "Hardware::Phy::TestPattern" )
      # At present, the CLI command for a particular test pattern is
      # upper case of the supported test pattern enum. If this behaviour 
      # changes add a new mapping function for the same.
      if entity.testPatternTx != testPattern.patternNone:
         cmds.addCommand( "phy diag transmitter test pattern %s" %
                          entity.testPatternTx.upper() )
      elif options.saveAll:
         cmds.addCommand( "no phy diag transmitter test pattern" )

      if entity.testPatternRx != testPattern.patternNone:
         cmds.addCommand( "phy diag receiver test pattern %s" %
                          entity.testPatternRx.upper() )
      elif options.saveAll:
         cmds.addCommand( "no phy diag receiver test pattern" )
      
      precoding = Tac.Type( "Hardware::Phy::Precoding" ) 
      if togglePhyPrecodingCmdEnabled():
         if entity.precodingTx == precoding.precodingDisabled:
            cmds.addCommand( "phy transmitter precoding disabled" )
         elif entity.precodingTx == precoding.precodingEnabled:
            cmds.addCommand( "phy transmitter precoding" )
         elif options.saveAll: 
            cmds.addCommand( "default phy transmitter precoding" )

         if entity.precodingRx == precoding.precodingDisabled:
            cmds.addCommand( "phy receiver precoding disabled" )
         elif entity.precodingRx == precoding.precodingEnabled:
            cmds.addCommand( "phy receiver precoding" )
         elif options.saveAll:
            cmds.addCommand( "default phy receiver precoding" )

   for entity in phyCliConfigDir.phyCliCoherentConfig.values():
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.name )
      cmds = mode[ 'PhyEee.intf' ]
      modulationEnum = Tac.Type( "Phy::Coherent::Modulation" )
      if entity.modulation != modulationEnum.modulationDefault:
         # pylint: disable-next=consider-using-in
         if ( entity.lineRate != LineRateEnum.lineRateNone and
              entity.lineRate != LineRateEnum.lineRateDefault ):
            cmds.addCommand( "phy coherent-dsp modulation %s rate %s" %
                             ( modulationEnumToCmd( entity.modulation ),
                               lineRateEnumToCmd( entity.lineRate ) ) )
         else:
            cmds.addCommand( "phy coherent-dsp modulation %s" %
                             modulationEnumToCmd( entity.modulation ) )
      elif options.saveAll:
         cmds.addCommand( "default phy coherent-dsp modulation" )

      if entity.rollOffConfigured:
         cmds.addCommand( "phy coherent-dsp transmitter roll-off %f" %
                          entity.rollOff )
      elif options.saveAll:
         cmds.addCommand( "default phy coherent-dsp transmitter roll-off" )
      fecEnum = Tac.Type( "Phy::Coherent::CoherentFecEncodingConfig" )
      if entity.fecConfig != fecEnum.coherentFecEncodingCfgDefault:
         cmds.addCommand( "error-correction encoding %s" %
                           coherentFecEnumToCmd( entity.fecConfig ) )
      if entity.sopFastTracking:
         cmds.addCommand( "phy coherent-dsp receiver state-of-polarization "
                          "tracking fast" )
      elif options.saveAll:
         cmds.addCommand( "default phy coherent-dsp receiver state-of-polarization "
                          "tracking fast" )

# This is specific cli save for the internal phy of TH4.
# When some phy at some arbitrary position requires FEC histogram config they
# will need to pull the phy model list and determine the correct phy number.
def saveTh4FecHistogramConfig( cmds, options, entity, phyFeatureStatusSliceDir ):
   cmdBase = 'phy diag error-correction histogram phy 0 bins'
   cmdFmt = cmdBase + ' %s'
   noCmd = 'no ' + cmdBase
   # we need the capabilities to emit correct commands
   # emit nothing if it does not exist.
   if not phyFeatureStatusSliceDir:
      return
   caps = getTh4FecHistoFeatureStatus( phyFeatureStatusSliceDir, entity.intfId )
   if caps is None:
      return

   if entity.fecHistogramBinSelect is not None:
      if entity.fecHistogramBinSelect != caps.defaultRange:
         cmds.addCommand( cmdFmt % entity.fecHistogramBinSelect.stringValue )
      elif options.saveAll:
         cmds.addCommand( noCmd )
   elif options.saveAll:
      # If config is not set, the only other time we emit something is when
      # saveAll is true.
      # In order to emit the default with saveAll, we must know the default
      # value via status
      cmds.addCommand( noCmd )

def baseTRxPreDistortionSupported( phyFeatureStatusSliceDir, intfId ):
   phyFeatureStatus = getPhyFeatureStatusNested( phyFeatureStatusSliceDir, intfId )
   if not phyFeatureStatus:
      return False
   else:
      return phyFeatureStatus.baseTRxPreDistortionSupported

@CliSave.saver( 'Tac::Dir', 'hardware/archer/phy/config/cli/feature/slice',
                requireMounts=( 'hardware/archer/phy/status/feature/slice', ) )
def savePhyFeatureConfig( sliceDir, root, requireMounts, options ):
   for phyFeatureConfigDir in sliceDir.values():
      for entity in phyFeatureConfigDir.config.values():
         mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
         cmds = mode[ 'PhyEee.intf' ]
         if toggleStrataFecHistogramsEnabled():
            saveTh4FecHistogramConfig(
               cmds, options, entity,
               requireMounts[ 'hardware/archer/phy/status/feature/slice' ] )
         for laneNum, profileName in entity.phyTxEqProfilesConfig.items():
            cmds.addCommand( 'phy transmitter equalization lane %d profile %s' %
                           ( laneNum - 1, profileName ) )
         if entity.txClockShift != entity.defaultTxClockShift:
            cmds.addCommand( 'phy transmitter clock shift %d' %
                             entity.txClockShift )
         elif options.saveAll:
            cmds.addCommand( 'no phy transmitter clock shift' )
         if entity.linkDetection != entity.defaultLinkDetection:
            cmds.addCommand( 'phy link detection aggressive' )
         elif options.saveAll:
            cmds.addCommand( 'no phy link detection aggressive' )
         if togglePhyLinkProfileSupportEnabled():
            if entity.phyLinkProfile != entity.defaultPhyLinkProfile:
               cmds.addCommand( 'phy link profile protection-switched' )
            elif options.saveAll:
               cmds.addCommand( 'no phy link profile protection-switched' )
         if entity.linkDetectionRF != linkDetectionRf.linkDetectionRfPassive:
            cmds.addCommand( 'phy link transmitter recovery' )
         elif options.saveAll:
            cmds.addCommand( 'no phy link transmitter recovery' )
         if entity.standaloneLinkTraining:
            cmds.addCommand( 'phy link training' )
         elif options.saveAll:
            cmds.addCommand( 'no phy link training' )
         if ( entity.linkStabilizationRfGeneration !=
                entity.defaultLinkStabilizationRfGeneration ):
            cmds.addCommand( 'phy link stabilization remote-fault disabled' )
         elif options.saveAll:
            cmds.addCommand( 'no phy link stabilization remote-fault disabled' )
         if entity.loopTimingPortType != entity.defaultLoopTimingPortType:
            portTypeEnum = Tac.Type( "Hardware::Phy::LoopTimingPortType" )
            typeStr = ''
            if entity.loopTimingPortType == portTypeEnum.advertiseSinglePort:
               typeStr = 'single'
            elif entity.loopTimingPortType == portTypeEnum.advertiseMultiPort:
               typeStr = 'multi'
            else:
               assert False, "Unexpected loop timing port type"
            cmds.addCommand( 'phy media base-t negotiation port type %s' % typeStr )
         elif options.saveAll:
            cmds.addCommand( 'no phy media base-t negotiation port type' )
         if entity.loopTimingPhyRole != entity.defaultLoopTimingPhyRole:
            PhyRoleEnum = Tac.Type( "Hardware::Phy::LoopTimingPhyRole" )
            typeStr = ''
            if entity.loopTimingPhyRole == PhyRoleEnum.loopTimingRoleSecondary:
               typeStr = 'secondary'
            elif entity.loopTimingPhyRole == PhyRoleEnum.loopTimingRolePrimary:
               typeStr = 'primary'
            elif entity.loopTimingPhyRole == PhyRoleEnum.loopTimingRoleAuto:
               typeStr = 'auto'
            elif entity.loopTimingPhyRole == PhyRoleEnum.loopTimingRoleNonSpecAuto:
               typeStr = 'auto non-spec'
            else:
               assert False, "Unexpected loop timing phy role"
            cmds.addCommand( 'phy media base-t negotiation role %s' % typeStr )
         elif options.saveAll:
            cmds.addCommand( 'no phy media base-t negotiation role' )
         if entity.baseTRxPreDistortion != entity.defaultBaseTRxPreDistortion:
            RxPreDistortionEnum = Tac.Type(
                  'Hardware::Phy::BaseTRxPreDistortionConfig' )
            valueStr = ''
            config = entity.baseTRxPreDistortion
            if config == RxPreDistortionEnum.baseTRxPreDistortionFilterB:
               valueStr = 'b'
            elif config == RxPreDistortionEnum.baseTRxPreDistortionFilterDisabled:
               valueStr = 'disabled'
            else:
               assert False, 'Unexpected BASE-T Rx pre-distortion config'
            cmd = 'phy media base-t receiver filter pre-distortion %s' % valueStr
            cmds.addCommand( cmd )
         elif options.saveAll:
            # If this feature is supported, we print out filter A in the output.
            # We model the default/existing filter option as the filter A. See
            # notes on BaseTRxPreDistortionConfig enum for details.
            if baseTRxPreDistortionSupported(
                  requireMounts[ 'hardware/archer/phy/status/feature/slice' ],
                  entity.intfId ):
               cmds.addCommand(
                     'phy media base-t receiver filter pre-distortion a' )
            else:
               cmds.addCommand(
                     'no phy media base-t receiver filter pre-distortion' )
         if entity.baseTTxGain != entity.defaultBaseTTxGain:
            BaseTTxGainEnum = Tac.Type( 'Hardware::Phy::BaseTTxGainConfig' )
            valueStr = ''
            if entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainPlus1Percent:
               valueStr = '+1'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainPlus2Percent:
               valueStr = '+2'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainPlus3Percent:
               valueStr = '+3'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainPlus4Percent:
               valueStr = '+4'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainPlus5Percent:
               valueStr = '+5'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainPlus6Percent:
               valueStr = '+6'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainPlus7Percent:
               valueStr = '+7'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainPlus8Percent:
               valueStr = '+8'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainPlus9Percent:
               valueStr = '+9'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainPlus25Percent:
               valueStr = '+25'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainMinus10Percent:
               valueStr = '-10'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainMinus20Percent:
               valueStr = '-20'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainMinus30Percent:
               valueStr = '-30'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainMinus35Percent:
               valueStr = '-35'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainMinus45Percent:
               valueStr = '-45'
            elif entity.baseTTxGain == BaseTTxGainEnum.baseTTxGainMinus50Percent:
               valueStr = '-50'
            else:
               assert False, 'Unexpected BASE-T Tx gain config'
            cmds.addCommand(
                  'phy media base-t transmitter gain %s percent' % valueStr )
         elif options.saveAll:
            cmds.addCommand( 'no phy media base-t transmitter gain' )
         if entity.fastRetrain != entity.defaultFastRetrain:
            cmds.addCommand(
                  'phy media base-t negotiation fast-retrain disabled' )
         elif options.saveAll:
            cmds.addCommand(
                  'no phy media base-t negotiation fast-retrain disabled' )
         if  entity.speedDownshifting != entity.defaultSpeedDownshifting:
            cmds.addCommand( 'phy media base-t negotiation downshifting disabled' )
         elif options.saveAll:
            cmds.addCommand(
                  'no phy media base-t negotiation downshifting disabled' )
         if ( entity.serdesMapping !=
               Tac.Type( "Hardware::Phy::SerdesMapping" ).serdesMappingDefault ):
            cmds.addCommand( 'phy link serdes mapping direct' )
         elif options.saveAll:
            cmds.addCommand( 'no phy link serdes mapping direct' )
         if entity.baseTPolarityPairA != baseTPairPolarity.baseTPairPolarityDefault:
            cmds.addCommand( 'phy media base-t polarity pair a %s' %
                             polarityEnumToCmd( entity.baseTPolarityPairA ) )
         elif options.saveAll:
            cmds.addCommand( 'no phy media base-t polarity pair a' )
         if entity.baseTPolarityPairB != baseTPairPolarity.baseTPairPolarityDefault:
            cmds.addCommand( 'phy media base-t polarity pair b %s' %
                             polarityEnumToCmd( entity.baseTPolarityPairB ) )
         elif options.saveAll:
            cmds.addCommand( 'no phy media base-t polarity pair b' )
         if entity.baseTPolarityPairC != baseTPairPolarity.baseTPairPolarityDefault:
            cmds.addCommand( 'phy media base-t polarity pair c %s' %
                             polarityEnumToCmd( entity.baseTPolarityPairC ) )
         elif options.saveAll:
            cmds.addCommand( 'no phy media base-t polarity pair c' )
         if entity.baseTPolarityPairD != baseTPairPolarity.baseTPairPolarityDefault:
            cmds.addCommand( 'phy media base-t polarity pair d %s' %
                             polarityEnumToCmd( entity.baseTPolarityPairD ) )
         elif options.saveAll:
            cmds.addCommand( 'no phy media base-t polarity pair d' )
         if entity.mdiCrossoverMode != entity.defaultMdiCrossoverMode:
            mdiModeEnum = TacLazyType( "Hardware::BaseT::MdiCrossoverMode" )
            typeStr = ''
            # Since we treat `no|default` and `... auto` as the same, as
            # we agreed with cli-review, this line will never show up in
            # the running-config. Including it here for completeness' sake.
            if entity.mdiCrossoverMode == mdiModeEnum.mdiAuto:
               typeStr = 'auto'
            elif entity.mdiCrossoverMode == mdiModeEnum.mdiStraight:
               typeStr = 'mdi'
            elif entity.mdiCrossoverMode == mdiModeEnum.mdiCrossover:
               typeStr = 'mdi-x'
            else:
               assert False, "Unexpected mdi crossover mode"
            cmds.addCommand( 'phy media base-t pair layout %s' % typeStr )
         elif options.saveAll:
            cmds.addCommand( 'phy media base-t pair layout auto' )
         if toggleParallelDetectionCliEnabled():
            if entity.parallelDetection != entity.defaultParallelDetection:
               cmds.addCommand(
                     'phy media base-x parallel-detection disabled' )
            elif options.saveAll:
               cmds.addCommand(
                     'no phy media base-x parallel-detection disabled' )
         cdrModeEnum = TacLazyType( "Hardware::Phy::CdrMode" )
         if entity.cdrMode != cdrModeEnum.defaultCdr:
            typeStr = ''
            if entity.cdrMode == cdrModeEnum.baudRateCdr:
               typeStr = 'baud-rate'
            elif entity.cdrMode == cdrModeEnum.oversamplingCdr:
               typeStr = 'oversampling'
            elif entity.cdrMode == cdrModeEnum.autoCdr:
               typeStr = 'auto'
            else:
               assert False, "Unexpected CDR mode"
            cmds.addCommand( 'phy receiver cdr mode %s' % typeStr )
         elif options.saveAll:
            cmds.addCommand( 'no phy receiver cdr mode' )
