# Copyright (c) 2009-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliSavePlugin import IntfCliSave
import Tac
import EthIntfUtil
import Tracing
from Toggles.MacMonToggleLib import (
   togglePortSecProtectStaticEnabled, togglePortSecStickyMACEnabled )
t0 = Tracing.trace0

VlanId = Tac.Type( "Bridging::VlanId" )
CliSave.GlobalConfigMode.addCommandSequence( 'PortSec.global' )
IntfCliSave.IntfConfigMode.addCommandSequence( 'PortSec.switchport',
                                               after=[ 'Ebra.switchport' ] )

@CliSave.saver( 'PortSec::Config', 'portsec/config' )
def savePortSecConfig( portSecConfig, root, requireMounts, options ):
   savePortSecGlobalConfig( portSecConfig, root, options )
   # Get the list of interfaces on which config needs to be displayed.
   if options.saveAllDetail:
      # We allow port-sec on L2 sub-interfaces (bridging ports on Sand),
      # hence include all interface names not just switchports
      cfgIntfNames = EthIntfUtil.allSwitchportNames( requireMounts,
                                                     includeEligible=True,
                                                     includeSubIntf=True )
      # all switchports, routed ports and bridged ports in intfConfig.
   elif options.saveAll:
      # We allow L2 configuration on routed ports. Set of intfs for config are
      switchportNames = EthIntfUtil.allSwitchportNames( requireMounts )
      cfgIntfNames = set( portSecConfig.intfConfig )
      cfgIntfNames.update( switchportNames )
   else:
      cfgIntfNames = portSecConfig.intfConfig

   for intfName in sorted( cfgIntfNames ):
      intfConfig = portSecConfig.intfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            # just creating a temporary instance. Mode (shutdown) is unimportant.
            intfConfig = Tac.Value( 'PortSec::IntfConfig', intfName )
         else:
            continue
      savePortSecIntfConfig( portSecConfig, intfConfig, root, options )

   for vlanConfig in portSecConfig.vlanConfig.values():
      savePortSecVlanConfig( portSecConfig, vlanConfig, root )
   for defaultVlanConfig in portSecConfig.defaultVlanConfig.values():
      savePortSecVlanConfig( portSecConfig, defaultVlanConfig, root )

def savePortSecIntfConfig( portSecConfig, portSecIntfConfig, root, options ):
   ifConfigMode = root[ IntfCliSave.IntfConfigMode ]
   mode = ifConfigMode.getOrCreateModeInstance( portSecIntfConfig.intfId )
   cmds = mode[ 'PortSec.switchport' ]
   if EthIntfUtil.isSubIntf( portSecIntfConfig.intfId ):
      # mac address limit Cli Save
      if portSecIntfConfig.mode == 'protect':
         cmds.addCommand( "mac address limit violation protect" )
      elif options.saveAll or options.saveAllDetail:
         cmds.addCommand( "mac address limit violation shutdown" )
      if portSecIntfConfig.enabled:
         cmd = 'mac address limit'
         if portSecIntfConfig.maxAddrs != 1:
            cmd += " %d" % portSecIntfConfig.maxAddrs
         cmds.addCommand( cmd )
      elif options.saveAll or options.saveAllDetail:
         cmds.addCommand( "no mac address limit" )
   else:
      if portSecIntfConfig.enabled:
         if portSecIntfConfig.mode == 'shutdown':
            cmds.addCommand( "switchport port-security" )
         else:
            if portSecIntfConfig.log:
               cmds.addCommand( "switchport port-security violation protect log" )
            else:
               cmds.addCommand( "switchport port-security violation protect" )
      elif options.saveAll:
         cmds.addCommand( "no switchport port-security" )

      if portSecIntfConfig.vlanBased:
         cmds.addCommand( "switchport port-security mac-address maximum disabled" )
      elif ( togglePortSecProtectStaticEnabled() and
            portSecIntfConfig.staticOnly ):
         cmds.addCommand( "switchport port-security mac-address maximum static" )
      elif portSecIntfConfig.maxAddrs != 1 or options.saveAll:
         # moving this above enabling port-security because it's really better
         # to set max addrs first to prevent a bunch of code to reevaluate the port
         cmd = "switchport port-security mac-address maximum %d"
         cmds.addCommand( cmd % portSecIntfConfig.maxAddrs )

def savePortSecVlanConfig( portSecConfig, portSecVlanConfig, root ):
   intfId = portSecVlanConfig.intfVlanKey.intfId
   ifConfigMode = root[ IntfCliSave.IntfConfigMode ]
   mode = ifConfigMode.getOrCreateModeInstance( intfId )
   cmds = mode[ 'PortSec.switchport' ]
   if not EthIntfUtil.isSubIntf( intfId ):
      vlanId = portSecVlanConfig.vlanId
      maxAddrs = portSecVlanConfig.maxAddrs
      cmd = "switchport port-security vlan %d mac-address maximum %d"
      defaultCmd = "switchport port-security vlan default mac-address maximum %d"
      if vlanId == VlanId.invalid:
         cmds.addCommand( defaultCmd % maxAddrs )
      else:
         cmds.addCommand( cmd % ( vlanId, maxAddrs ) )

def savePortSecGlobalConfig( portSecConfig, root, options ):
   cmds = root[ 'PortSec.global' ]

   if portSecConfig.allowSecureAddressAging:
      cmds.addCommand( "switchport port-security mac-address aging" )
   elif options.saveAllDetail or options.saveAll:
      cmds.addCommand( "no switchport port-security mac-address aging" )

   if portSecConfig.allowSecureAddressMoves:
      cmds.addCommand( "switchport port-security mac-address moveable" )
   elif portSecConfig.allowSecureAddressMovesData:
      cmds.addCommand( "switchport port-security mac-address moveable data" )
   elif portSecConfig.allowSecureAddressMovesPhone:
      cmds.addCommand( "switchport port-security mac-address moveable phone" )
   elif options.saveAllDetail or options.saveAll:
      cmds.addCommand( "no switchport port-security mac-address moveable" )

   if not portSecConfig.persistenceEnabled:
      cmds.addCommand( "switchport port-security persistence disabled" )
   elif options.saveAllDetail or options.saveAll:
      cmds.addCommand( "no switchport port-security persistence disabled" )

   if togglePortSecStickyMACEnabled() and portSecConfig.persistenceShutdownEnabled:
      cmds.addCommand( "switchport port-security persistence shutdown" )
   elif togglePortSecStickyMACEnabled() and (
         options.saveAllDetail or options.saveAll ):
      cmds.addCommand( "no switchport port-security persistence shutdown" )

   if portSecConfig.chipBased:
      cmds.addCommand( "switchport port-security violation protect chip-based" )
   elif options.saveAllDetail or options.saveAll:
      cmds.addCommand( "no switchport port-security violation protect chip-based" )
