#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from collections import defaultdict
import Cell
import CliSave
import EthIntfUtil
import PtpLib
import Tac
import Tracing
from CliSavePlugin.IntfCliSave import IntfConfigMode
from IpLibConsts import DEFAULT_VRF
from MultiRangeRule import multiRangeToCanonicalString

from CliMode.PtpUnicastNegotiation import PtpUnicastNegotiationMode

from Toggles.PtpLibToggleLib import togglePtpIsolatedSlaveEnabled

__defaultTraceHandle__ = Tracing.Handle( 'PtpCli' )
PtpMode = Tac.Type( 'Ptp::PtpMode' )
CliConstants = Tac.Type( "Ptp::Constants" )
CliRegionConfigConstants = Tac.Type( "Ptp::CliRegionConfigConstants" )
mtn = Tac.Type( "Ptp::MessageTypeNumber" )
PortChannelIntfId = Tac.Type( "Arnet::PortChannelIntfId" )

class UnicastNegotiationConfigMode( PtpUnicastNegotiationMode, CliSave.Mode ):
   def __init__( self, param ):
      PtpUnicastNegotiationMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( UnicastNegotiationConfigMode,
                                       before=[ IntfConfigMode ] )
UnicastNegotiationConfigMode.addCommandSequence( 'Ptp.unicastNegotiation' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ptp.global',
                                             before=[ IntfConfigMode ] )
IntfConfigMode.addCommandSequence( 'Ptp.intf' )

# Mapping of enum names to CLI names
ptpModes = { 'ptpDisabled' : 'disabled',
             'ptpBoundaryClock' : 'boundary',
             'ptpOneStepBoundaryClock' : 'boundary one-step',
             'ptpEndToEndTransparentClock' : 'e2etransparent',
             'ptpOneStepEndToEndTransparentClock' : 'e2etransparent one-step',
             'ptpPeerToPeerTransparentClock' : 'p2ptransparent',
             'ptpOrdinaryMasterClock' : 'ordinarymaster',
             'ptpGeneralized' : 'gptp' }
# using None because we're setting it with [no|default] ptp profile
# There is no specific keyword to specify the default profile
ptpProfiles = { 'ptpDefaultProfile' : None,
                'ptpG8275_2' : 'g8275.2',
                'ptpG8275_1' : 'g8275.1',
              }

delayMechanisms = { 'e2e' : 'e2e',
                    'p2p' : 'p2p' }

transportModes = { 'layer2' : 'layer2',
                   'ipv4' : 'ipv4',
                   'ipv6' : 'ipv6' }

roles = { 'dynamic' : 'dynamic',
          'master' : 'master' }

EthAddr = Tac.Type( "Arnet::EthAddr" )
G82751MacAddrs = { EthAddr.ethAddrPtp : 'forwardable',
                   EthAddr.ethAddrPtpPeerDelay : 'non-forwardable' }

# Mapping of non-enum global attribute names to CLI names
globalAttrs = { 'ptpMode' : ( 'mode', ptpModes ),
                'ptpProfile' : ( 'profile', ptpProfiles ),
                'priority1' : 'priority1',
                'priority2' : 'priority2',
                'localPriority': 'local-priority',
                'clockIdentity' : 'clock-identity',
                'clockQuality' : 'clock-accuracy',
                'domainNumber' : 'domain',
                'srcIp4' : 'source ip',
                'srcIp6' : 'source ipv6',
                'ttl' : 'ttl',
                'holdPtpTimeInterval' : 'hold-ptp-time',
                'globalDscpEvent' : 'message-type event dscp',
                'globalDscpGeneral' : 'message-type general dscp',
                'syncFromSystemTime' : 'free-running source clock',
                'numSkewSamplesToCalibrated' : 'skew sample size' }

# Mapping of interface attribute names to CLI names
intfAttrs = { 'logAnnounceInterval' : 'announce interval',
              'logMinDelayReqInterval' : 'delay-req interval',
              'logMinPdelayReqInterval' : 'pdelay-req interval',
              'linkDelayThreshold' : 'pdelay-neighbor-threshold',
              ( 'logSyncInterval', 'syncIntervalUseDeprecatedCmd', True ) :
                 'sync interval',
              ( 'logSyncInterval', 'syncIntervalUseDeprecatedCmd', False ) :
                 'sync-message interval',
              'syncReceiptTimeout' : 'sync timeout',
              'announceReceiptTimeout' : 'announce timeout',
              'delayMechanism' : ( 'delay-mechanism', delayMechanisms ),
              'transportMode' : ( 'transport', transportModes ),
              'role' : ( 'role', roles ),
              'dscpEvent' : 'message-type event dscp',
              'dscpGeneral' : 'message-type general dscp',
              'localPriority': 'local-priority',
              'pktTxMacAddress' : ( 'profile g8275.1 destination mac-address',
                 G82751MacAddrs ),
              'vlanConfig' : 'vlan' }

intfAttrs[ 'loopbackSourceIntf' ] = 'local-interface'

managementIntfAttrs = {
   'logAnnounceInterval' : 'announce interval',
   'logMinDelayReqInterval' : 'delay-req interval',
   'logMinPdelayReqInterval' : 'pdelay-req interval',
   'linkDelayThreshold' : 'pdelay-neighbor-threshold',
   'announceReceiptTimeout' : 'announce timeout',
   'delayMechanism' : 'delay-mechanism',
   'transportMode' : 'transport',
   'domainNumber' : 'domain',
   'priority1' : 'priority1',
   'priority2' : 'priority2',
   'ttl' : 'ttl',
}

intfHasGlobalCmd = { 'dscpEvent' : 'globalDscpEvent',
                     'dscpGeneral' : 'globalDscpGeneral' }

intfRegionAttrs = { 'networkSideDomainNumber' : 'domain-number' }

# These tac attributes use "no ptp ..." in the save-all case when they are at the
# default value.
# keys: TACC attribute in the config entity.
# values: "no" or "default". The one that should be displayed in save-all.
attrToNoOrDefault = {
}

attrToNoOrDefault[ 'loopbackSourceIntf' ] = 'no'

UcastNegAttrToCliToken = {
      'announceProfile': 'announce',
      'syncProfile': 'sync',
      'delayRespProfile': 'delay-resp',
      'duration': 'duration',
      'logInterval': 'interval'
}

def attrSortKey( item ):
   value = item[ 1 ]
   if isinstance( value, tuple ):
      return value[ 0 ]
   else:
      return value

@CliSave.saver( 'Ptp::Config', 'ptp/config', requireMounts=( 'ptp/status', ) )
def savePtpConfig( entity, root, requireMounts, options ):

   cmds = root[ 'Ptp.global' ]
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   ptpStatus = requireMounts[ 'ptp/status' ]

   if not ptpStatus.ptpSupported:
      saveAll = False
      saveAllDetail = False

   for tacAttr, cliAttr in sorted( globalAttrs.items(), key=attrSortKey ):
      if isinstance( cliAttr, tuple ):
         cliAttr, attrValues = cliAttr
      else:
         attrValues = None
      currentValue = getattr( entity, tacAttr )
      if tacAttr == "clockIdentity":
         currentValue = [ getattr( currentValue, 'v%d' % i ) for i in range( 8 ) ]
         currentValue = ':'.join( [ '%02x' % x for x in currentValue ] )
         if getattr( entity, tacAttr + "Configured" ):
            defaultValue = 'x'
         else:
            defaultValue = currentValue
      elif tacAttr == "clockQuality":
         # only for clock accuracy for now
         if cliAttr == "clock-accuracy":
            currentValue = getattr( currentValue, "clockAccuracy" )
            if getattr( entity, tacAttr + "Configured" ):
               defaultValue = getattr( CliConstants, "clockAccuracyDefault" )
            else:
               defaultValue = currentValue
         else:
            continue
      elif tacAttr.startswith( 'srcIp' ) and entity.unicastNegotiation:
         continue
      elif tacAttr == "syncFromSystemTime":
         # only deal with system vs default config for now
         defaultValue = 'hardware'
         if getattr( entity, tacAttr ):
            currentValue = 'system'
         else:
            currentValue = defaultValue
      else:
         defaultValue = getattr( CliConstants, tacAttr + "Default" )
      if ( currentValue != defaultValue ) or saveAllDetail or \
         ( saveAll and entity.ptpMode != 'ptpDisabled' ):
         if attrValues:
            currentValue = attrValues[ currentValue ]
         # At the moment, we only have dscp commands that are both per interface
         # and global. Need to special case these commands as the value is in
         # the middle of the command.
         if tacAttr.startswith( 'global' ):
            cmds.addCommand( 'ptp %s %s default' % ( cliAttr, str( currentValue ) ) )
         elif currentValue is None:
            cmds.addCommand( 'no ptp %s' % cliAttr )
         else:
            cmds.addCommand( 'ptp %s %s' % ( cliAttr, str( currentValue ) ) )

   if entity.freeRunningQuiet:
      cmds.addCommand( 'no ptp free-running' )

   key = Tac.Value( "Acl::AclTypeAndVrfName", 'ip', DEFAULT_VRF )
   aclName = entity.serviceAclTypeVrfMap.aclName.get( key )
   if aclName:
      cmds.addCommand( 'ptp ip access-group %s in' % aclName )

   if saveAllDetail:
      forceSave = True
   else:
      forceSave = ( saveAll and entity.ptpMode != 'ptpDisabled' )
   if entity.forwardPtpV1:
      cmds.addCommand( 'ptp forward-v1' )
   elif forceSave:
      cmds.addCommand( 'no ptp forward-v1' )
   if entity.forwardUnicast:
      cmds.addCommand( 'ptp forward-unicast' )
   elif forceSave:
      cmds.addCommand( 'no ptp forward-unicast' )
   if entity.managementEnabled:
      cmds.addCommand( 'ptp management all' )
   elif forceSave:
      cmds.addCommand( 'no ptp management all' )
   if entity.netSyncMonitor:
      cmds.addCommand( 'ptp netsync-monitor delay-request' )
   elif forceSave:
      cmds.addCommand( 'no ptp netsync-monitor delay-request' )
   if not entity.monitorEnabled:
      cmds.addCommand( 'no ptp monitor' )
   elif forceSave:
      cmds.addCommand( 'ptp monitor' )
   if ( entity.offsetFromMasterThreshold !=
        CliConstants.invalidOffsetFromMasterThreshold ):
      cmds.addCommand( 'ptp monitor threshold offset-from-master %d' %
                          entity.offsetFromMasterThreshold )
   elif forceSave:
      cmds.addCommand( 'no ptp monitor threshold offset-from-master' )

   if entity.meanPathDelayThreshold != CliConstants.invalidMeanPathDelayThreshold:
      cmds.addCommand( 'ptp monitor threshold mean-path-delay %d' %
                          entity.meanPathDelayThreshold )
   elif forceSave:
      cmds.addCommand( 'no ptp monitor threshold mean-path-delay' )

   if entity.skewThreshold != CliConstants.invalidSkewThreshold:
      cmds.addCommand( 'ptp monitor threshold skew %g' % entity.skewThreshold )
   elif forceSave:
      cmds.addCommand( 'no ptp monitor threshold skew' )
   if entity.skewDropThreshold != CliConstants.invalidSkewDropThreshold:
      cmds.addCommand( 'ptp monitor threshold skew %g drop' %
                       entity.skewDropThreshold )
   elif forceSave:
      cmds.addCommand( 'no ptp monitor threshold skew drop' )
   if entity.meanPathDelayDropThreshold !=\
      CliConstants.invalidMeanPathDelayDropThreshold:
      cmds.addCommand( 'ptp monitor threshold mean-path-delay %d nanoseconds drop' %
                       entity.meanPathDelayDropThreshold )
   elif forceSave:
      cmds.addCommand( 'no ptp monitor threshold mean-path-delay nanoseconds drop' )
   if entity.offsetFromMasterDropThreshold !=\
      CliConstants.invalidOffsetFromMasterDropThreshold:
      cmds.addCommand(
            'ptp monitor threshold offset-from-master %d nanoseconds drop' %
            entity.offsetFromMasterDropThreshold )
   elif forceSave:
      cmds.addCommand(
         'no ptp monitor threshold offset-from-master nanoseconds drop' )
   for msgType, msgName in [ ( mtn.messageSync, "sync" ),
                             ( mtn.messageFollowUp, "follow-up" ),
                             ( mtn.messageAnnounce, "announce" ) ]:
      msgTimer = entity.logMissingMessageTimer.get( msgType,
                 CliConstants.logMissingMessagesDefault )
      msgThreshold = entity.missingTimerThresholds.get( msgType,
                     CliConstants.missingMessageThresholdDefault )
      if msgTimer != CliConstants.logMissingMessagesDefault:
         cmds.addCommand(
               'ptp monitor threshold missing-message {} {} intervals'.format(
               msgName, msgThreshold ) )
      elif forceSave:
         cmds.addCommand(
               'no ptp monitor threshold missing-message {} intervals'.format(
               msgName ) )
   defaultMonitorMode = False
   missingMsgList = []
   for msgType, msgName in [ ( mtn.messageSync, "sync" ),
                             ( mtn.messageFollowUp, "follow-up" ),
                             ( mtn.messageDelayResp, "delay-resp" ),
                             ( mtn.messageAnnounce, "announce" ) ]:
      msgSeqId = entity.logMissingMessageSeqId.get( msgType,
                 CliConstants.logMissingMessagesDefault )
      msgThreshold = entity.missingSequenceThresholds.get( msgType,
                     CliConstants.missingMessageThresholdDefault )
      if msgSeqId != CliConstants.logMissingMessagesDefault:
         if msgThreshold == CliConstants.monitorSeqIdDefault:
            defaultMonitorMode = True
         else:
            missingMsgList.append(
                  'ptp monitor threshold missing-message {} {} sequence-ids'.format(
                     msgName, msgThreshold ) )
      elif forceSave:
         missingMsgList.append(
               'no ptp monitor threshold missing-message {} sequence-ids'.format(
               msgName ) )
   if entity.logMissingMessages and defaultMonitorMode:
      cmds.addCommand( 'ptp monitor sequence-id' )
   elif forceSave:
      cmds.addCommand( 'no ptp monitor sequence-id' )
   for monitorCmd in missingMsgList:
      cmds.addCommand( monitorCmd )
   if entity.mpassSyncTimeout != \
         CliConstants.mpassDefaultSyncTimeout or forceSave:
      cmds.addCommand( 'ptp mpass message sync timeout %d messages' % \
            entity.mpassSyncTimeout )
   if entity.mpassDelayReqTimeout != \
         CliConstants.mpassDefaultDelayReqTimeout or forceSave:
      cmds.addCommand( 'ptp mpass message delay-req timeout %d seconds' % \
            entity.mpassDelayReqTimeout )

   allSwitchIntfs = set( EthIntfUtil.allSwitchportNames( requireMounts,
                                                         includeEligible=True ) )
   regionIntfs = { key.intf for key in entity.intfRegionConfig.keys() }
   cfgIntfs = entity.intfConfig
   if forceSave:
      # Superset of configured interfaces, and switch interfaces
      cfgIntfs = allSwitchIntfs.union( set( entity.intfConfig ) )
      regionIntfs = allSwitchIntfs.union( regionIntfs )

   profiles = list( entity.ucastNegProfile )
   for profileName in sorted( profiles, key=str.lower ):
      savePtpUcastNegProfile( entity, root, profileName, saveAll )

   for intf in cfgIntfs:
      savePtpIntfConfig( entity, root, intf, saveAll )

   for intf in regionIntfs:
      saveIntfRegionConfig( entity, root, intf, saveAll )

def saveIntfRegionConfig( ptpConfig, root, intf, saveAll ):
   def minVlanSetKey( item ):
      # Iterate a dict of { some-key : listOfVlans } based on the lowest vlan ID
      # in listOfVlans
      return min( item[ 1 ] )

   def getConfigValueToVlanList( configTacAttr ):
      '''Region config under intf config is planned to be of the form:
      "ptp region [ vlan ( VLAN_SET | all ) ] region-attr VALUE"
      This method produces a dict which is keyed by VALUE for the region-attr
      and the value is a set of vlanIds which have that value.
      '''
      defaultVlanConfigValues = []
      configValueToVlans = defaultdict( list )
      for portKey, intfRegionConfig in ptpConfig.intfRegionConfig.items():
         if portKey.intf != intf:
            continue
         currentValue = getattr( intfRegionConfig, configTacAttr )
         if portKey.vlanId == CliConstants.defaultPortDSVlanId:
            # ptp region region-attr VALUE
            defaultVlanConfigValues.append( currentValue )
         else:
            # ptp region vlan VLAN_SET region-attr VALUE
            configValueToVlans[ currentValue ].append( portKey.vlanId )
      return configValueToVlans, defaultVlanConfigValues

   def resolveVlanString( vlans ):
      '''Get the vlanPart piece(s) of the regionCmdFmt.
      Returns a vlanPart per command required. If vlan 0 is configured,
      that actually corresponds to the case where vlanPart was not provided.
      Strings returned will be space delimited for insertion in regionCmdFmt
      '''
      # Handle the possible vlan part of the context.
      vlanStr = multiRangeToCanonicalString( vlans )
      if CliConstants.defaultPortDSVlanId in vlans:
         # Assertion here because the code calling resolveVlanString should
         # handle vlanID 0 seperately since it produces a seperate CLI.
         assert len( vlans ) == 1
         # Default only is how we configure the "lack" of a ptp vlan.
         return ' '
      return f' vlan {vlanStr} '

   # ptp region [ vlan ( VLAN_SET | all ) ] domain-number DOMAIN_NUMBER for example.
   regionCmdFmt = 'ptp region{vlanPart}{cliAttr} {cliValue}'
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
   intfCmds = mode[ 'Ptp.intf' ]

   # For each CLI, there should be a intfRegionAttrs entry.
   # We sort first by CLI (initially there is only 'domain-number').
   # The second layer of sorting is per CLI we sort by the lowest vlan ID in the
   # vlan ID range this CLI is applied to.
   for tacAttr, cliAttr in sorted( intfRegionAttrs.items(), key=attrSortKey ):
      # Each attribute must have a default defined in the PtpConstants.
      defaultValue = getattr( CliRegionConfigConstants, tacAttr + 'Default' )
      # Based on the current data in Sysdb, generate a dict where the keys
      # are the setting's value in Sysdb for tacAttr and the values are a set of
      # vlan IDs on which this setting has been applied.
      configValueToVlans, defaultVlanValues = getConfigValueToVlanList( tacAttr )
      # If nothing is configured for this attribute, we can emit the generic
      # "no ptp region <cli-attr>"
      anyConfigured = False

      # For non-default values in sysdb, generate the expected commands.
      # mainly this involves converting the VLANs that are set to that value into
      # a range string. If the value is configured in sysdb where VLAN=0, we
      # emit the CLI which doesn't include "vlan VLAN".
      for attrValue in defaultVlanValues:
         if attrValue != defaultValue:
            anyConfigured = True
            noVlanString = resolveVlanString( [ CliConstants.defaultPortDSVlanId ] )
            intfCmds.addCommand( regionCmdFmt.format(
               vlanPart=noVlanString, cliAttr=cliAttr, cliValue=attrValue ) )

      for attrValue, vlans in sorted( configValueToVlans.items(),
                                      key=minVlanSetKey ):
         if attrValue != defaultValue:
            anyConfigured = True
            # Region configs are stored under the intf's config section.
            vlanString = resolveVlanString( vlans )
            intfCmds.addCommand( regionCmdFmt.format(
               vlanPart=vlanString, cliAttr=cliAttr, cliValue=attrValue ) )

      if not anyConfigured and saveAll:
         intfCmds.addCommand( f'no ptp region {cliAttr}' )

def savePtpUcastNegProfile( ptpConfig, root, name, saveAll ):
   profile = ptpConfig.ucastNegProfile[ name ]
   mode = root[ UnicastNegotiationConfigMode ].getOrCreateModeInstance( name )
   cmds = mode[ 'Ptp.unicastNegotiation' ]

   msgProfiles = [ 'announceProfile', 'syncProfile', 'delayRespProfile' ]
   profileAttrs = [ 'logInterval', 'duration' ]

   for msgProfile in msgProfiles:
      for profileAttr in profileAttrs:
         value = getattr( getattr( profile, msgProfile ), profileAttr )
         cmds.addCommand( '{} {} {}'.format( UcastNegAttrToCliToken[ msgProfile ],
                                             UcastNegAttrToCliToken[ profileAttr ],
                                             value ) )

def savePtpIntfConfig( ptpConfig, root, intf, saveAll ):
   if intf in ptpConfig.intfConfig:
      intfConfig = ptpConfig.intfConfig[ intf ]
   else:
      # If creating a new instance, adding default native vlan to preserve the old
      # behaviour on a trunk port
      intfConfig = Tac.newInstance( 'Ptp::IntfConfig', 'Ethernet1', False, False )
      intfConfig.vlanConfig.newMember( 0 )
      if ptpConfig.ptpMode == PtpMode.ptpGeneralized:
         ptpType = '802Dot1AS'
      else:
         ptpType = '1588V2'
      for attr in [ 'logSyncInterval', 'logAnnounceInterval',
                    'logMinPdelayReqInterval' ]:
         defaultValue = getattr( CliConstants, attr + 'Default' + ptpType )
         setattr( intfConfig, attr, defaultValue )

   mode = None
   intfCmds = None
   enableCmd = "ptp enable"

   if saveAll or intfConfig.enabled:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      intfCmds = mode[ 'Ptp.intf' ]
      # iteration order is undeterministic, so no gurantee the master will
      # remain master
      if intfConfig.syncTestEnabled:
         enableCmd = 'ptp enable synctest'
      if intfConfig.enabled:
         intfCmds.addCommand( enableCmd )
      elif saveAll:
         intfCmds.addCommand( 'no %s' % enableCmd )

   if saveAll or intfConfig.disableManagementMessageForwarding:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      intfCmds = mode[ 'Ptp.intf' ]
      if intfConfig.disableManagementMessageForwarding:
         intfCmds.addCommand( 'ptp management drop' )
      elif saveAll:
         intfCmds.addCommand( 'no ptp management drop' )

   if togglePtpIsolatedSlaveEnabled():
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      intfCmds = mode[ 'Ptp.intf' ]
      if intfConfig.isolatedSlaveEnabled:
         intfCmds.addCommand( 'ptp diag role isolated-follower' )
      elif saveAll:
         intfCmds.addCommand( 'no ptp diag role isolated-follower' )

      if intfConfig.isolatedSlaveVlanId:
         configuredVlanSet = set( intfConfig.isolatedSlaveVlanId )

         # prune default 0 vlan from the set
         if 0 in configuredVlanSet:
            configuredVlanSet.remove( 0 )
         currentValue = multiRangeToCanonicalString( configuredVlanSet )

         if currentValue:
            intfCmds.addCommand( 'ptp diag isolated-follower vlan %s' %
                                 currentValue )
      elif saveAll:
         intfCmds.addCommand( 'no ptp diag isolated-follower vlan' )

   if PortChannelIntfId.isPortChannelIntfId( intf ) and \
         ( saveAll or intfConfig.mpassEnabled ):
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      intfCmds = mode[ 'Ptp.intf' ]
      if intfConfig.mpassEnabled:
         intfCmds.addCommand( 'ptp mpass' )
      elif saveAll:
         intfCmds.addCommand( 'no ptp mpass' )

   if PortChannelIntfId.isPortChannelIntfId( intf ):
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      intfCmds = mode[ 'Ptp.intf' ]
      if intfConfig.txMessageToIntfMap:
         if ( len( intfConfig.txMessageToIntfMap ) == len( PtpLib.lagTxMessageStr )
              and
              all( v[ 0 ] == v for v in intfConfig.txMessageToIntfMap.values() ) ):
            # all 5 message types have the same tx-interface
            intfCmds.addCommand( 'ptp debug tx-interface %s' %
                                 next(
                                       iter( intfConfig.txMessageToIntfMap.values() )
                                      ) )
         else:
            for message, txIntf in intfConfig.txMessageToIntfMap.items():
               msg = PtpLib.lagTxMessageStr[ message ]
               intfCmds.addCommand( 'ptp debug message %s tx-interface %s' %
                                    ( msg, txIntf ) )
      elif saveAll:
         intfCmds.addCommand( 'no ptp debug tx-interface' )

   for tacAttr, cliAttr in sorted( intfAttrs.items(), key=attrSortKey ):
      if isinstance( tacAttr, tuple ):
         tacAttr, attrName, attrValue = tacAttr
         if getattr( intfConfig, attrName ) != attrValue:
            continue
      if isinstance( cliAttr, tuple ):
         cliAttr, cliAttrValues = cliAttr
      else:
         cliAttrValues = None

      ptpMode = ''
      if tacAttr in [ 'logSyncInterval', 'logAnnounceInterval',
                      'logMinPdelayReqInterval' ]:
         if ptpConfig.ptpMode == PtpMode.ptpGeneralized:
            ptpMode = '802Dot1AS'
         else:
            ptpMode = '1588V2'

      if tacAttr == 'vlanConfig':
         # Default value is 0, added when intfConfig is created, to preserve the
         # old default behaviour ( sending untagged packet on a trunk port )
         defaultValue = '0'
         intfVlanConfig = getattr( intfConfig, tacAttr )
         configuredVlanSet = \
            { vlan.vlanId for vlan in intfVlanConfig.values() }
         currentValue = multiRangeToCanonicalString( configuredVlanSet )
         # No PTP VLAN information to add, just skip
         if currentValue == defaultValue:
            continue
      else:
         intfVerCmdConfiged = False
         defaultValue = None
         if tacAttr in intfHasGlobalCmd:
            intfVerCmdConfiged = getattr( intfConfig, tacAttr + "Configured" )
            defaultValue = getattr( CliConstants, intfHasGlobalCmd[ tacAttr ] + \
                                    'Default' + ptpMode )
         else:
            defaultValue = getattr( CliConstants, tacAttr + 'Default' + ptpMode )
         currentValue = getattr( intfConfig, tacAttr )

      # for cmds that have global counterparts, we don't want to save them unless
      # they are configured.
      isDefault = defaultValue == currentValue
      globConfig = saveAll or intfVerCmdConfiged or not isDefault
      if globConfig:
         if mode is None or intfCmds is None:
            mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
            intfCmds = mode[ 'Ptp.intf' ]

         if isDefault and tacAttr in attrToNoOrDefault:
            # If there is no value for the [no|default] case show "no" for saveAll.
            noOrDefaultStr = attrToNoOrDefault[ tacAttr ]
            intfCmds.addCommand( '%s ptp %s' % ( noOrDefaultStr, cliAttr ) )
         else:
            # Use the default or configured value in running-config.
            if cliAttrValues:
               cliValue = cliAttrValues[ currentValue ]
            else:
               cliValue = str( currentValue )
            intfCmds.addCommand( 'ptp %s %s' % ( cliAttr, cliValue ) )

   # Save Unicast Negotiation grantor/grantee associated profiles
   if mode is None or intfCmds is None:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      intfCmds = mode[ 'Ptp.intf' ]
   grantorIpToProfileMap = intfConfig.ucastNegGrantorIpToProfileMap
   granteeIpToProfileMap = intfConfig.ucastNegGranteeIpToProfileMap
   for cliToken, profileMap in ( ( 'candidate-grantor', grantorIpToProfileMap ),
                                 ( 'remote-grantee', granteeIpToProfileMap ) ) :
      for ipGenAddr, profileName in profileMap.items():
         profileToken = 'profile'
         profileNameToken = profileName
         # If we use the default profile, we don't need to specify the 'profile'
         # token in the CLI cmd
         if profileName == 'default':
            profileToken = ''
            profileNameToken = ''
         cmd = "ptp unicast-negotiation {} {} {} {}".format( cliToken,
                                                             ipGenAddr,
                                                             profileToken,
                                                             profileNameToken )
         intfCmds.addCommand( cmd )

@CliSave.saver( 'Ptp::SystemClockConfig', 'sys/time/ptp/config', requireMounts=(
   'sys/time/clock/config',
   'hardware/' + Cell.path( 'phy/ethtool/config' ), ) )
def savePtpSystemClockConfig( entity, root, requireMounts, options ):
   phyEthtoolConfig = requireMounts[
      'hardware/' + Cell.path( 'phy/ethtool/config' ) ]
   clockConfig = requireMounts[ 'sys/time/clock/config' ]
   cmds = root[ 'Ptp.global' ]
   if options.saveAllDetail:
      saveAll = True
   else:
      saveAll = ( options.saveAll and
                  any( ic.enabled for ic in entity.managementIntfConfig.values() ) )

   allIntfs = []
   cfgIntfs = []
   ptpSystemClockSupported = False
   if phyEthtoolConfig:
      for name in phyEthtoolConfig.phy:
         if phyEthtoolConfig.phy[ name ].ptpHardwareTimestampingSupported:
            intfId = phyEthtoolConfig.phy[ name ].intfId
            allIntfs.append( intfId )
            if intfId in entity.managementIntfConfig:
               cfgIntfs.append( intfId )
            ptpSystemClockSupported = True

   # Special case for Ptp intfs that have a config but were not instantiated yet
   for ptpIntf in ( intf for intf in entity.managementIntfConfig
                    if intf.startswith( 'Ptp' ) ):
      if ptpIntf not in allIntfs:
         allIntfs.append( ptpIntf )
      if ptpIntf not in cfgIntfs:
         cfgIntfs.append( ptpIntf )

   if not ptpSystemClockSupported:
      saveAll = False

   if ptpSystemClockSupported and clockConfig.source == 'ptp' and (
         clockConfig.source != clockConfig.defaultSource or saveAll ):
      cmds.addCommand( 'clock source ptp' )

   currentSystemClockIntf = entity.systemClockSourceIntf or None
   if ptpSystemClockSupported and currentSystemClockIntf is not None:
      cmds.addCommand(
         'ptp system-clock source interface %s' % currentSystemClockIntf )
   elif saveAll:
      cmds.addCommand( 'no ptp system-clock source interface' )

   if saveAll:
      cfgIntfs = allIntfs

   for intf in sorted( cfgIntfs ):
      savePtpManagementPortIntfConfig( entity, root, intf, saveAll )

def savePtpManagementPortIntfConfig( ptpConfig, root, intf, saveAll ):
   if intf in ptpConfig.managementIntfConfig:
      intfConfig = ptpConfig.managementIntfConfig[ intf ]
   else:
      # Create a dummy entity with default values
      intfConfig = Tac.newInstance(
         'Ptp::ManagementIntfConfig', 'Management1', False )

   mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
   intfCmds = mode[ 'Ptp.intf' ]
   if intfConfig.enabled:
      intfCmds.addCommand( 'ptp enable' )
   elif saveAll:
      intfCmds.addCommand( 'no ptp enable' )

   for tacAttr, cliAttr in sorted( managementIntfAttrs.items(),
                                   key=lambda item: item[ 1 ] ):
      if hasattr( CliConstants, tacAttr + 'Default' ):
         defaultValue = getattr( CliConstants, tacAttr + 'Default' )
      else:
         defaultValue = getattr( CliConstants, tacAttr + 'Default1588V2' )
      currentValue = getattr( intfConfig, tacAttr )

      configured = getattr( intfConfig, tacAttr + 'Configured', False )
      if ( defaultValue != currentValue ) or configured or saveAll:
         cliValue = str( currentValue )
         # The only name mismatch between tac and cli is autoConfigure <--> auto
         if tacAttr == 'delayMechanism' and cliValue == 'autoConfigure':
            cliValue = 'auto'

         intfCmds.addCommand( 'ptp %s %s' % ( cliAttr, cliValue ) )
