# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
import Ethernet
from CliMode.Generator import (
   GeneratorsModeBase,
   ProfilesModeBase,
)
from CliSavePlugin.IntfCliSave import IntfConfigMode
from EoamTypes import (
   enumToRateUnitMap,
   FeatureEnabledEnum,
)
from GeneratorCliUtils import (
   generatorTypeKwRFC2544,
   generatorDirectionEnum,
   generatorDirectionKwStr,
)
from RFC2544InitiatorCliUtils import (
   benchmarkTestSupportedKwStr,
   packetSizeKwStr,
)
from TypeFuture import TacLazyType

EthAddr = TacLazyType( 'Arnet::EthAddr' )

#---------------------------------------------------------------------
# Modes
#---------------------------------------------------------------------
class Rfc2544InitiatorCliSaveMode( GeneratorsModeBase, CliSave.Mode ):
   def __init__( self, param ):
      GeneratorsModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class Rfc2544ProfilesCliSaveMode( ProfilesModeBase, CliSave.Mode ):
   def __init__( self, param ):
      ProfilesModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( Rfc2544InitiatorCliSaveMode,
                                       after=[ IntfConfigMode ] )
Rfc2544InitiatorCliSaveMode.addCommandSequence( 'Eoam.featureConfig' )
Rfc2544InitiatorCliSaveMode.addChildMode( Rfc2544ProfilesCliSaveMode,
                                          before=[ 'Eoam.featureConfig' ] )
Rfc2544ProfilesCliSaveMode.addCommandSequence( 'Eoam.rfc2544ProfileConfig' )

@CliSave.saver( 'FlowGenerator::FeatureConfig', 'generator/featureConfig' )
def saveFeatureConfig( entity, root, requireMounts, options ):
   rfc2544InitiatorCliSaveMode = root[
      Rfc2544InitiatorCliSaveMode ].getOrCreateModeInstance(
      generatorTypeKwRFC2544 )
   cmds = rfc2544InitiatorCliSaveMode[ 'Eoam.featureConfig' ]
   if FeatureEnabledEnum.rfc2544 in entity.featureEnabled:
      cmds.addCommand( 'no disabled' )
   elif options.saveAll or options.saveAllDetail:
      cmds.addCommand( 'disabled' )

@CliSave.saver( 'Rfc2544Initiator::ProfileConfigDir',
                'rfc2544Initiator/profileConfigDir' )
def saveRfc2544ProfileConfig( entity, root, requireMounts, options ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   # Use get() instead of items(), since this python code does not run
   # in context of the Activity Loop. Refer Bug 86221 for more info.
   for profileName in entity.profileConfig:
      profile = entity.profileConfig.get( profileName )
      rfc2544InitiatorCliSaveMode = root[
         Rfc2544InitiatorCliSaveMode ].getOrCreateModeInstance(
         generatorTypeKwRFC2544 )
      rfc2544ProfilesCliSaveMode = rfc2544InitiatorCliSaveMode[
         Rfc2544ProfilesCliSaveMode ].getOrCreateModeInstance(
         ( generatorTypeKwRFC2544, profileName ) )
      cmds = rfc2544ProfilesCliSaveMode[ 'Eoam.rfc2544ProfileConfig' ]

      # ProfileConfig.direction
      if profile.direction != generatorDirectionEnum.flowDirectionUnknown:
         cmds.addCommand( 'direction %s' %
                          generatorDirectionKwStr[ profile.direction ] )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no direction' )

      # ProfileConfig.benchmarkTest
      if profile.benchmarkTest:
         testStr = ''
         for test in profile.benchmarkTest:
            # CliPlugin only allows supported tests to be configured.
            # But I still think having an additional check is better
            # to avoid corner cases like someone updated the profileConfig
            # from Acons and added some un-supported test.
            if test in benchmarkTestSupportedKwStr:
               if testStr:
                  testStr += ' '
               testStr += benchmarkTestSupportedKwStr[ test ]
         cmds.addCommand( 'test %s' % testStr )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no test' )

      # ProfileConfig.packetRateAndUnit
      if profile.packetRateAndUnit:
         cmds.addCommand( 'traffic rate %d %s' % \
                          ( profile.packetRateAndUnit.packetRateValue,
                            enumToRateUnitMap[ profile.packetRateAndUnit.
                                               packetRateUnit ] ) )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no traffic rate' )

      # ProfileConfig.packetSizeInBytes
      if profile.packetSizeInBytes:
         packetSizeStr = ''
         for packetSizeEnum in profile.packetSizeInBytes:
            if packetSizeStr:
               packetSizeStr += ' '
            packetSizeStr += packetSizeKwStr[ packetSizeEnum ]
         cmds.addCommand( 'packet size %s bytes' % packetSizeStr )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no packet size' )

      # ProfileConfig.testDurationInSeconds
      if profile.testDurationInSeconds:
         cmds.addCommand(
            'traffic duration %d seconds' % profile.testDurationInSeconds )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no traffic duration' )

      # ProfileConfig.sourceMac
      macAddrStr = ''
      if profile.sourceMac != EthAddr.ethAddrZero:
         macAddrStr += 'source ' + \
                       Ethernet.convertMacAddrToDisplay( profile.sourceMac )
      if profile.destinationMac != EthAddr.ethAddrZero:
         if macAddrStr:
            macAddrStr += ' '
         macAddrStr += 'destination ' + \
                       Ethernet.convertMacAddrToDisplay( profile.destinationMac )
      if macAddrStr:
         cmds.addCommand( 'mac address %s' % macAddrStr )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no mac address' )
