# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from Toggles import ( IpRibLibToggleLib, RoutingLibToggleLib,
                     RouterGeneralCliToggleLib)

from CliMode.RouterGeneral import (
   RouterGeneralAutoVersionBaseMode,
   RouterGeneralBaseMode,
   RouterGeneralMetricProfileBaseMode,
   RouterGeneralRoutePriorityBaseMode,
   RouterGeneralVrfBaseMode,
   RouterGeneralVrfRoutePriorityBaseMode,
   RouterGeneralVrfFilteredRibBaseMode,
)
import CliSave
from IpLibConsts import DEFAULT_VRF
from CliSavePlugin.IntfCliSave import IntfConfigMode
# Need to import this to get the ConfigTag.config config sequence
import CliSavePlugin.ConfigTagCliSave # pylint: disable=unused-import
import Tac
from TypeFuture import TacLazyType
# pkgdeps: rpmwith %{_libdir}/preinit/RouterGeneralCli

defaultRouterIdV4 = Tac.Value( 'Arnet::IpAddr' ).ipAddrZero
defaultRouterIdV6 = Tac.Value( 'Arnet::Ip6Addr' )
RoutePreference = TacLazyType( 'Routing::RoutePreference' )
RoutingGeneralConfig = TacLazyType( 'Routing::General::Config' )

def getVrfName( vrf ):
   return DEFAULT_VRF if vrf == '' else vrf

def getRouterGeneralVrfMode( rtrGeneralMode, vrfName ):
   # Get reference to the 'vrf ...' mode
   vrfName = '' if vrfName == DEFAULT_VRF else vrfName
   rtrGeneralVrfMode = rtrGeneralMode[ RouterGeneralVrfCliSaveMode ].\
                        getOrCreateModeInstance( vrfName )
   return rtrGeneralVrfMode

def getRouterGeneralVrfModeCtx( root, vrfName ):
   # Get reference to 'router general' mode
   rtrGeneralMode = root[ RouterGeneralCliSaveMode ].getSingletonInstance()

   # Get reference to the 'vrf ...' mode
   return getRouterGeneralVrfMode( rtrGeneralMode, vrfName )

def getRouterGeneralMetricProfileMode( rtrGeneralMode, profileName ):
   # Get reference to the 'metric profile ...' mode
   rtrGeneralMetricProfileMode = \
                     rtrGeneralMode[ RouterGeneralMetricProfileCliSaveMode ]. \
                     getOrCreateModeInstance(  profileName  )
   return rtrGeneralMetricProfileMode

def getRouterGeneralAutoVersionMode( rtrGeneralMode ):
   # Get reference to the 'auto-version' mode
   rtrGeneralAutoVersionMode = \
                     rtrGeneralMode[ RouterGeneralAutoVersionCliSaveMode ].\
                     getOrCreateModeInstance( ( 'autoVersion' ) )
   return rtrGeneralAutoVersionMode

def getRouterGeneralFilteredRibMode( rtrGeneralVrfMode, af, filteredRibName ):
   param = ( rtrGeneralVrfMode.vrfName, af, filteredRibName )
   rtrGeneralVrfFilteredRibMode = \
         rtrGeneralVrfMode[ RouterGeneralVrfFilteredRibCliSaveMode ].\
         getOrCreateModeInstance( param )
   return rtrGeneralVrfFilteredRibMode

class RouterGeneralCliSaveMode( RouterGeneralBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      RouterGeneralBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RouterGeneralCliSaveMode,
                                       after=[ 'ConfigTag.config',
                                          IntfConfigMode ] )
RouterGeneralCliSaveMode.addCommandSequence( 'config' )

class RouterGeneralVrfCliSaveMode( RouterGeneralVrfBaseMode, CliSave.Mode ):

   def __init__( self, param ):
      RouterGeneralVrfBaseMode.__init__( self, getVrfName( param ) )
      CliSave.Mode.__init__( self, param )

RouterGeneralCliSaveMode.addChildMode( RouterGeneralVrfCliSaveMode,
                                       after=[ 'config' ] )
RouterGeneralVrfCliSaveMode.addCommandSequence( 'vrfConfig' )

class RouterGeneralMetricProfileCliSaveMode( RouterGeneralMetricProfileBaseMode, 
                                             CliSave.Mode ):
   def __init__( self, param ):
      RouterGeneralMetricProfileBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterGeneralCliSaveMode.addChildMode( RouterGeneralMetricProfileCliSaveMode,
                                       after=[ 'config' ] )
RouterGeneralMetricProfileCliSaveMode.addCommandSequence( 'metricProfile' )

class RouterGeneralRoutePriorityCliSaveMode( RouterGeneralRoutePriorityBaseMode,
                                             CliSave.Mode ):
   def __init__( self, param ):
      RouterGeneralRoutePriorityBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

RouterGeneralCliSaveMode.addChildMode( RouterGeneralRoutePriorityCliSaveMode,
                                       after=[ 'config' ] )
RouterGeneralRoutePriorityCliSaveMode.addCommandSequence( 'routePriority' )

class RouterGeneralVrfRoutePriorityCliSaveMode(
      RouterGeneralVrfRoutePriorityBaseMode,
      CliSave.Mode ):
   def __init__( self, param ):
      RouterGeneralVrfRoutePriorityBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterGeneralVrfCliSaveMode.addChildMode( RouterGeneralVrfRoutePriorityCliSaveMode,
                                          after=[ 'vrfConfig' ] )
RouterGeneralVrfRoutePriorityCliSaveMode.addCommandSequence( 'routePriority' )

class RouterGeneralVrfFilteredRibCliSaveMode( RouterGeneralVrfFilteredRibBaseMode,
                                              CliSave.Mode ):
   def __init__( self, param ):
      RouterGeneralVrfFilteredRibBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterGeneralVrfCliSaveMode.addChildMode( RouterGeneralVrfFilteredRibCliSaveMode,
                                          after=[ 'vrfConfig' ] )
RouterGeneralVrfFilteredRibCliSaveMode.addCommandSequence( 'filteredRib' )

class RouterGeneralAutoVersionCliSaveMode( RouterGeneralAutoVersionBaseMode,
                                           CliSave.Mode ):
   def __init__( self, param ):
      RouterGeneralAutoVersionBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

RouterGeneralCliSaveMode.addChildMode( RouterGeneralAutoVersionCliSaveMode,
                                       after=[ 'config' ] )
RouterGeneralAutoVersionCliSaveMode.addCommandSequence( 'autoVersion' )

#---------------------------------------------------------------------------------
#                        C L I      S A V E R S
#---------------------------------------------------------------------------------

@CliSave.saver( 'Routing::General::Config', 'routing/general/config/global' )
def saveGeneralConfig( entity, root, requireMounts, options ):
   saveAll = options.saveAll
   routingProtocolString = Tac.newInstance( "Routing::Rib::RoutingProtocolString" )

   mode = root[ RouterGeneralCliSaveMode ].getSingletonInstance()
   cmds = mode[ 'config' ]
   if entity.fecPerNexthop:
      cmds.addCommand( 'next-hops fec dedicated' )
   elif saveAll:
      cmds.addCommand( 'no next-hops fec dedicated' )

   if entity.fecEcmpEmulated:
      cmds.addCommand( 'rib fib fec ecmp emulated' )
   elif saveAll:
      cmds.addCommand( 'no rib fib fec ecmp emulated' )

   if entity.fecEcmpOrdered:
      cmds.addCommand( 'rib fib fec ecmp ordered' )
   elif saveAll:
      cmds.addCommand( 'no rib fib fec ecmp ordered' )

   if IpRibLibToggleLib.toggleHfecSupportEnabled():
      if entity.hfecOnResolvingProto:
         for proto in entity.hfecOnResolvingProto:
            cmds.addCommand( 'rib fib fec hierarchical resolution protocol %s' %
                              routingProtocolString.hfecProtocolName( proto ) )
      elif saveAll:
         cmds.addCommand( 'no rib fib fec hierarchical resolution protocol static' )

   if RoutingLibToggleLib.toggleIpOverIpHfecEnabled():
      if entity.ipOverIpHfec:
         cmds.addCommand( 'rib fib fec hierarchical resolution' )
      elif saveAll:
         cmds.addCommand( 'no rib fib fec hierarchical resolution' )

   if entity.bfdMapAdminDownDown:
      cmds.addCommand( 'route static bfd map admin-down down' )
   elif saveAll:
      cmds.addCommand( 'no route static bfd map admin-down down' )

   if entity.ipv4Routable0ClassA:
      cmds.addCommand( 'ipv4 routable 0.0.0.0/8' )
   elif saveAll:
      cmds.addCommand( 'no ipv4 routable 0.0.0.0/8' )   

   if entity.ipv4Routable240ClassE:
      cmds.addCommand( 'ipv4 routable 240.0.0.0/4' )
   elif saveAll:
      cmds.addCommand( 'no ipv4 routable 240.0.0.0/4' )

   if entity.routerIdV4 != defaultRouterIdV4:
      cmds.addCommand( 'router-id ipv4 %s' % entity.routerIdV4 )
   elif saveAll:
      cmds.addCommand( 'no router-id ipv4' )

   if entity.routerIdV6 != defaultRouterIdV6:
      cmds.addCommand( 'router-id ipv6 %s' % entity.routerIdV6 )
   elif saveAll:
      cmds.addCommand( 'no router-id ipv6' )

   if entity.routeStaticGlobalPreferencePresent:
      cmds.addCommand( 'route static ipv4 ipv6 preference %d' %
                      entity.routeStaticGlobalPreference )
   elif saveAll:
      cmds.addCommand( 'no route static ipv4 ipv6 preference %d' %
                       RoutePreference().value )

   if entity.fecEcmpMixTunnelIp:
      cmds.addCommand( 'rib fib fec ecmp compatible tunnel ip' )
   elif saveAll:
      cmds.addCommand( 'no rib fib fec ecmp compatible tunnel ip' )

   if entity.prefixAutoExplicitNull:
      cmds.addCommand( 'route forwarding label explicit-null tunnel af-mismatch' )
   elif saveAll:
      cmds.addCommand( 'no route forwarding label explicit-null tunnel '
                       'af-mismatch' )

   if entity.resolveOverAggregates:
      cmds.addCommand( 'next-hops resolution bgp aggregates allowed' )
   elif saveAll:
      cmds.addCommand( 'no next-hops resolution bgp aggregates allowed' )

   if entity.aigpStaticAccumulate:
      cmds.addCommand( 'next-hops resolution static aigp-metric accumulated' )
   elif saveAll:
      cmds.addCommand( 'no next-hops resolution static aigp-metric accumulated' )

   if entity.aigpRouteInputAccumulate:
      cmds.addCommand( 'next-hops resolution user aigp-metric accumulated' )
   elif saveAll:
      cmds.addCommand( 'no next-hops resolution user aigp-metric accumulated' )

   if IpRibLibToggleLib.toggleUcmpEligibleEnabled():
      if entity.ucmpEligible:
         cmds.addCommand( 'rib ucmp eligibility mode link-bandwidth all' )
      elif saveAll:
         cmds.addCommand( 'no rib ucmp eligibility mode link-bandwidth all' )

   if entity.resolveNexthopGroupVias:
      cmds.addCommand( 'route static nexthop-group unresolved invalid' )
   elif saveAll:
      cmds.addCommand( 'no route static nexthop-group unresolved invalid' )

   if entity.fwdDevMtu != RoutingGeneralConfig.defaultFwdDevMtu or saveAll:
      cmds.addCommand( 'software forwarding hardware offload mtu %d' %
                       entity.fwdDevMtu )

   routeTrackConfig = entity.routeTrackConfig
   if RoutingLibToggleLib.toggleRouteVrfPrioritizationEnabled() and \
      routeTrackConfig.vrfName != '':
      cmds.addCommand( f'route event track vrf {routeTrackConfig.vrfName}' + \
                       f' {routeTrackConfig.prefix}' )
   for vrfName in sorted( entity.vrfConfig ):
      # pylint: disable=unused-variable
      vrfMode = getRouterGeneralVrfMode( mode, vrfName )
      # pylint: enable=unused-variable
      vrfCmds = vrfMode[ 'vrfConfig' ]
      vrfEntity = entity.vrfConfig[ vrfName ]
      if vrfEntity.routerIdV4 != defaultRouterIdV4:
         vrfCmds.addCommand( 'router-id ipv4 %s' % vrfEntity.routerIdV4 )
      elif saveAll:
         vrfCmds.addCommand( 'no router-id ipv4' )

      if vrfEntity.routerIdV6 != defaultRouterIdV6:
         vrfCmds.addCommand( 'router-id ipv6 %s' % vrfEntity.routerIdV6 )
      elif saveAll:
         vrfCmds.addCommand( 'no router-id ipv6' )

      if vrfEntity.fwdDevMtu:
         vrfCmds.addCommand( 'software forwarding hardware offload mtu %d' %
                           vrfEntity.fwdDevMtu )
      elif saveAll:
         vrfCmds.addCommand( 'no software forwarding hardware offload mtu' )

      if IpRibLibToggleLib.toggleIsisConditionalAdvertiseEnabled():
         for filteredRibName in sorted( vrfEntity.filteredRib ):
            filteredRib = vrfEntity.filteredRib[ filteredRibName ]
            ribAf = str( filteredRib.af )
            filteredRibMode = getRouterGeneralFilteredRibMode( vrfMode, ribAf,
                                                               filteredRibName )
            filteredRibCmd = filteredRibMode[ 'filteredRib' ]
            if filteredRib.rcfName:
               # The config model stores the function name, but it must be entered in
               # the CLI with parenthesis. Add them back in.
               filteredRibCmd.addCommand( f'match rcf {filteredRib.rcfName}()' )
            elif saveAll:
               filteredRibCmd.addCommand( 'no match rcf' )

   for profileName in sorted( entity.metricProfile ):
      profileMode = getRouterGeneralMetricProfileMode( mode, profileName )
      profileCmd = profileMode[ 'metricProfile' ]
      profileEntity = entity.metricProfile[ profileName ]
      if profileEntity.baseMetric != profileEntity.metricDefault:
         profileCmd.addCommand( 'metric %d' % profileEntity.baseMetric )
      elif saveAll:
         profileCmd.addCommand( 'default metric' )
   
      if profileEntity.metricRatio and profileEntity.metricRatio.speed != 0:
         ratio = profileEntity.metricRatio
         profileCmd.addCommand( 'metric ratio %s %s' % ( ratio.speed,
                                                          ratio.speedUnit ) )
      elif saveAll:
         profileCmd.addCommand( 'no metric ratio' )
   
      for metricRule in sorted( profileEntity.metricRules ):
         rule = profileEntity.metricRules[ metricRule ]
         profileCmd.addCommand( 'metric %d if speed <= %d %s'  
                                  % ( rule.metric,
                                      rule.speed,
                                      rule.speedUnit ) )
         
# route static ucmp forwarding ... cmd
@CliSave.saver( 'Routing::Ucmp::VrfUcmpConfigDir', 'routing/ucmp/static/vrf/config' )
def saveUcmpConfig( entity, root, requireMounts, options ):
   if not RouterGeneralCliToggleLib.toggleStaticUcmpLocalForwardingEnabled():
      return

   mode = root[ RouterGeneralCliSaveMode ].getSingletonInstance()

   for vrfName in sorted( entity.vrfConfig ):
      ucmpConfig = entity.vrfConfig[ vrfName ]
      cmd = 'route static ucmp forwarding'
      if ucmpConfig.maxUcmp != ucmpConfig.maxUcmpInvalid:
         cmd += f' fec maximum-size {ucmpConfig.maxUcmp}'
         if ucmpConfig.ucmpDeviation != ucmpConfig.ucmpDeviationInvalid:
            cmd += f' deviation {ucmpConfig.ucmpDeviation} percent'

      vrfMode = getRouterGeneralVrfMode( mode, vrfName )
      vrfCmds = vrfMode[ 'vrfConfig' ]
      vrfCmds.addCommand( cmd )

@CliSave.saver( 'Routing::AutoVersion::Config', 'routing/autoVersion/config' )
def saveAutoVersionConfig( entity, root, requireMounts, options ):
   mode = root[ RouterGeneralCliSaveMode ].getSingletonInstance()
   autoVersionMode = getRouterGeneralAutoVersionMode( mode )
   autoVersionCmds = autoVersionMode[ 'autoVersion' ]
   if entity.nhgAutoVersion:
      autoVersionCmds.addCommand( 'nexthop-group' )
   elif options.saveAll:
      autoVersionCmds.addCommand( 'no nexthop-group' )

@CliSave.saver( 'Routing::VrfPrioritizationConfigAll',
                'routing/general/config/vrfPrioritization' )
def savePrioritizationConfig( entity, root, requireMounts, options ):
   if not RoutingLibToggleLib.toggleRouteVrfPrioritizationEnabled():
      return
   saveAll = options.saveAll
   mode = root[ RouterGeneralCliSaveMode ].getSingletonInstance()
   priorityMode = mode[ RouterGeneralRoutePriorityCliSaveMode ].\
      getSingletonInstance()
   protocolCmd = priorityMode[ 'routePriority' ]
   globalPriority = entity.globalPriority.protocolPriority
   priorityMap = {
      'routePriorityLow' : 'low',
      'routePriorityMedium' : 'medium',
      'routePriorityHigh' : 'high'
   }
   defaultPriority = Tac.Type(
      'Routing::VrfPrioritizationConfig' ).defaultPriority().protocolPriority
   if saveAll:
      for proto, priority in sorted( defaultPriority.items() ):
         if proto in globalPriority:
            priority = globalPriority[ proto ]
         priority = priorityMap[ priority ]
         protocolCmd.addCommand( f'protocol {proto} {priority}' )
   else:
      for proto, priority in sorted( globalPriority.items() ):
         priority = priorityMap[ priority ]
         protocolCmd.addCommand( f'protocol {proto} {priority}' )

   # Under VRF submode, we only show explicit configuration. Default
   # configuration is unspecified and we inherit from global.
   for vrfName in sorted( entity.vrfConfig ):
      protocolPriority = entity.vrfConfig[ vrfName ].protocolPriority
      rtrGVrfMode = getRouterGeneralVrfMode( mode, vrfName )
      vrfMode = \
         rtrGVrfMode[ RouterGeneralVrfRoutePriorityCliSaveMode ].\
         getOrCreateModeInstance( vrfName )
      vrfCmds = vrfMode[ 'routePriority' ]
      for proto, priority in sorted( protocolPriority.items() ):
         priority = priorityMap[ priority ]
         vrfCmds.addCommand( f'protocol {proto} {priority}' )
