#-------------------------------------------------------------------------------
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#-------------------------------------------------------------------------------
# This module implements the CliSave code for routing isis commands
#-------------------------------------------------------------------------------

# pylint: disable=consider-using-f-string
# pylint: disable=consider-using-in

import CliSave, Tac
from RibCliLib import protoDict
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.Security import mgmtSecurityConfigPath
from IsisCliLib import LVL_KW_TO_TAC_TYPE_MAP
from CliMode.Isis import RoutingIsisMode, RoutingIsisAfMode, RoutingIsisSrMplsMode, \
      RoutingIsisMetricProfileMode, RoutingIsisTeMode, RoutingIsisAreaProxyMode, \
      RoutingIsisAfSrv6Mode, RoutingIsisAfSrv6LocatorMode
import ReversibleSecretCli
from IpLibConsts import DEFAULT_VRF
from RoutingIntfUtils import allRoutingProtocolIntfNames
from Toggles import gatedToggleLib, IpRibLibToggleLib

U32_MAX_VALUE =  0xFFFFFFFF
MILLISEC_IN_SEC = 1000

addrFamilyEnum = Tac.Type( "Routing::Isis::AddressFamily" )
protectionModeEnum = Tac.Type( "Routing::Isis::ProtectionMode" )
tacIsisLevel = Tac.Type( "Routing::Isis::Level" )
SegmentIdentifier = Tac.Type( 'Mpls::SegmentRouting::SegmentIdentifier' )
AuthMode = Tac.Type( 'Routing::Isis::AuthMode' )

#-------------------------------------------------------------------------------
# Object used for saving commands in router-isis mode.
#-------------------------------------------------------------------------------
class RouterIsisConfigMode( RoutingIsisMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingIsisMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RouterIsisConfigMode,
                                       after=[ IntfConfigMode ] )
RouterIsisConfigMode.addCommandSequence( 'Isis.instance' )

def modeToString( authMode ):
   if authMode == 'clearText':
      return 'text'
   elif authMode == 'md5':
      return 'md5'
   elif authMode == 'sha':
      return 'sha'
   elif authMode == 'sha_1':
      return 'sha-1'
   elif authMode == 'sha_224':
      return 'sha-224'
   elif authMode == 'sha_256':
      return 'sha-256'
   elif authMode == 'sha_384':
      return 'sha-384'
   elif authMode == 'sha_512':
      return 'sha-512'
   else:
      return 'None'

def getProtectionModeCmd( protectionConfig, ipv4=False, ipv6=False ):
   protectionCmd = "fast-reroute ti-lfa mode"
   if ipv4 and ipv6:
      protectionCmd = "isis " + protectionCmd
   elif ipv4:
      protectionCmd = "isis ipv4 " + protectionCmd
   elif ipv6:
      protectionCmd = "isis ipv6 " + protectionCmd

   if protectionConfig.mode == protectionModeEnum.protectionDisabled:
      protectionCmd += ' disabled'
      return protectionCmd

   if protectionConfig.mode == protectionModeEnum.linkProtection:
      protectionCmd += " link-protection"
   elif protectionConfig.mode == protectionModeEnum.nodeProtection:
      protectionCmd += " node-protection"
      
   if protectionConfig.level == tacIsisLevel.level1:
      protectionCmd += " level-1"
   elif protectionConfig.level == tacIsisLevel.level2:
      protectionCmd += " level-2"

   return protectionCmd

protectionSrlgCmd = { 'srlgLoose' : 'srlg',
                      'srlgStrict' : 'srlg strict',
                      'srlgDisabled' : 'srlg disabled' }

#------------------------------------------------------------------------------
# Object used to save 'address-family <ipv4 | ipv6 > unicast' sub mode
# commands under ISIS instance config
#------------------------------------------------------------------------------
class RouterIsisAfConfigMode( RoutingIsisAfMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingIsisAfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )
RouterIsisConfigMode.addChildMode( RouterIsisAfConfigMode )
RouterIsisAfConfigMode.addCommandSequence( 'Isis.afConfig.ipv4' )
RouterIsisAfConfigMode.addCommandSequence( 'Isis.afConfig.ipv6' )
IntfConfigMode.addCommandSequence( 'Isis.intf', after=[ 'Sr.intf' ] )

#------------------------------------------------------------------------------
# Objects used to save 'segment-routing mpls' sub mode commands under
# ISIS instance config
#------------------------------------------------------------------------------
class RouterIsisSrMplsConfigMode( RoutingIsisSrMplsMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingIsisSrMplsMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )
RouterIsisConfigMode.addChildMode( RouterIsisSrMplsConfigMode )
RouterIsisSrMplsConfigMode.addCommandSequence( 'Isis.srMpls' )

#------------------------------------------------------------------------------
# Objects used to save 'segment-routing ipv6' sub mode commands under
# IPv6 Addr-Family Config
#------------------------------------------------------------------------------
class RouterIsisAfSrv6ConfigMode( RoutingIsisAfSrv6Mode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingIsisAfSrv6Mode.__init__( self, param )
      CliSave.Mode.__init__( self, param )
RouterIsisAfConfigMode.addChildMode( RouterIsisAfSrv6ConfigMode )
RouterIsisAfSrv6ConfigMode.addCommandSequence( 'Isis.srv6' )

#------------------------------------------------------------------------------
# Objects used to save 'locator <locatorName>' sub mode commands under
# Segement Routing IPv6 Config
#------------------------------------------------------------------------------
class RouterIsisAfSrv6LocatorConfigMode( RoutingIsisAfSrv6LocatorMode,
                                         CliSave.Mode ):
   def __init__( self, param ):
      RoutingIsisAfSrv6LocatorMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )
RouterIsisAfSrv6ConfigMode.addChildMode( RouterIsisAfSrv6LocatorConfigMode )
RouterIsisAfSrv6LocatorConfigMode.addCommandSequence( 'Isis.srv6Locator' )

#------------------------------------------------------------------------------
# Objects used to save 'metric profile <profileName>' sub mode commands under
# ISIS instance config
#------------------------------------------------------------------------------
class RouterIsisMetricProfileConfigMode( RoutingIsisMetricProfileMode,
                                          CliSave.Mode ):
   def __init__( self, param ):
      RoutingIsisMetricProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )
RouterIsisConfigMode.addChildMode( RouterIsisMetricProfileConfigMode )
RouterIsisMetricProfileConfigMode.addCommandSequence( 'Isis.metricProfile' )

#------------------------------------------------------------------------------
# Objects used to save 'traffic-engineering' sub mode commands under
# ISIS instance config
#------------------------------------------------------------------------------
class RouterIsisTeConfigMode( RoutingIsisTeMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingIsisTeMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterIsisConfigMode.addChildMode( RouterIsisTeConfigMode )
RouterIsisTeConfigMode.addCommandSequence( 'Isis.te' )

#------------------------------------------------------------------------------
# Objects used to save 'area proxy' sub mode commands under
# ISIS instance config
#------------------------------------------------------------------------------
class RouterIsisAreaProxyConfigMode( RoutingIsisAreaProxyMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingIsisAreaProxyMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterIsisConfigMode.addChildMode( RouterIsisAreaProxyConfigMode )
RouterIsisAreaProxyConfigMode.addCommandSequence( 'Isis.area-proxy' )

def saveIntfConfig( isisConfig, entity, root, requireMounts, options ):

   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Isis.intf' ]
   securityConfig = requireMounts[ 'mgmt/security/config' ]

   if entity.instanceName != entity.instanceNameInvalid:
      enableCmd = f'isis enable {entity.instanceName}'
      if IpRibLibToggleLib.toggleIsisConditionalAdvertiseEnabled():
         instanceRibFilter = entity.instanceRibFilter.get( entity.instanceName )
         if instanceRibFilter:
            filters = ' '.join( sorted( instanceRibFilter.ribFilterName ) )
            if filters: # Empty string if collection was empty
               enableCmd += f' condition unicast-rib {filters}'
      cmds.addCommand( enableCmd )
   elif options.saveAll:
      cmds.addCommand( 'no isis enable' )

   for instanceName in sorted( entity.multiInstanceName ):
      cmds.addCommand( 'isis instance %s' % instanceName )

   bfdConfigEnum = Tac.Type( "Routing::Isis::IsisIntfBfdConfig" )
   if entity.bfdV4 == bfdConfigEnum.isisIntfBfdEnabled:
      cmds.addCommand( 'isis bfd' )
   elif entity.bfdV4 == bfdConfigEnum.isisIntfBfdDisabled:
      cmds.addCommand( 'no isis bfd' )
   elif options.saveAll:
      cmds.addCommand( 'default isis bfd' )
   if entity.bfdV6 == bfdConfigEnum.isisIntfBfdEnabled:
      cmds.addCommand( 'isis ipv6 bfd' )
   elif entity.bfdV6 == bfdConfigEnum.isisIntfBfdDisabled:
      cmds.addCommand( 'no isis ipv6 bfd' )
   elif options.saveAll:
      cmds.addCommand( 'default isis ipv6 bfd' )
   if entity.multiInstanceRfc8202 != entity.multiInstanceRfc8202Default:
      cmds.addCommand('isis rfc8202 disabled')
   elif options.saveAll:
      cmds.addCommand('no isis rfc8202 disabled')

   circuitTypeEnum = Tac.Type( "Routing::Isis::Level" )
   if entity.circuitType != entity.circuitTypeDefault or options.saveAll:
      if entity.circuitType == circuitTypeEnum.level1:
         cmds.addCommand( 'isis circuit-type level-1')
      elif entity.circuitType == circuitTypeEnum.level2:
         cmds.addCommand( 'isis circuit-type level-2')
      elif entity.circuitType == circuitTypeEnum.level1_2:
         cmds.addCommand( 'isis circuit-type level-1-2')

   if entity.areaProxyBoundary != entity.areaProxyBoundaryDefault:
      cmds.addCommand( 'isis area proxy boundary' )
   elif options.saveAll:
      cmds.addCommand( 'no isis area proxy boundary' )

   if entity.priority != entity.priorityDefault or options.saveAll:
      cmds.addCommand( 'isis priority %d' % entity.priority )

   if entity.helloInterval != entity.helloIntervalDefault or options.saveAll:
      cmds.addCommand( 'isis hello-interval %d' % entity.helloInterval )

   if entity.helloMultiplier != entity.helloMultiplierDefault or options.saveAll:
      cmds.addCommand( 'isis hello-multiplier %d' % entity.helloMultiplier )

   isisMetric = Tac.Type( "Routing::Isis::Metric" )
   maxIntfMetric = isisMetric.isisMetricMax
   if entity.confMetric.isSet or options.saveAll:
      if entity.confMetric.value == maxIntfMetric:
         cmds.addCommand( 'isis metric maximum' )
      elif entity.confMetric.isSet:
         cmds.addCommand( 'isis metric %d' % entity.confMetric.value )
      else:
         cmds.addCommand( 'default isis metric' ) 
         
   if entity.confMetricV6.isSet or options.saveAll:
      if entity.confMetricV6.value == maxIntfMetric:
         cmds.addCommand( 'isis ipv6 metric maximum' )
      elif entity.confMetricV6.isSet:
         cmds.addCommand( 'isis ipv6 metric %d' % entity.metricV6 )
      else:
         cmds.addCommand( 'default isis ipv6 metric' )
   
   if entity.metricProfile:
      cmds.addCommand( 'isis metric profile %s' % entity.metricProfile )
   elif options.saveAll:
      cmds.addCommand( 'no isis metric profile %s' % entity.metricProfile )

   if entity.metricProfileV6:
      cmds.addCommand( 'isis ipv6 metric profile %s' % entity.metricProfileV6 )
   elif options.saveAll:
      cmds.addCommand( 'no isis ipv6 metric profile %s' % entity.metricProfileV6 )

   if entity.mtAddressFamily == addrFamilyEnum.addressFamilyIPv4 or \
          entity.mtAddressFamily == addrFamilyEnum.addressFamilyBoth:
      cmds.addCommand( 'isis multi-topology address-family ipv4 unicast' )

   if entity.mtAddressFamily == addrFamilyEnum.addressFamilyIPv6 or \
          entity.mtAddressFamily == addrFamilyEnum.addressFamilyBoth:
      cmds.addCommand( 'isis multi-topology address-family ipv6 unicast' )

   if entity.lspInterval != entity.lspIntervalDefault or options.saveAll:
      cmds.addCommand( 'isis lsp tx interval %d' % entity.lspInterval )

   if entity.passive != entity.passiveDefault:
      cmds.addCommand( 'isis passive' )
   elif options.saveAll:
      cmds.addCommand( 'no isis passive' )

   IntfHelloPaddingEnum = Tac.Type( 'Routing::Isis::IntfHelloPadding' )
   if entity.helloPadding != entity.helloPaddingDefault:
      if entity.helloPadding == IntfHelloPaddingEnum.intfHelloPaddingOn:
         cmds.addCommand( 'isis hello padding' )
      elif entity.helloPadding == IntfHelloPaddingEnum.intfHelloPaddingAlwaysOff:
         cmds.addCommand( 'isis hello padding disabled' )
      else:
         cmds.addCommand( 'no isis hello padding' )
   elif options.saveAll:
      cmds.addCommand( 'default isis hello padding' )   

   if entity.interfaceType != entity.interfaceTypeDefault:
      cmds.addCommand( 'isis network point-to-point' )
   elif options.saveAll:
      cmds.addCommand( 'no isis network point-to-point' )

   for adjSid in sorted( entity.srSingleAdjacencySegment.values(),
                         key=lambda element: element.key ):
      af = "ipv6" if adjSid.isIpV6 else "ipv4"
      sidVal = adjSid.sid.index
      backup = " backup-eligible" if adjSid.backup else ""
      
      if adjSid.sid.isValue is True:
         cmds.addCommand( 'adjacency-segment %s p2p label %d%s' % ( af, sidVal,
                                                                     backup) )
      else:
         cmds.addCommand( 'adjacency-segment %s p2p index %d global%s' %
                          ( af, sidVal, backup ) )

   for adjSid in sorted( entity.srMultipleAdjacencySegment.values(),
                         key=lambda element: element.key ):
      af = "ipv6" if adjSid.isIpV6 else "ipv4"
      sidVal = adjSid.sid.index
      backup = " backup-eligible" if adjSid.backup else ""      
      if adjSid.sid.isValue is True:
         cmds.addCommand( 'adjacency-segment %s p2p multiple label %d%s' %
                          ( af, sidVal, backup ) )
      else:
         cmds.addCommand( 'adjacency-segment %s p2p multiple index %d global%s' %
                          ( af, sidVal, backup ) )

   keyidL1 = entity.authModeKeyidL1
   keyidL2 = entity.authModeKeyidL2
   idDefault = Tac.Value( 'Routing::Isis::AuthKeyid' ).value
   keyidStrL1 = '' if keyidL1 == idDefault else  ' key-id %d' % keyidL1
   keyidStrL2 = '' if keyidL2 == idDefault else  ' key-id %d' % keyidL2

   sharedSecretProfileNameL1 = entity.sharedSecretProfileNameL1
   sharedSecretProfileNameL2 = entity.sharedSecretProfileNameL2
   profileNameDefault = entity.profileNameDefault
   sharedSecretStrL1 = '' if sharedSecretProfileNameL1 == profileNameDefault \
         else 'shared-secret profile %s algorithm ' % sharedSecretProfileNameL1
   sharedSecretStrL2 = '' if sharedSecretProfileNameL2 == profileNameDefault \
         else 'shared-secret profile %s algorithm ' % sharedSecretProfileNameL2

   authModesEqual = ( entity.authModeL1 == entity.authModeL2 )
   profileNamesEqual = ( sharedSecretProfileNameL1 == sharedSecretProfileNameL2 )

   if authModesEqual and profileNamesEqual and keyidL1 == keyidL2:
      if entity.authModeL1 != entity.authModeDefault:
         cmd = 'isis authentication mode {}{}{}'.format(
               sharedSecretStrL1,
               modeToString( entity.authModeL1 ),
               keyidStrL1 )
         if entity.authRxDisabledL1 == entity.authRxDisabledL2 and \
            entity.authRxDisabledL1 != entity.authRxDisabledDefault:
            cmds.addCommand( cmd + ' rx-disabled' )
         elif entity.authRxDisabledL1 != entity.authRxDisabledDefault:
            cmds.addCommand( cmd + ' rx-disabled' + ' level-1' )
            cmds.addCommand( cmd + ' level-2' )
         elif entity.authRxDisabledL2 != entity.authRxDisabledDefault:
            cmds.addCommand( cmd + ' rx-disabled' + ' level-2' )
            cmds.addCommand( cmd + ' level-1' )
         else:
            cmds.addCommand( cmd )
      elif options.saveAll:
         cmds.addCommand( 'no isis authentication mode' )
   else:
      # Both levels have different modes/keys
      if entity.authModeL1 != entity.authModeDefault:
         cmd = 'isis authentication mode {}{}{}'.format(
               sharedSecretStrL1,
               modeToString( entity.authModeL1 ),
               keyidStrL1 )
         if entity.authRxDisabledL1 != entity.authRxDisabledDefault:
            cmd += ' rx-disabled'
         cmds.addCommand( cmd + ' level-1' )
      if entity.authModeL2 != entity.authModeDefault:
         cmd = 'isis authentication mode {}{}{}'.format(
               sharedSecretStrL2,
               modeToString( entity.authModeL2 ),
               keyidStrL2 )
         if entity.authRxDisabledL2 != entity.authRxDisabledDefault:
            cmd += ' rx-disabled'
         cmds.addCommand( cmd + ' level-2' )
   
   for authKey in sorted( entity.authKeyColl.values(),
                          key=lambda element: element.id ):
      securityConfig = requireMounts[ mgmtSecurityConfigPath ]
      useKey = None if securityConfig.commonKeyEnabled else \
               entity.instanceName + '_' + str( entity.authModeL1 )
      authKeyL1Sanitized = ReversibleSecretCli.getCliSaveCommand( '{}',
                                                                  securityConfig,
                                                                  authKey.authKeyL1,
                                                                  useKey,
                                                                  'DES' )
      useKey = None if securityConfig.commonKeyEnabled else \
               entity.instanceName + '_' + str( entity.authModeL2 )
      authKeyL2Sanitized = ReversibleSecretCli.getCliSaveCommand( '{}',
                                                                  securityConfig,
                                                                  authKey.authKeyL2,
                                                                  useKey,
                                                                  'DES' )
      saveAuthKeys( cmds, authKey, entity, authKeyL1Sanitized, authKeyL2Sanitized,
                    True )

   if entity.routeTagV4 != entity.routeTagNotConfigured:
      cmds.addCommand( 'isis ipv4 route-tag %d' % entity.routeTagV4 )
   elif options.saveAll:
      cmds.addCommand( 'no isis ipv4 route-tag' )
   if entity.routeTagV6 != entity.routeTagNotConfigured:
      cmds.addCommand( 'isis ipv6 route-tag %d' % entity.routeTagV6 )
   elif options.saveAll:
      cmds.addCommand( 'no isis ipv6 route-tag' )

   if entity.protectionConfigV4:
      cmd = getProtectionModeCmd( entity.protectionConfigV4, ipv4=True )
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no isis ipv4 fast-reroute ti-lfa mode' )

   if entity.protectionConfigV6:
      cmd = getProtectionModeCmd( entity.protectionConfigV6, ipv6=True )
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no isis ipv6 fast-reroute ti-lfa mode' )

   if entity.protectionConfigV4V6:
      cmd = getProtectionModeCmd( entity.protectionConfigV4V6, ipv4=True, ipv6=True )
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no isis fast-reroute ti-lfa mode' )

   if entity.protectionSrlgV4 != entity.protectionSrlgDefault:
      srlg = protectionSrlgCmd[ entity.protectionSrlgV4 ]
      cmds.addCommand( "isis ipv4 fast-reroute ti-lfa %s" % srlg )
   elif options.saveAll:
      cmds.addCommand( 'no isis ipv4 fast-reroute ti-lfa srlg' )

   if entity.protectionSrlgV6 != entity.protectionSrlgDefault:
      srlg = protectionSrlgCmd[ entity.protectionSrlgV6 ]
      cmds.addCommand( "isis ipv6 fast-reroute ti-lfa %s" % srlg )
   elif options.saveAll:
      cmds.addCommand( 'no isis ipv6 fast-reroute ti-lfa srlg' )

   if entity.protectionSrlgV4V6 != entity.protectionSrlgDefault:
      srlg = protectionSrlgCmd[ entity.protectionSrlgV4V6 ]
      cmds.addCommand( "isis fast-reroute ti-lfa %s" % srlg )
   elif options.saveAll:
      cmds.addCommand( 'no isis fast-reroute ti-lfa srlg' )

# When saving redistributed protocols, OSPF variants have a different syntax
ospfRedistCommands = {
   'protoOspf': ' match internal',
   'protoOspfAse': ' match external',
   'protoOspfNssa': ' match nssa-external',
   'protoOspf3': ' match internal',
   'protoOspf3Ase': ' match external',
   'protoOspf3Nssa': ' match nssa-external'
   }

def saveRedistribute( cmds, proto, routeMap=None, rcfName=None,
                      includeLeaked=False ):
   command = 'redistribute'
   if 'Ospf3' in proto:
      command += ' ospfv3'
   elif proto in ospfRedistCommands:
      command += ' ospf'
   elif 'Isis' in proto:
      command += ' isis instance'
   else:
      command += ' %s' % protoDict[ proto ]

   if includeLeaked:
      command += ' include leaked'
   command += ospfRedistCommands.get( proto, '' )
   if routeMap:
      assert not rcfName
      command += f' route-map { routeMap }'
   if rcfName:
      assert not routeMap
      command += f' rcf { rcfName }()'
   cmds.addCommand( command )

def saveInstanceConfig( instanceConfig, root, isisConfig, requireMounts, options ):
   saveAll = options.saveAll
   mode = root[ RouterIsisConfigMode ].getOrCreateModeInstance(
      ( instanceConfig.instanceName, instanceConfig.instanceId,
        instanceConfig.vrfName ) )
   cmds = mode[ 'Isis.instance' ]
   securityConfig = requireMounts[ 'mgmt/security/config' ]
   
   if instanceConfig.shutdown != instanceConfig.shutdownDefault:
      cmds.addCommand( 'shutdown' )
   elif saveAll:
      cmds.addCommand( 'no shutdown' )

   HelloPaddinEnum = Tac.Type( 'Routing::Isis::HelloPadding' )
   if instanceConfig.helloPadding != instanceConfig.helloPaddingDefault:
      if instanceConfig.helloPadding == HelloPaddinEnum.helloPaddingAlwaysOff:
         cmds.addCommand( 'hello padding disabled' )
      elif instanceConfig.helloPadding == HelloPaddinEnum.helloPaddingTillAdjUp:
         cmds.addCommand( 'no hello padding' )
   elif saveAll:
      cmds.addCommand( 'hello padding' )

   areaId = instanceConfig.areaIdColl[ 0 ]
   if instanceConfig.systemId != instanceConfig.systemIdInvalid:
      cmds.addCommand( f'net {areaId}.{instanceConfig.systemId}.00' )
   maxArea = Tac.Value( 'Routing::Isis::AreaIdIndex' ).max
   for index in range( 1, maxArea + 1 ):
      if instanceConfig.areaIdColl[ index ] != instanceConfig.areaIdInvalid:
         cmds.addCommand( 'area address %s' % instanceConfig.areaIdColl[ index ] )

   if instanceConfig.isisHostname != instanceConfig.isisHostnameInvalid:
      cmds.addCommand( 'is-hostname %s' % instanceConfig.isisHostname )

   if instanceConfig.routerIdV4 != instanceConfig.routerIdV4Default:
      cmds.addCommand( 'router-id ipv4 %s' % instanceConfig.routerIdV4 )
   elif saveAll:
      cmds.addCommand( 'no router-id ipv4' )

   if instanceConfig.routerIdV6 != Tac.Value( 'Arnet::Ip6Addr' ):
      cmds.addCommand( 'router-id ipv6 %s' % instanceConfig.routerIdV6 )
   elif saveAll:
      cmds.addCommand( 'no router-id ipv6' )

   isTypeEnum = Tac.Type('Routing::Isis::Level')
   if instanceConfig.level != instanceConfig.levelDefault or saveAll:
      if instanceConfig.level == isTypeEnum.level1:
         cmds.addCommand( 'is-type level-1')
      elif instanceConfig.level == isTypeEnum.level2:
         cmds.addCommand( 'is-type level-2')
      elif instanceConfig.level == isTypeEnum.level1_2:
         cmds.addCommand( 'is-type level-1-2')
  
   if instanceConfig.poiEnabledL1 != instanceConfig.poiEnabledDefault or \
         instanceConfig.poiEnabledL2 != instanceConfig.poiEnabledDefault:
      if instanceConfig.poiEnabledL1 is True and instanceConfig.poiEnabledL2 is True:
         cmds.addCommand( 'lsp purge origination-identification' )
      elif instanceConfig.poiEnabledL1 is True:
         cmds.addCommand( 'lsp purge origination-identification level-1' )
      elif instanceConfig.poiEnabledL2 is True:
         cmds.addCommand( 'lsp purge origination-identification level-2' )
   elif saveAll:
      cmds.addCommand( 'no lsp purge origination-identification' )

   saveInstanceAddrFamilyConfig( instanceConfig, mode, saveAll, requireMounts )
   
   if instanceConfig.adjacencyLogging != instanceConfig.adjacencyLoggingDefault:
      cmds.addCommand( 'log-adjacency-changes' )
   elif saveAll:
      cmds.addCommand( 'no log-adjacency-changes' )

   if instanceConfig.duplicateSystemIDLogging !=\
            instanceConfig.duplicateSystemIDLoggingDefault:
      cmds.addCommand( 'log duplicate-system-id disabled' )
   elif saveAll:
      cmds.addCommand( 'no log duplicate-system-id disabled' )

   if instanceConfig.maxLSPLifetime != instanceConfig.maxLSPLifetimeDefault or \
                                       saveAll:
      cmds.addCommand( 'max-lsp-lifetime %d' % instanceConfig.maxLSPLifetime )

   if instanceConfig.maxLSPSize != instanceConfig.maxLSPSizeDefault or saveAll:
      cmds.addCommand( 'lsp size maximum %d' % instanceConfig.maxLSPSize )

   if instanceConfig.mplsLdpSync != instanceConfig.mplsLdpSyncDefault: 
      cmds.addCommand( 'mpls ldp sync default' )
   elif saveAll:
      cmds.addCommand( 'no mpls ldp sync default' )

   for protocol in sorted( instanceConfig.redistributeConfig.values(),
                           key=lambda p: p.proto ):
      saveRedistribute( cmds, protocol.proto, routeMap=protocol.routeMap,
                        rcfName=protocol.rcfFunction,
                        includeLeaked=protocol.includeLeaked )

   if instanceConfig.microLoopPreventionProtectedConfig:
      cmd = 'timers local-convergence-delay'
      if instanceConfig.microLoopConvergenceDelay != \
         instanceConfig.microLoopConvergenceDelayDefault:
         cmd += ' %d' % instanceConfig.microLoopConvergenceDelay
      cmd += ' protected-prefixes'
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( 'no timers local-convergence-delay protected-prefixes' )

   if instanceConfig.overloadBit != instanceConfig.overloadBitDefault:
      cmds.addCommand( 'set-overload-bit' )

   if instanceConfig.overloadStartupDelay != \
          instanceConfig.overloadStartupDelayInvalid:
      if instanceConfig.overloadWaitForBgp:
         if instanceConfig.overloadStartupDelay != \
                instanceConfig.overloadWaitForBgpTimeoutDefault or saveAll:
            command = 'set-overload-bit on-startup wait-for-bgp timeout %d' % \
                instanceConfig.overloadStartupDelay
         else:
            command = 'set-overload-bit on-startup wait-for-bgp'
      else:
         command = 'set-overload-bit on-startup %d' % \
             instanceConfig.overloadStartupDelay
      cmds.addCommand( command )
      
   if saveAll and instanceConfig.overloadStartupDelay == \
          instanceConfig.overloadStartupDelayInvalid and \
          instanceConfig.overloadBit == instanceConfig.overloadBitDefault:
      cmds.addCommand( 'no set-overload-bit' )
   
   highMetricsCmd = ''
   # check if any of advertise high metrics is set
   if instanceConfig.advertiseHighMetric or \
      ( instanceConfig.advertiseHighMetricInclRedis or
        instanceConfig.advertiseHighMetricStartupDelay !=
        instanceConfig.advertiseHighMetricStartupDelayInvalid ):
      highMetricsCmd += 'advertise high-metrics'

      # check if we need to include redistributed
      if instanceConfig.advertiseHighMetricInclRedis:
         highMetricsCmd += ' include redistributed'

      # check if we need to honor startup delay
      if instanceConfig.advertiseHighMetricStartupDelay != \
            instanceConfig.advertiseHighMetricStartupDelayInvalid:
         if instanceConfig.advertiseHighMetricWaitForBgp:
            if instanceConfig.advertiseHighMetricStartupDelay != \
               instanceConfig.advertiseHighMetricWaitForBgpTimeoutDefault or \
               saveAll:
               highMetricsCmd += ' on-startup wait-for-bgp timeout %d' % \
                                 instanceConfig.advertiseHighMetricStartupDelay
            else:
               highMetricsCmd += ' on-startup wait-for-bgp'
         else:
            highMetricsCmd += ' on-startup %d' % \
                              instanceConfig.advertiseHighMetricStartupDelay

      # add the command
      cmds.addCommand( highMetricsCmd )

   if saveAll and not highMetricsCmd:
      # high metrics wasn't added but this saveAll, so save default
      cmds.addCommand( 'no advertise high-metrics' )  

   if instanceConfig.advertisePassiveOnly != \
                  instanceConfig.advertisePassiveOnlyDefault:
      if isisConfig.advertisePassiveOnlyUseNewSyntax:
         cmds.addCommand( 'advertise ip-reachability passive-only' )
      else:
         cmds.addCommand( 'advertise passive-only' )
   elif saveAll:
      cmds.addCommand( 'advertise ip-reachability' )

   if instanceConfig.advertiseIntfAddrPassiveOnly != \
            instanceConfig.advertiseIntfAddrPassiveOnlyDefault:
      cmds.addCommand( 'advertise interface-address passive-only' )
   elif saveAll:
      cmds.addCommand( 'advertise interface-address' )

   spfInt = instanceConfig.spfInterval
   if spfInt.spfIntervalsUnitsConfigured !=\
         instanceConfig.spfIntervalsUnitsConfiguredDefault:
      def _getSpfMaxIntervalString():
         if spfInt.maxWaitIntConfiguredInMsec != \
            instanceConfig.maxWaitIntConfiguredInMsecDefault:
            return '%d milliseconds' % spfInt.spfMaxWaitInt
         else:
            return '%d seconds' % ( spfInt.spfMaxWaitInt // MILLISEC_IN_SEC )
      if spfInt.spfHoldInt != \
            instanceConfig.spfHoldIntDefault or saveAll:
         cmds.addCommand( 'spf-interval %s %d milliseconds %d milliseconds' %
                          ( _getSpfMaxIntervalString(),
                            spfInt.spfStartInt,
                            spfInt.spfHoldInt ) )
      elif spfInt.spfStartInt != instanceConfig.spfStartIntDefault:
         cmds.addCommand( 'spf-interval %s %d milliseconds' %
                          ( _getSpfMaxIntervalString(),
                            spfInt.spfStartInt ) )
      elif spfInt.spfMaxWaitInt != \
              instanceConfig.spfMaxWaitIntDefault:
         cmds.addCommand( 'spf-interval %s' % _getSpfMaxIntervalString() )
   else:
      spfMaxWaitSec = spfInt.spfMaxWaitInt // MILLISEC_IN_SEC
      if spfInt.spfHoldInt != \
            instanceConfig.spfHoldIntDefault or saveAll:
         cmds.addCommand( 'spf-interval %d %d %d' %
                          ( spfMaxWaitSec,
                            spfInt.spfStartInt,
                            spfInt.spfHoldInt ) )
      elif spfInt.spfStartInt != instanceConfig.spfStartIntDefault:
         cmds.addCommand( 'spf-interval %d %d' %
                          ( spfMaxWaitSec,
                            spfInt.spfStartInt ) )
      elif spfInt.spfMaxWaitInt != \
              instanceConfig.spfMaxWaitIntDefault:
         cmds.addCommand( 'spf-interval %d' % spfMaxWaitSec )

   if instanceConfig.spfFlagsPrcEnabled != instanceConfig.spfFlagsEnabledDefault:
      cmds.addCommand( 'no spf partial prefix' )
   elif saveAll:
      cmds.addCommand( 'spf partial prefix' )
   if instanceConfig.csnpGenInterval != \
      instanceConfig.csnpGenIntervalDefault or saveAll:
      cmds.addCommand( 'timers csnp generation interval %d seconds'
                       % ( instanceConfig.csnpGenInterval ) )

   if instanceConfig.p2pInstPeriodicCsnp != \
      instanceConfig.p2pInstPeriodicCsnpDefault:
      cmds.addCommand( 'timers csnp generation p2p disabled' )
   elif saveAll:
      cmds.addCommand( 'no timers csnp generation p2p disabled' )

   if instanceConfig.lspOutDelayTimerConfig != \
      instanceConfig.lspOutDelayTimerDefault:
      cmds.addCommand( 'timers lsp out-delay %d'
                       % ( instanceConfig.lspOutDelayTimerConfig ) )
   elif saveAll:
      cmd = "no timers lsp out-delay"
      cmds.addCommand( cmd )

   if instanceConfig.lspRefreshInterval != \
         instanceConfig.lspRefreshIntervalDefault:
      cmds.addCommand( 'timers lsp refresh %d'
                       % ( instanceConfig.lspRefreshInterval ) )
   elif saveAll:
      cmd = "no timers lsp refresh"
      cmds.addCommand( cmd )

   if instanceConfig.lspGenInterval.lspGenHoldInt != \
         instanceConfig.lspGenHoldIntDefault or saveAll:
      cmds.addCommand( 'timers lsp generation %d %d %d' % 
                       ( instanceConfig.lspGenInterval.lspGenMaxWaitInt, 
                         instanceConfig.lspGenInterval.lspGenStartInt, 
                         instanceConfig.lspGenInterval.lspGenHoldInt ) )
   elif instanceConfig.lspGenInterval.lspGenStartInt != \
         instanceConfig.lspGenStartIntDefault:
      cmds.addCommand( 'timers lsp generation %d %d' % 
                       ( instanceConfig.lspGenInterval.lspGenMaxWaitInt, 
                         instanceConfig.lspGenInterval.lspGenStartInt ) )
   elif instanceConfig.lspGenInterval.lspGenMaxWaitInt != \
         instanceConfig.lspGenMaxWaitIntDefault:
      cmds.addCommand( 'timers lsp generation %d' % 
                       ( instanceConfig.lspGenInterval.lspGenMaxWaitInt ) )

   if instanceConfig.minLspRemainingLifetime == \
         instanceConfig.minLspRemainingLifetimeDisabled:
      cmds.addCommand('no timers lsp min-remaining-lifetime')
   elif instanceConfig.minLspRemainingLifetime != \
         instanceConfig.minLspRemainingLifetimeDefault or saveAll:
      cmds.addCommand('timers lsp min-remaining-lifetime %d' % 
            ( instanceConfig.minLspRemainingLifetime ) )

   keyidL1 = instanceConfig.authModeKeyidL1
   keyidL2 = instanceConfig.authModeKeyidL2
   idDefault = Tac.Value( 'Routing::Isis::AuthKeyid' ).value
   keyidStrL1 = '' if keyidL1 == idDefault else  ' key-id %d' % keyidL1
   keyidStrL2 = '' if keyidL2 == idDefault else  ' key-id %d' % keyidL2

   sharedSecretProfileNameL1 = instanceConfig.sharedSecretProfileNameL1
   sharedSecretProfileNameL2 = instanceConfig.sharedSecretProfileNameL2
   profileNameDefault = instanceConfig.profileNameDefault
   sharedSecretStrL1 = '' if sharedSecretProfileNameL1 == profileNameDefault \
         else 'shared-secret profile %s algorithm ' % sharedSecretProfileNameL1
   sharedSecretStrL2 = '' if sharedSecretProfileNameL2 == profileNameDefault \
         else 'shared-secret profile %s algorithm ' % sharedSecretProfileNameL2

   authModesEqual = ( instanceConfig.authModeL1 == instanceConfig.authModeL2 )
   profileNamesEqual = ( sharedSecretProfileNameL1 == sharedSecretProfileNameL2 )

   if authModesEqual and profileNamesEqual and keyidL1 == keyidL2:
      if instanceConfig.authModeL1 != instanceConfig.authModeDefault:
         cmd = 'authentication mode {}{}{}'.format(
               sharedSecretStrL1,
               modeToString( instanceConfig.authModeL1 ),
               keyidStrL1 )
         if instanceConfig.authRxDisabledL1 == instanceConfig.authRxDisabledL2 and \
            instanceConfig.authRxDisabledL1 != instanceConfig.authRxDisabledDefault:
            cmds.addCommand( cmd + ' rx-disabled' )
         elif instanceConfig.authRxDisabledL1 != \
               instanceConfig.authRxDisabledDefault:
            cmds.addCommand( cmd + ' rx-disabled' + ' level-1' )
            cmds.addCommand( cmd + ' level-2' )
         elif instanceConfig.authRxDisabledL2 != \
               instanceConfig.authRxDisabledDefault:
            cmds.addCommand( cmd + ' rx-disabled' + ' level-2' )
            cmds.addCommand( cmd + ' level-1' )
         else:
            cmds.addCommand( cmd )
      elif saveAll:
         cmds.addCommand( 'no authentication mode' )
   # Both levels have different auth modes/keys
   else:
      if instanceConfig.authModeL1 != instanceConfig.authModeDefault:
         cmd = 'authentication mode {}{}{}'.format(
               sharedSecretStrL1,
               modeToString( instanceConfig.authModeL1 ),
               keyidStrL1 )               
         if instanceConfig.authRxDisabledL1 != instanceConfig.authRxDisabledDefault:
            cmd += ' rx-disabled'
         cmds.addCommand( cmd + ' level-1' )
      if instanceConfig.authModeL2 != instanceConfig.authModeDefault:
         cmd = 'authentication mode {}{}{}'.format(
               sharedSecretStrL2,
               modeToString( instanceConfig.authModeL2 ),
               keyidStrL2 ) 
         if instanceConfig.authRxDisabledL2 != instanceConfig.authRxDisabledDefault:
            cmd += ' rx-disabled'
         cmds.addCommand( cmd + ' level-2' )

   srDataPlaneEnum = Tac.Type( 'Routing::SegmentRoutingCli::SrDataPlane' )
   if instanceConfig.srDataPlane == srDataPlaneEnum.srDataPlaneMpls:
      saveInstanceSrMplsConfig( instanceConfig, mode, saveAll, requireMounts )
   elif saveAll:
      cmds.addCommand( 'no segment-routing mpls' )

   if instanceConfig.gracefulRestart != instanceConfig.gracefulRestartDefault:
      cmds.addCommand( 'graceful-restart' )
   elif saveAll:
      cmds.addCommand( 'no graceful-restart' )

   if instanceConfig.grHelper != instanceConfig.grHelperDefault:
      cmds.addCommand( 'no graceful-restart-helper' )
   elif saveAll:
      cmds.addCommand( 'graceful-restart-helper' )

   if instanceConfig.grTimerT2L1 != instanceConfig.grTimerT2Default or saveAll:
      cmds.addCommand( 'graceful-restart t2 level-1 %d' %
                       instanceConfig.grTimerT2L1 )
   if instanceConfig.grTimerT2L2 != instanceConfig.grTimerT2Default or saveAll:
      cmds.addCommand( 'graceful-restart t2 level-2 %d' %
                       instanceConfig.grTimerT2L2 )
   if instanceConfig.grHoldTime != instanceConfig.grHoldTimeDefault:
      cmds.addCommand( 'graceful-restart restart-hold-time %d' %
                       instanceConfig.grHoldTime )
   elif saveAll:
      cmds.addCommand( 'no graceful-restart restart-hold-time' )

   if instanceConfig.isisTeMode != instanceConfig.isisTeModeDefault:
      saveInstanceTeConfig( instanceConfig, mode, saveAll )
   elif saveAll and instanceConfig.vrfName == DEFAULT_VRF:
      cmds.addCommand( 'no traffic-engineering' )

   if instanceConfig.isisAreaProxyMode != instanceConfig.isisAreaProxyModeDefault:
      saveInstanceAreaProxyConfig( instanceConfig, mode, saveAll )
   elif saveAll and instanceConfig.vrfName == DEFAULT_VRF:
      cmds.addCommand( 'no area proxy' )

   for authKey in sorted( instanceConfig.authKeyColl.values(),
                          key=lambda element: element.id ):
      securityConfig = requireMounts[ mgmtSecurityConfigPath ]
      useKey = None if securityConfig.commonKeyEnabled else \
               instanceConfig.instanceName + '_' + str( instanceConfig.authModeL1 )
      authKeyL1Sanitized = ReversibleSecretCli.getCliSaveCommand( '{}',
                                                                  securityConfig,
                                                                  authKey.authKeyL1,
                                                                  useKey,
                                                                  'DES' )
      useKey = None if securityConfig.commonKeyEnabled else \
               instanceConfig.instanceName + '_' + str( instanceConfig.authModeL2 )
      authKeyL2Sanitized = ReversibleSecretCli.getCliSaveCommand( '{}',
                                                                  securityConfig,
                                                                  authKey.authKeyL2,
                                                                  useKey,
                                                                  'DES' )
      saveAuthKeys( cmds, authKey, instanceConfig, authKeyL1Sanitized,
                    authKeyL2Sanitized, False )

   if instanceConfig.attachBitRouteMap != instanceConfig.attachBitRouteMapInvalid:
      cmds.addCommand( 'set-attached-bit route-map %s' 
                       % instanceConfig.attachBitRouteMap )
   elif saveAll:
      cmds.addCommand( 'no set-attached-bit route-map' )

   for metricProfileConfig in sorted( instanceConfig.metricProfiles.values(),
                                      key=lambda element: element.name ):
      saveMetricProfile( instanceConfig.vrfName, metricProfileConfig, mode, saveAll )

   if instanceConfig.publishLsdb != instanceConfig.publishLsdbDefault:
      cmds.addCommand( 'database publish' )
   elif saveAll:
      cmds.addCommand( 'no database publish' )

   if saveAll:
      if ( not instanceConfig.areaLeaderEnabledL1 and
             not instanceConfig.areaLeaderEnabledL2 ):
         cmds.addCommand( 'area leader disabled' )
      elif ( instanceConfig.areaLeaderPriorityL1 ==
         instanceConfig.areaLeaderPriorityL2 ):
         cmds.addCommand( 'area leader priority %u' %
                          instanceConfig.areaLeaderPriorityL1 )
      else:
         if not instanceConfig.areaLeaderEnabledL1:
            cmds.addCommand( 'area leader level-1 disabled' )
         else:
            cmds.addCommand( 'area leader priority %u level-1' %
                             instanceConfig.areaLeaderPriorityL1 )
         if not instanceConfig.areaLeaderEnabledL2:
            cmds.addCommand( 'area leader level-2 disabled' )
         else:
            cmds.addCommand( 'area leader priority %u level-2' %
                             instanceConfig.areaLeaderPriorityL2 )
   elif ( instanceConfig.areaLeaderEnabledL1 and
          instanceConfig.areaLeaderEnabledL2 and
          instanceConfig.areaLeaderPriorityL1 ==
          instanceConfig.areaLeaderPriorityL2 and
          instanceConfig.areaLeaderPriorityL1 !=
          instanceConfig.areaLeaderPriorityDefault ):
      cmds.addCommand( 'area leader priority %u' %
                       instanceConfig.areaLeaderPriorityL1 )
   elif ( instanceConfig.areaLeaderEnabledL1 !=
          instanceConfig.areaLeaderEnabledDefault and
          instanceConfig.areaLeaderEnabledL2 !=
          instanceConfig.areaLeaderEnabledDefault ):
      cmds.addCommand( 'area leader disabled' )
   else:
      if not instanceConfig.areaLeaderEnabledL1:
         cmds.addCommand( 'area leader level-1 disabled' )
      elif instanceConfig.areaLeaderPriorityL1 != \
         instanceConfig.areaLeaderPriorityDefault:
         cmds.addCommand( 'area leader priority %u level-1' %
                          instanceConfig.areaLeaderPriorityL1 )
      if not instanceConfig.areaLeaderEnabledL2:
         cmds.addCommand( 'area leader level-2 disabled' )
      elif instanceConfig.areaLeaderPriorityL2 != \
         instanceConfig.areaLeaderPriorityDefault:
         cmds.addCommand( 'area leader priority %u level-2' %
                          instanceConfig.areaLeaderPriorityL2 )

   if instanceConfig.dynamicFloodingL1 != instanceConfig.dynamicFloodingDefault and \
      instanceConfig.dynamicFloodingL2 != instanceConfig.dynamicFloodingDefault:
      cmds.addCommand( 'lsp flooding dynamic' )
   elif instanceConfig.dynamicFloodingL1 != instanceConfig.dynamicFloodingDefault:
      cmds.addCommand( 'lsp flooding dynamic level-1' )
   elif instanceConfig.dynamicFloodingL2 != instanceConfig.dynamicFloodingDefault:
      cmds.addCommand( 'lsp flooding dynamic level-2' )
   elif saveAll:
      cmds.addCommand( 'no lsp flooding dynamic' )

   if instanceConfig.routePrefRfc7775 != instanceConfig.routePrefRfc7775Default:
      cmds.addCommand( 'route preference rfc7775' )
   elif saveAll:
      cmds.addCommand( 'no route preference rfc7775' )

   if instanceConfig.maxRedistRoutesLimitL1 != instanceConfig. \
                                                  maxRedistRoutesLimitDefault:
      cmds.addCommand( 'route redistributed limit level-1 %d' %
                          instanceConfig.maxRedistRoutesLimitL1 )
   elif saveAll:
      cmds.addCommand( 'no route redistributed limit level-1' )
         
   if instanceConfig.maxRedistRoutesLimitL2 != instanceConfig. \
                                                  maxRedistRoutesLimitDefault:
      cmds.addCommand( 'route redistributed limit level-2 %d' %
                          instanceConfig.maxRedistRoutesLimitL2 )
   elif saveAll:
      cmds.addCommand( 'no route redistributed limit level-2' )

   if instanceConfig.lspRfc8202RxDisabled != \
         instanceConfig.lspRfc8202RxDisabledDefault:
      cmds.addCommand( 'lsp rfc8202 rx disabled' )
   elif saveAll and instanceConfig.instanceId != 0:
      cmds.addCommand( 'no lsp rfc8202 rx disabled' )

   if instanceConfig.rfc8202Disabled != \
         instanceConfig.rfc8202DisabledDefault:
      cmds.addCommand( 'rfc8202 disabled' )
   elif saveAll and instanceConfig.instanceId != 0:
      cmds.addCommand( 'no rfc8202 disabled' )

   if instanceConfig.matchAdjacencyNlpid != \
      instanceConfig.matchAdjacencyNlpidDefault:
      cmds.addCommand( 'adjacency address-family match disabled' )
   elif saveAll:
      cmds.addCommand( 'no adjacency address-family match disabled' )
   if instanceConfig.msdBaseMplsImposition != \
         instanceConfig.msdBaseMplsImpositionDefault:
      cmds.addCommand( 'no maximum-sid-depth base-mpls-imposition' )
   elif saveAll:
      cmds.addCommand( 'maximum-sid-depth base-mpls-imposition' )

def saveMetricProfile( vrfName, metricProfileConfig, root, saveAll ):
   mode = root[ RouterIsisMetricProfileConfigMode ].getOrCreateModeInstance(
      ( vrfName, metricProfileConfig.name ) )[ 'Isis.metricProfile' ]

   if metricProfileConfig.metric != metricProfileConfig.metricDefault:
      mode.addCommand( 'metric %d' % metricProfileConfig.metric )
   elif saveAll:
      mode.addCommand( 'default metric' )

   if metricProfileConfig.metricRatio:
      ratio = metricProfileConfig.metricRatio
      mode.addCommand( f'metric ratio {ratio.speed} {ratio.speedUnit}' )
   elif saveAll:
      mode.addCommand( 'no metric ratio' )

   for metricRule in metricProfileConfig.metricRules:
      rule = metricProfileConfig.metricRules[ metricRule ]
      mode.addCommand( 'metric %d if speed <= %d %s' % ( rule.metric,
                                                         rule.speed,
                                                         rule.speedUnit ) )

def saveInstanceAddrFamilyConfig( instanceConfig, root, saveAll, requireMounts ):
   if instanceConfig.addressFamily == addrFamilyEnum.addressFamilyNone:
      return

   if instanceConfig.addressFamily == addrFamilyEnum.addressFamilyIPv4 or \
          instanceConfig.addressFamily == addrFamilyEnum.addressFamilyBoth:
      modeIpV4 = root[ RouterIsisAfConfigMode ].getOrCreateModeInstance(
         ( instanceConfig.vrfName, 'ipv4', 'unicast', instanceConfig.instanceId ) )
      cmdsIpV4 = modeIpV4[ 'Isis.afConfig.ipv4' ]
      
      if instanceConfig.metricAllIntfV4 != instanceConfig.metricAllIntfDefault:
         cmdsIpV4.addCommand( 'metric %d' % instanceConfig.metricAllIntfV4 )
      elif saveAll:
         cmdsIpV4.addCommand( 'default metric' )
 
      if instanceConfig.maxEcmpV4 != instanceConfig.maxEcmpV4Invalid:
         cmdsIpV4.addCommand( 'maximum-paths %d' % instanceConfig.maxEcmpV4 )
      elif saveAll:
         routingHwStatus = requireMounts[ 'routing/hardware/status' ]
         if routingHwStatus and routingHwStatus.maxLogicalProtocolEcmp > 0:
            cmdsIpV4.addCommand(
               'maximum-paths %d' % routingHwStatus.maxLogicalProtocolEcmp )

      for protocol in sorted( instanceConfig.redistributeConfigV4.values(),
                              key=lambda p: p.proto ):
         saveRedistribute( cmdsIpV4, protocol.proto, routeMap=protocol.routeMap,
                           rcfName=protocol.rcfFunction,
                           includeLeaked=protocol.includeLeaked )

      command = 'tunnel source-protocol bgp ipv4 labeled-unicast'
      if instanceConfig.bgpLuDirectRedistributeV4:
         if instanceConfig.bgpLuDirectRedistRcfV4 != '':
            command += ' rcf %s()' % instanceConfig.bgpLuDirectRedistRcfV4
         cmdsIpV4.addCommand( command )
      elif saveAll:
         cmdsIpV4.addCommand( 'no ' + command )

      if instanceConfig.microLoopPreventionProtectedConfigV4:
         cmd = 'timers local-convergence-delay'
         if instanceConfig.microLoopConvergenceDelayV4 != \
            instanceConfig.microLoopConvergenceDelayDefault:
            cmd += ' %d' % instanceConfig.microLoopConvergenceDelayV4
         cmd += ' protected-prefixes'
         cmdsIpV4.addCommand( cmd )
      elif saveAll:
         cmdsIpV4.addCommand( 'no timers local-convergence-delay '
                              'protected-prefixes' )

      if instanceConfig.leakRouteMapL1ToL2V4 != instanceConfig.leakRouteMapInvalid:
         cmdsIpV4.addCommand( "redistribute isis level-1 into level-2 route-map %s" %
                              instanceConfig.leakRouteMapL1ToL2V4 )
      if instanceConfig.leakRouteMapL2ToL1V4 != instanceConfig.leakRouteMapInvalid:
         cmdsIpV4.addCommand( "redistribute isis level-2 into level-1 route-map %s" %
                              instanceConfig.leakRouteMapL2ToL1V4 )
      
      for addr in sorted( instanceConfig.summaryAddressL1ToL2LeakV4 ):
         cmdsIpV4.addCommand( 
            "redistribute isis level-1 into level-2 summary-address %s" % addr ) 

      if instanceConfig.bfdEnabledV4:
         cmdsIpV4.addCommand( 'bfd all-interfaces' )
      elif saveAll:
         cmdsIpV4.addCommand( 'no bfd all-interfaces' )

      if instanceConfig.ignoreAttachedBitV4 != \
            instanceConfig.ignoreAttachedBitV4Default:
         cmdsIpV4.addCommand( 'lsp match flag attached action none' )
      elif saveAll:
         cmdsIpV4.addCommand(
                  'lsp match flag attached action install default-route' )

      if instanceConfig.distanceV4L1 != instanceConfig.distanceDefault or saveAll:
         cmdsIpV4.addCommand( "distance %d level-1" % instanceConfig.distanceV4L1 )
      if instanceConfig.distanceV4L2 != instanceConfig.distanceDefault or saveAll:
         cmdsIpV4.addCommand( "distance %d level-2" % instanceConfig.distanceV4L2 )
      if instanceConfig.protectionConfigAllIntfV4:
         configProtection = instanceConfig.protectionConfigAllIntfV4
         cmd = getProtectionModeCmd( configProtection )
         cmdsIpV4.addCommand( cmd )
      elif saveAll:
         cmdsIpV4.addCommand( "no fast-reroute ti-lfa mode" )

      if instanceConfig.protectionSrlgV4 != instanceConfig.protectionSrlgDefault:
         srlg = protectionSrlgCmd[ instanceConfig.protectionSrlgV4 ]
         cmdsIpV4.addCommand( "fast-reroute ti-lfa %s" % srlg )
      elif saveAll:
         cmdsIpV4.addCommand( "no fast-reroute ti-lfa srlg" )

      if instanceConfig.igpShortcutV4 != instanceConfig.igpShortcutV4Default:
         cmdsIpV4.addCommand( "igp shortcut disabled" )
      elif saveAll:
         cmdsIpV4.addCommand( "no igp shortcut disabled" )

      if gatedToggleLib.toggleRsvpTrafficSteeringIsisEnabled():
         for tunnelName in sorted( instanceConfig.trafficSteeringRsvpV4 ):
            prefixList = instanceConfig.trafficSteeringRsvpV4[ tunnelName ]
            cmdsIpV4.addCommand( 'next-hop tunnel rsvp %s prefix-list %s' %
                                 ( tunnelName, prefixList ) )

   if instanceConfig.addressFamily == addrFamilyEnum.addressFamilyIPv6 or \
          instanceConfig.addressFamily == addrFamilyEnum.addressFamilyBoth:
      modeIpV6 = root[ RouterIsisAfConfigMode ].getOrCreateModeInstance(
         ( instanceConfig.vrfName, 'ipv6', 'unicast', instanceConfig.instanceId ) )
      cmdsIpV6 = modeIpV6[ 'Isis.afConfig.ipv6' ]

      if instanceConfig.bfdEnabledV6:
         cmdsIpV6.addCommand( 'bfd all-interfaces' )
      elif saveAll:
         cmdsIpV6.addCommand( 'no bfd all-interfaces' )
      if instanceConfig.metricAllIntfV6 != instanceConfig.metricAllIntfDefault:
         cmdsIpV6.addCommand( 'metric %d' % instanceConfig.metricAllIntfV6 )
      elif saveAll:
         cmdsIpV6.addCommand( 'default metric' ) 
      if instanceConfig.maxEcmpV6 != instanceConfig.maxEcmpV6Invalid:
         cmdsIpV6.addCommand( 'maximum-paths %d' % instanceConfig.maxEcmpV6 )
      elif saveAll:
         routing6HwStatus = requireMounts[ 'routing6/hardware/status' ]
         if routing6HwStatus and routing6HwStatus.maxLogicalProtocolEcmp > 0:
            cmdsIpV6.addCommand(
               'maximum-paths %d' % routing6HwStatus.maxLogicalProtocolEcmp )

      if instanceConfig.microLoopPreventionProtectedConfigV6:
         cmd = 'timers local-convergence-delay'
         if instanceConfig.microLoopConvergenceDelayV6 != \
            instanceConfig.microLoopConvergenceDelayDefault:
            cmd += ' %d' % instanceConfig.microLoopConvergenceDelayV6
         cmd += ' protected-prefixes'
         cmdsIpV6.addCommand( cmd )
      elif saveAll:
         cmdsIpV6.addCommand( 'no timers local-convergence-delay '
                              'protected-prefixes' )

      for protocol in sorted( instanceConfig.redistributeConfigV6.values(),
                              key=lambda p: p.proto ):
         saveRedistribute( cmdsIpV6, protocol.proto, routeMap=protocol.routeMap,
                           rcfName=protocol.rcfFunction,
                           includeLeaked=protocol.includeLeaked )

      if instanceConfig.multiTopology:
         cmdsIpV6.addCommand( 'multi-topology' )
      elif saveAll:
         cmdsIpV6.addCommand( 'no multi-topology' )

      if instanceConfig.leakRouteMapL1ToL2V6 != instanceConfig.leakRouteMapInvalid:
         cmdsIpV6.addCommand( "redistribute isis level-1 into level-2 route-map %s" %
                              instanceConfig.leakRouteMapL1ToL2V6 )
      if instanceConfig.leakRouteMapL2ToL1V6 != instanceConfig.leakRouteMapInvalid:
         cmdsIpV6.addCommand( "redistribute isis level-2 into level-1 route-map %s" %
                              instanceConfig.leakRouteMapL2ToL1V6 )
      
      for addr in sorted( instanceConfig.summaryAddressL1ToL2LeakV6 ):
         cmdsIpV6.addCommand( 
            "redistribute isis level-1 into level-2 summary-address %s" % addr )

      if instanceConfig.ignoreAttachedBitV6 != \
            instanceConfig.ignoreAttachedBitV6Default:
         cmdsIpV6.addCommand( 'lsp match flag attached action none' )
      elif saveAll:
         cmdsIpV6.addCommand(
                  'lsp match flag attached action install default-route' )

      if instanceConfig.distanceV6L1 != instanceConfig.distanceDefault or saveAll:
         cmdsIpV6.addCommand( "distance %d level-1" % instanceConfig.distanceV6L1 )
      if instanceConfig.distanceV6L2 != instanceConfig.distanceDefault or saveAll:
         cmdsIpV6.addCommand( "distance %d level-2" % instanceConfig.distanceV6L2 )

      if instanceConfig.protectionConfigAllIntfV6:
         configProtection = instanceConfig.protectionConfigAllIntfV6
         cmd = getProtectionModeCmd( configProtection )
         cmdsIpV6.addCommand( cmd )
      elif saveAll:
         cmdsIpV6.addCommand( "no fast-reroute ti-lfa mode" )

      if instanceConfig.protectionSrlgV6 != instanceConfig.protectionSrlgDefault:
         srlg = protectionSrlgCmd[ instanceConfig.protectionSrlgV6 ]
         cmdsIpV6.addCommand( "fast-reroute ti-lfa %s" % srlg )
      elif saveAll:
         cmdsIpV6.addCommand( "no fast-reroute ti-lfa srlg" )

      if instanceConfig.igpShortcutV6 != \
         instanceConfig.igpShortcutV6Default:
         cmdsIpV6.addCommand( "igp shortcut" )
      elif saveAll:
         cmdsIpV6.addCommand( "no igp shortcut" )

      if gatedToggleLib.toggleIsisSrv6Enabled():
         if instanceConfig.srv6Configured != instanceConfig.srv6ConfiguredDefault:
            saveInstanceSrv6Config( instanceConfig, modeIpV6, saveAll,
                                    requireMounts )
         elif saveAll:
            cmdsIpV6.addCommand( 'no segment-routing ipv6' )

def saveInstanceTeConfig( instanceConfig, root, saveAll ):
   mode = root[ RouterIsisTeConfigMode ].getOrCreateModeInstance(
      instanceConfig.instanceName )
   cmds = mode[ 'Isis.te' ]

   if instanceConfig.isisTeEnabled != instanceConfig.isisTeEnabledDefault:
      cmds.addCommand( 'no shutdown' )
   elif saveAll:
      cmds.addCommand( 'shutdown' )

   
   isTypeEnum = Tac.Type( 'Routing::Isis::Level' )
   if instanceConfig.isisTeLevel != instanceConfig.isisTeLevelDefault or saveAll:
      if instanceConfig.isisTeLevel == isTypeEnum.level1:
         cmds.addCommand( 'is-type level-1' )
      elif instanceConfig.isisTeLevel == isTypeEnum.level2:
         cmds.addCommand( 'is-type level-2' )
      elif instanceConfig.isisTeLevel == isTypeEnum.level1_2:
         cmds.addCommand( 'is-type level-1-2' )

def saveInstanceSrMplsConfig( instanceConfig, root, saveAll, requireMounts ):
   mode = root[ RouterIsisSrMplsConfigMode ].getOrCreateModeInstance(
      instanceConfig.instanceName )
   cmds = mode[ 'Isis.srMpls' ]
   if instanceConfig.srRouterId != instanceConfig.routerIdV4Default:
      cmds.addCommand( 'router-id %s' % instanceConfig.srRouterId )
   elif saveAll:
      cmds.addCommand( 'no router-id' )
   if instanceConfig.srEnabled != instanceConfig.srEnabledDefault:
      cmds.addCommand( 'no shutdown' )
   elif saveAll:
      cmds.addCommand( 'shutdown' )

   if instanceConfig.srSegmentPathVerification != \
      instanceConfig.srSegmentPathVerificationDefault:
      cmds.addCommand(
         'segment path verification prefix-segment adjacency-segment' )
   elif saveAll:
      cmds.addCommand(
         'no segment path verification prefix-segment adjacency-segment' )

   # pylint: disable-next=singleton-comparison
   if instanceConfig.inspectProxyAttachedFlag == True:
      cmds.addCommand( 'proxy-node-segment attached-flag inspect' )
   elif saveAll:
      cmds.addCommand( 'no proxy-node-segment attached-flag inspect' )

   srAdjAllocationOptions = Tac.Type( 
       "Routing::SegmentRoutingCli::SrAdjAllocationType" )
   if instanceConfig.srAdjSegmentAlloc == \
          srAdjAllocationOptions.srAdjacencyAllocationNone:
      cmds.addCommand( 'adjacency-segment allocation none' )
   elif instanceConfig.srAdjSegmentAlloc == \
          srAdjAllocationOptions.srAdjacencyAllocationAll:
      cmds.addCommand( 'adjacency-segment allocation all-interfaces' )
   elif instanceConfig.srAdjSegmentAlloc == \
        srAdjAllocationOptions.srAdjacencyAllocationSrOnlyBackup:
      cmds.addCommand( 'adjacency-segment allocation sr-peers backup-eligible')
   elif instanceConfig.srAdjSegmentAlloc == \
         srAdjAllocationOptions.srAdjacencyAllocationAllBackup:
      cmds.addCommand( 'adjacency-segment allocation all-interfaces'
                       ' backup-eligible' )
   elif saveAll:
      cmds.addCommand( 'adjacency-segment allocation sr-peers')

   if instanceConfig.srAdjSegmentSidReuseTimeout == U32_MAX_VALUE:
      cmds.addCommand( 'adjacency-segment sid reuse timeout infinite' )
   elif instanceConfig.srAdjSegmentSidReuseTimeout != \
        instanceConfig.srAdjSegmentSidReuseTimeoutDefault:
      cmds.addCommand( 'adjacency-segment sid reuse timeout %s' %
                       instanceConfig.srAdjSegmentSidReuseTimeout )
   elif saveAll:
      cmds.addCommand( 'adjacency-segment sid reuse timeout %s' %
                       instanceConfig.srAdjSegmentSidReuseTimeoutDefault )

   proxyNodeSegmentList = []
   prefixSegmentList = []
   for prefixSegment in instanceConfig.srPrefixSegments.values():
      if prefixSegment.isProxyNode:
         proxyNodeSegmentList.append( prefixSegment )
      else:
         prefixSegmentList.append( prefixSegment )

   levelTypeEnum = Tac.Type( 'Routing::Isis::Level' )
   for entry in proxyNodeSegmentList:
      if entry.level == levelTypeEnum.level1:
         level = ' level-1'
      elif entry.level == levelTypeEnum.level2:
         level = ' level-2'
      else:
         level = ''

      segmentType = 'index'
      segmentValue = entry.index
      if entry.label != entry.labelInvalid:
         segmentType = 'label'
         segmentValue = entry.label

      if entry.isAttached:
         attachedKw = ' attached'
      else:
         attachedKw = ''

      if entry.range != 1:
         rangeKw = f' range {entry.range}'
      else:
         rangeKw = ''

      ignoreConflictKw = ''
      if entry.ignoreConflict:
         ignoreConflictKw = ' conflict ignore'
      
      cmds.addCommand( 'proxy-node-segment %s %s %d%s%s%s%s' % (
                                           entry.algoNamePrefix.prefix,
                                           segmentType,
                                           segmentValue,
                                           rangeKw,
                                           ignoreConflictKw,
                                           level,
                                           attachedKw ) )

   for entry in prefixSegmentList:
      algoName = ''
      if entry.algoNamePrefix.algoName != '':
         algoName = f' flex-algo {entry.algoNamePrefix.algoName}'
      explicitNullKw = ' explicit-null' if entry.isExplicitNull else ''
      noPhpKw = ' no-php' if ( entry.isNoPhp and not entry.isExplicitNull ) else ''
      segmentType = 'index'
      segmentValue = entry.index
      if entry.label != entry.labelInvalid:
         segmentType = 'label'
         segmentValue = entry.label
      
      cmds.addCommand( 'prefix-segment %s %s %d%s%s%s' % (
                                          entry.algoNamePrefix.prefix,
                                          segmentType,
                                          segmentValue,
                                          algoName,
                                          noPhpKw,
                                          explicitNullKw ) )

   for entry in sorted( instanceConfig.flexAlgoAlgorithms ):
      app = instanceConfig.flexAlgoAlgorithms[ entry ]
      cmd = 'flex-algo ' + app.name
      if app.level != app.levelDefault:
         cmd += ' ' + LVL_KW_TO_TAC_TYPE_MAP.reverse()[ app.level ]
      elif saveAll:
         cmd += ' level-1-2'
      if app.advertised != app.advertisedDefault:
         cmd += ' advertised'
      cmds.addCommand( cmd )

def saveInstanceSrv6Config( instanceConfig, root, saveAll, requireMounts ):
   mode = root[ RouterIsisAfSrv6ConfigMode ].getOrCreateModeInstance(
      instanceConfig.instanceName )
   cmds = mode[ 'Isis.srv6' ]
   if instanceConfig.srv6Enabled != \
      instanceConfig.srv6EnabledDefault:
      cmds.addCommand( 'disabled' )
   elif saveAll:
      cmds.addCommand( 'no disabled' )
   for locatorName in instanceConfig.srv6Locator:
      locatorMode = mode[ RouterIsisAfSrv6LocatorConfigMode ].\
                    getOrCreateModeInstance( locatorName )
      cmds = locatorMode[ 'Isis.srv6Locator' ]

def saveInstanceAreaProxyConfig( instanceConfig, root, saveAll ):
   mode = root[ RouterIsisAreaProxyConfigMode ].getOrCreateModeInstance(
      instanceConfig.instanceName )
   cmds = mode[ 'Isis.area-proxy' ]

   if instanceConfig.proxySystemId != instanceConfig.systemIdInvalid:
      cmds.addCommand( 'net {}.{}.00'.format( instanceConfig.proxyAreaId,
                                          instanceConfig.proxySystemId ) )

   if instanceConfig.proxyHostname != instanceConfig.isisHostnameInvalid:
      cmds.addCommand( 'is-hostname %s' % instanceConfig.proxyHostname )

   for prefix in sorted( instanceConfig.areaSegment ):
      sid = instanceConfig.areaSegment[ prefix ]
      indexLabel = "label" if sid.isValue else "index"
      cmds.addCommand( 'area segment %s %s %s' %
                       ( prefix, indexLabel, str( sid.index ) ) )

   if instanceConfig.proxyRouterIdV4 != instanceConfig.routerIdV4Default:
      cmds.addCommand( 'router-id ipv4 %s' % instanceConfig.proxyRouterIdV4 )
   elif saveAll:
      cmds.addCommand( 'no router-id ipv4' )

   if instanceConfig.proxyRouterIdV6 != Tac.Value( 'Arnet::Ip6Addr' ):
      cmds.addCommand( 'router-id ipv6 %s' % instanceConfig.proxyRouterIdV6 )
   elif saveAll:
      cmds.addCommand( 'no router-id ipv6' )

   if ( instanceConfig.isisAreaProxyEnabled !=
        instanceConfig.isisAreaProxyEnabledDefault ):
      cmds.addCommand( 'no shutdown' )
   elif saveAll:
      cmds.addCommand( 'shutdown' )

authAlgoEnum = Tac.Type( "Routing::Isis::AuthAlgo" )
shaEnumToStr = {
   authAlgoEnum.sha1 : 'sha-1',
   authAlgoEnum.sha224 : 'sha-224',
   authAlgoEnum.sha256 : 'sha-256',
   authAlgoEnum.sha384 : 'sha-384',
   authAlgoEnum.sha512 : 'sha-512'
}

def saveAuthKeys( cmds, authKey, config, authKeyL1Sanitized, authKeyL2Sanitized,
                  isIntfConfig ):
   prefix = ''
   if isIntfConfig:
      prefix = 'isis '
   keyidStr = ''
   shaApadRfc5310L1 = 'rfc-5310 ' if authKey.shaApadRfc5310L1 else ''
   shaApadRfc5310L2 = 'rfc-5310 ' if authKey.shaApadRfc5310L2 else ''

   idDefault = Tac.Value( 'Routing::Isis::AuthKeyid' ).value
   if authKey.id != idDefault:
      keyidStr = f'key-id {authKey.id} algorithm {shaEnumToStr[authKey.algorithm]} '
   if ( authKey.authKeyL1.clearTextEqual( authKey.authKeyL2.getClearText() ) and
        config.authModeL1 == config.authModeL2 and
        shaApadRfc5310L1 == shaApadRfc5310L2 ):
      # pylint: disable=protected-access
      authKeyL1Sanitized.format_ = ( f'{prefix}authentication '
                                     f'{keyidStr}{shaApadRfc5310L1}key ' +
                                     authKeyL1Sanitized.format_ )
      cmds.addCommand( authKeyL1Sanitized )
   else:
      # pylint: disable=protected-access
      if not authKey.authKeyL1.clearTextEqual( authKey.authKeyInvalid ):
         authKeyL1Sanitized.format_ = (
            f'{prefix}authentication {keyidStr}{shaApadRfc5310L1}key ' +
            authKeyL1Sanitized.format_ + ' level-1' )
         cmds.addCommand( authKeyL1Sanitized )
      if not authKey.authKeyL2.clearTextEqual( authKey.authKeyInvalid ):
         authKeyL2Sanitized.format_ = (
            f'{prefix}authentication {keyidStr}{shaApadRfc5310L2}key ' +
            authKeyL2Sanitized.format_ + ' level-2' )
         cmds.addCommand( authKeyL2Sanitized )

#-------------------------------------------------------------------------------
# save config under global config mode
#-------------------------------------------------------------------------------

CliSave.GlobalConfigMode.addCommandSequence( 'Isis.config',
                                             after=[ RouterIsisConfigMode ] )

@CliSave.saver( 'Routing::Isis::Config', 'routing/isis/config',
                requireMounts = ( 'interface/config/all', 
                                  'interface/status/all',
                                  'routing/hardware/statuscommon',
                                  'routing/hardware/status', 
                                  'routing6/hardware/status',
                                  mgmtSecurityConfigPath,
                                  'routing/sr/config' ) )
def saveConfig( isisConfig, root, requireMounts, options ):

   # Check if any Isis configuration is made.
   isisInstanceConfigured = len( isisConfig.instanceConfig ) > 0
   isisInterfacesConfigured = len( isisConfig.intfConfig ) > 0

   isisConfigEnabled = ( isisInstanceConfigured or isisInterfacesConfigured )

   # No need to show isis configs if nothing is configured
   if not isisConfigEnabled and not options.saveAllDetail:
      return

   for instanceName in sorted( isisConfig.instanceConfig ):
      saveInstanceConfig( isisConfig.instanceConfig[ instanceName ],
                          root, isisConfig, requireMounts, options )

   if options.saveAllDetail:
      rpIntfs = allRoutingProtocolIntfNames( requireMounts, includeEligible=True )
      intfs = set( rpIntfs )
      intfs.update( isisConfig.intfConfig )
   elif options.saveAll:
      # Display Isis configuration on routing protocol interfaces as well as
      # switchports present in intfConfig collection.
      rpIntfs = allRoutingProtocolIntfNames( requireMounts )
      intfs = set( rpIntfs )
      intfs.update( isisConfig.intfConfig )
   else:
      intfs = isisConfig.intfConfig

   # save config for each interface
   for intfName in intfs:
      intfConfig = isisConfig.intfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Routing::Isis::IntfConfig', intfName )
         else:
            continue
      saveIntfConfig( isisConfig, intfConfig, root, requireMounts, options )

srSidInvalid = Tac.Type( "Routing::SegmentRoutingCli::Constants" ).srSidInvalid
