# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements the CliSave code routing ospf commands
#-------------------------------------------------------------------------------
import CliSave, Arnet, Tac
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.Security import mgmtSecurityConfigPath
import ReversibleSecretCli
from RibCliLib import protoDict
from IpUtils import IpAddress
from CliMode.Ospf import (
      RoutingOspfMode,
      RoutingOspfTeMode,
      RoutingOspfSrMplsMode,
      RoutingOspfGeneralMode,
      RouterOspfAreaShamLinkModeBase,
      )
from RoutingIntfUtils import (
      isLoopback,
      isManagement,
      allIpIntfNames,
      allRoutingProtocolIntfNames,
      )
from IpLibConsts import DEFAULT_VRF
import OspfConsts
import Toggles.gatedToggleLib
import Toggles.OspfToggleLib

ISIS_LEVEL_MAP = { 'level1' : 'level-1',
                   'level2' : 'level-2',
                   'level1_2' : 'level-1-2' }

def getDomainIdStr( domainId ):
   domainId = Tac.newInstance( 'Routing::Ospf::DomainId', domainId )
   return domainId.stringValue()
def getProtectionModeCmd( protectionConfig, intf=False ):
   protectionModeEnum = Tac.Type( "Routing::Ospf::ProtectionMode" )
   protectionCmd = "fast-reroute ti-lfa mode"
   if intf:
      protectionCmd = "ip ospf " + protectionCmd

   if protectionConfig == protectionModeEnum.protectionDisabled:
      protectionCmd += ' disabled'
      return protectionCmd

   if protectionConfig == protectionModeEnum.linkProtection:
      protectionCmd += " link-protection"
   elif protectionConfig == protectionModeEnum.nodeProtection:
      protectionCmd += " node-protection"
      
   return protectionCmd
#-------------------------------------------------------------------------------
# Object used for saving commands in router-ospf mode.
#-------------------------------------------------------------------------------
class RouterOspfConfigMode( RoutingOspfMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingOspfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterOspfGeneralConfigMode( RoutingOspfGeneralMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingOspfGeneralMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterOspfTeConfigMode( RoutingOspfTeMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingOspfTeMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterOspfSrMplsConfigMode( RoutingOspfSrMplsMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingOspfSrMplsMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterOspfAreaShamLinkConfigMode( RouterOspfAreaShamLinkModeBase,
                                        CliSave.Mode ):
   def __init__( self, param ):
      RouterOspfAreaShamLinkModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RouterOspfGeneralConfigMode,
                                       after=[ IntfConfigMode ] )
RouterOspfGeneralConfigMode.addCommandSequence( 'Ospf.general' )

CliSave.GlobalConfigMode.addChildMode( RouterOspfConfigMode,
                                       after=[ RouterOspfGeneralConfigMode ] )
RouterOspfConfigMode.addCommandSequence( 'Ospf.instance' )

IntfConfigMode.addCommandSequence( 'Ospf.intf', after=[ 'Ira.ipIntf' ] )

RouterOspfConfigMode.addChildMode( RouterOspfTeConfigMode )
RouterOspfTeConfigMode.addCommandSequence( 'Ospf.te' )

RouterOspfConfigMode.addChildMode( RouterOspfSrMplsConfigMode )
RouterOspfSrMplsConfigMode.addCommandSequence( 'Ospf.srMpls' )

RouterOspfConfigMode.addChildMode( RouterOspfAreaShamLinkConfigMode )
RouterOspfAreaShamLinkConfigMode.addCommandSequence( 'Ospf.shamLink' )

def saveIntfConfig( entity, root, requireMounts, options ):
   if isLoopback( entity.name ):
      saveLoopbackIntfConfig( entity, root, options )
   else:
      saveNonLoopbackIntfConfig( entity, root, requireMounts, options )

def saveLoopbackIntfConfig( entity, root, options ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Ospf.intf' ]
   saveAll = options.saveAll

   if entity.cost != entity.costDefault:
      cmds.addCommand( 'ip ospf cost %d' % entity.cost )
   elif entity.metricProfileName:
      cmds.addCommand( 'ip ospf cost profile %s' % entity.metricProfileName )
   elif saveAll:
      cmds.addCommand( 'no ip ospf cost' )

   if entity.areaIdPresent != entity.areaIdPresentDefault:
      cmds.addCommand( 'ip ospf area %s' % entity.areaId )

def saveNonLoopbackIntfConfig( entity, root, requireMounts, options ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Ospf.intf' ]
   saveAll = options.saveAll

   if entity.bfdIntfState == 'ospfIntfBfdEnabled':
      cmds.addCommand( 'ip ospf neighbor bfd' )
   elif entity.bfdIntfState == 'ospfIntfBfdDisabled':
      cmds.addCommand( 'no ip ospf neighbor bfd' )
   elif saveAll:
      cmds.addCommand( 'default ip ospf neighbor bfd' )

   if entity.cost != entity.costDefault:
      cmds.addCommand( 'ip ospf cost %d' % entity.cost )
   elif entity.metricProfileName:
      cmds.addCommand( 'ip ospf cost profile %s' % entity.metricProfileName )
   elif saveAll:
      cmds.addCommand( 'no ip ospf cost' )

   if entity.routerDeadInterval != entity.routerDeadIntervalDefault or saveAll:
      cmds.addCommand( 'ip ospf dead-interval %d' % entity.routerDeadInterval )
      
   if entity.helloInterval != entity.helloIntervalDefault or saveAll:
      cmds.addCommand( 'ip ospf hello-interval %d' % entity.helloInterval )
      
   if entity.priority != entity.priorityDefault or saveAll:
      cmds.addCommand( 'ip ospf priority %d' % entity.priority )
      
   if entity.rxInt != entity.rxIntDefault or saveAll:
      cmds.addCommand( 'ip ospf retransmit-interval %d' % entity.rxInt )
      
   if not entity.enable:
      cmds.addCommand( 'ip ospf disabled' )
   elif saveAll:
      cmds.addCommand( 'no ip ospf disabled' )
      
   if entity.transDelay != entity.transDelayDefault or saveAll:
      cmds.addCommand( 'ip ospf transmit-delay %d' % entity.transDelay )

   if entity.intfType != entity.intfTypeDefault:
      cmds.addCommand( 'ip ospf network %s' % {
         'intfTypePointToPoint': 'point-to-point',
         'intfTypePointToMultipoint': 'point-to-multipoint',
         'intfTypeNonBroadcast': 'nonbroadcast' }[ entity.intfType ] )

   authType = entity.authType
   if authType == 'ospfAuthTypeSimple':
      cmds.addCommand( 'ip ospf authentication' )
   elif authType == 'ospfAuthTypeMd':
      cmds.addCommand( 'ip ospf authentication message-digest' )
   elif saveAll:
      cmds.addCommand( 'no ip ospf authentication' )

   securityConfig = requireMounts[ mgmtSecurityConfigPath ]
   if entity.simpleAuth:
      myKey = entity.name + '_passwd'
      cmd = ReversibleSecretCli.getCliSaveCommand( 'ip ospf authentication-key {}',
                                                   securityConfig,
                                                   entity.simpleAuth,
                                                   uniqueKey=myKey,
                                                   algorithm='DES' )
      cmds.addCommand( cmd )

   if entity.mtuIgnore:
      cmds.addCommand( 'ip ospf mtu-ignore' )
   elif saveAll:
      cmds.addCommand( 'no ip ospf mtu-ignore' )

   if entity.areaIdPresent != entity.areaIdPresentDefault:
      cmds.addCommand( 'ip ospf area %s' % entity.areaId )
   
   # process message digest / cryptographic auth keys
   for keyId in sorted( entity.mdAuth ):
      authAlgorithm = entity.mdAuth[ keyId ].authAlgo
      myKey = entity.name + '_%sKey_%d' % ( authAlgorithm, keyId )
      cmd = ReversibleSecretCli.getCliSaveCommand(
         f'ip ospf message-digest-key {keyId:d} {authAlgorithm} {{}}',
         securityConfig,
         entity.mdAuth[ keyId ].authKey,
         uniqueKey=myKey,
         algorithm='DES' )
      cmds.addCommand( cmd )
   if Toggles.gatedToggleLib.toggleOspfTilfaEnabled():   
      if entity.protectionModeTiLfa != entity.modeDefault:
         configProtection = entity.protectionModeTiLfa
         cmd = getProtectionModeCmd( configProtection , True )
         cmds.addCommand( cmd )
      elif saveAll:
         cmds.addCommand( "no ip ospf fast-reroute ti-lfa mode" )
   
   

def _saveInstanceConfig( entity, root, ospfConfig, requireMounts, saveAll ):
   mode = root[ RouterOspfConfigMode ].getOrCreateModeInstance( ( entity.instance,
                                                                  entity.vrfName ) )
   cmds = mode[ 'Ospf.instance' ]

   if entity.routerId != entity.routerIdDefault:
      cmds.addCommand( 'router-id %s' % entity.routerId )
   elif saveAll:
      cmds.addCommand( 'no router-id' )

   if not entity.enable:
      cmds.addCommand( 'shutdown' )
   elif saveAll:
      cmds.addCommand( 'no shutdown' )

   if entity.refBw != entity.refBwDefault:
      cmds.addCommand( 'auto-cost reference-bandwidth %s' % entity.refBw )
   elif saveAll:
      cmds.addCommand( 'no auto-cost reference-bandwidth' )

   if entity.bfdEnabledState:
      cmds.addCommand( 'bfd default' )
   elif saveAll:
      cmds.addCommand( 'no bfd default' )

   if entity.bfdAnyState:
      cmds.addCommand( 'bfd adjacency state any' )
   elif saveAll:
      cmds.addCommand( 'no bfd adjacency state any' )

   if entity.unnumberedHelloMaskTx:
      cmds.addCommand( 'interface unnumbered hello mask tx 0.0.0.0' )
   elif saveAll:
      cmds.addCommand( 'no interface unnumbered hello mask tx 0.0.0.0' )

   if entity.internalPref != entity.internalPrefDefault or saveAll:
      cmds.addCommand( 'distance ospf intra-area %d' % entity.internalPref )

   if entity.asePref != entity.asePrefDefault or saveAll:
      cmds.addCommand( 'distance ospf external %d' % entity.asePref )

   if entity.interPref != entity.interPrefDefault or saveAll:
      cmds.addCommand( 'distance ospf inter-area %d' % entity.interPref )

   if entity.passiveByDefault:
      cmds.addCommand( 'passive-interface default' )
   
   instanceId = entity.instance
   # The passive-interface config is actually in PassiveIntfConfig.
   for intfName in Arnet.sortIntf( ospfConfig.passiveIntfConfig ):
      passiveIntfConfig = ospfConfig.passiveIntfConfig[ intfName ]
      if instanceId not in passiveIntfConfig.passiveInstances:
         continue
      if passiveIntfConfig.passiveInstances[ instanceId ] == 'ospfIntfPassive':
         cmds.addCommand( 'passive-interface %s' % intfName )
      elif passiveIntfConfig.passiveInstances[ instanceId ] == 'ospfIntfActive':
         cmds.addCommand( 'no passive-interface %s' % intfName )

   for protocol in sorted( entity.redistributeConfig.values(), 
                        key=lambda p: p.proto ):
      redistOspf = protocol.proto in ( 'protoOspf', 'protoOspfAse', 'protoOspfNssa' )
      command = 'redistribute %s' % \
                  ( protoDict[ protocol.proto ] if not redistOspf else 'ospf' )
      # isisLevel will be set to either level1/level2/level1_2 for Isis
      # It will be set to levelNone for all other protocols
      if redistOspf:
         if protocol.allowOspfInstance:
            command += ' instance'
            command += ' include leaked' if protocol.includeLeaked else ''
         else:
            assert protocol.includeLeaked
            command += ' leaked'
         if protocol.matchRouteType != '':
            command += ' match %s' % protocol.matchRouteType
      elif protocol.includeLeaked:
         command += ' include leaked'
      if protocol.isisLevel != 'levelNone':
         command += ' %s' % ISIS_LEVEL_MAP[ protocol.isisLevel ]
      if protocol.routeMap:
         command += ' route-map %s' % protocol.routeMap       
      cmds.addCommand( command )

   if entity.distListIn.distType == "distRouteMap":
      command = "distribute-list route-map %s in" % (
         entity.distListIn.distName )
      cmds.addCommand( command )
   elif entity.distListIn.distType == "distPrefixList":
      command = "distribute-list prefix-list %s in" % (
         entity.distListIn.distName )
      cmds.addCommand( command )

   # XXXXX TODO handle auth attributes

   for area in sorted( entity.areaConfig, key=IpAddress ):
      areaConfig = entity.areaConfig[ area ]

      if areaConfig.defaultMetric != areaConfig.defaultMetricDefault or saveAll:
         cmds.addCommand( 'area %s default-cost %d' %
                          ( area, areaConfig.defaultMetric ) )

      if areaConfig.areaType in { 'areaTypeNssa', 'areaTypeNssaNoSummary' }:
         cmd = 'area %s nssa' % area
         if not areaConfig.nssaTranslateAlways and not areaConfig.nssaOnly and \
            areaConfig.nssaDefInfoOrigin != 'ospfDefInfoOriginEnable' and \
            areaConfig.areaType != 'areaTypeNssaNoSummary':
            cmds.addCommand( cmd )
         if areaConfig.areaType == 'areaTypeNssaNoSummary':
            cmdNoSummary = cmd + ' no-summary'
            cmds.addCommand( cmdNoSummary )
         if areaConfig.nssaTranslateAlways:
            cmd1 = "area %s not-so-stubby lsa type-7 convert type-5" % area
            cmds.addCommand( cmd1 )
         if areaConfig.nssaOnly:
            cmd2 = cmd + ' nssa-only'
            cmds.addCommand( cmd2 )
         if areaConfig.nssaDefInfoOrigin == 'ospfDefInfoOriginEnable':
            cmd3 = cmd + ' default-information-originate'
            if areaConfig.nssaDefInfoMetric != areaConfig.nssaDefInfoMetricDefault \
               or saveAll:
               cmd3 += ' metric %d' % areaConfig.nssaDefInfoMetric
            if areaConfig.nssaDefInfoMetricType != \
               areaConfig.nssaDefInfoMetricTypeDefault or saveAll:
               cmd3 += ' metric-type %d' % \
               int( areaConfig.nssaDefInfoMetricType.lstrip( "extMetricType" ) )
            if ( ( areaConfig.nssaDefInfoTag != areaConfig.nssaDefInfoTagDefault
               or saveAll )
               and Toggles.OspfToggleLib.toggleOspfDIOTagSetEnabled() ):
               cmd3 += ' tag %d' % areaConfig.nssaDefInfoTag
            if areaConfig.nssaDefInfoNssaOnly != \
               areaConfig.nssaDefInfoNssaOnlyDefault: 
               cmd3 += ' nssa-only'
            cmds.addCommand( cmd3 )
         elif areaConfig.nssaDefInfoOrigin == 'ospfDefInfoOriginSuppress':
            cmds.addCommand( 'no ' + cmd + ' default-information-originate' )
      elif areaConfig.areaType == 'areaTypeStub':
         cmds.addCommand( 'area %s stub' % area )
      elif areaConfig.areaType == 'areaTypeStubNoSummary':
         cmds.addCommand( 'area %s stub no-summary' % area )
      for prefix in sorted( areaConfig.summaryFilter,
                            key=lambda p: ( IpAddress( p.address ), int( p.len ) ) ):
         # BUG25925
         cmds.addCommand( f'area {area} filter {prefix.stringValue}' )

      if areaConfig.summaryPrefixList:
         cmds.addCommand( 'area %s filter prefix-list %s'
                          % ( area, areaConfig.summaryPrefixList ) )

      for prefix in sorted( areaConfig.networkList,
                            key=lambda p: ( IpAddress( p.address ), int( p.len ) ) ):
         # BUG25925
         cmd = f'area {area} range {prefix.stringValue}'
         if areaConfig.networkList[ prefix ].restricted:
            cmd += ' not-advertise'
         elif saveAll:
            cmd += ' advertise'
         if areaConfig.networkList[ prefix ].cost != 0:
            cmd += ' cost %d' % areaConfig.networkList[ prefix ].cost

         cmds.addCommand( cmd )

      if Toggles.RoutingLibToggleLib.toggleOspfShamLinkEnabled():
         for destinationEp in sorted ( areaConfig.shamLink, key=IpAddress ):
            shamLink = areaConfig.shamLink[ destinationEp ]
            shamLinkMode = mode[ RouterOspfAreaShamLinkConfigMode ].\
               getOrCreateModeInstance( ( entity.vrfName, area,\
                                          shamLink.destinationEp ) )
            policyCmds = shamLinkMode[ 'Ospf.shamLink' ]

            if shamLink.sourceEp != shamLink.sourceEpDefault:
               policyCmds.addCommand( "source %s" % shamLink.sourceEp )
            elif saveAll:
               policyCmds.addCommand( 'no source' )
            if shamLink.cost != shamLink.costDefault:
               policyCmds.addCommand( 'cost %d' % shamLink.cost )
            elif saveAll:
               policyCmds.addCommand( 'no cost' )
            if shamLink.helloInterval != shamLink.helloIntervalDefault:
               policyCmds.addCommand( 'hello-interval %d' % shamLink.helloInterval )
            elif saveAll:
               policyCmds.addCommand( 'no hello-interval' )
            if shamLink.routerDeadInterval != shamLink.routerDeadIntervalDefault:
               policyCmds.addCommand( 'dead-interval %d' % \
                     shamLink.routerDeadInterval )
            elif saveAll:
               policyCmds.addCommand( 'no dead-interval' )
            if shamLink.retransmitInterval != shamLink.retransmitIntervalDefault:
               policyCmds.addCommand( 'retransmit-interval %d' % \
                     shamLink.retransmitInterval )
            elif saveAll:
               policyCmds.addCommand( 'no retransmit-interval' )
            if shamLink.transDelay != shamLink.transDelayDefault:
               policyCmds.addCommand( 'transmit-delay %d' % shamLink.transDelay )
            elif saveAll:
               policyCmds.addCommand( 'no transmit-delay' )
            if shamLink.mtuIgnore:
               policyCmds.addCommand( 'mtu-ignore' )
            elif saveAll:
               policyCmds.addCommand( 'no mtu-ignore' )
            if shamLink.disabled:
               policyCmds.addCommand( 'disabled' )
            elif saveAll:
               policyCmds.addCommand( 'no disabled' )

   if entity.rfc1583Compat:
      cmds.addCommand( 'compatible rfc1583' )

   if Toggles.gatedToggleLib.toggleOspfAbrSummaryNonBackboneIgnoreEnabled():
      if entity.spfAbrSumLsaNonBackboneIgnore:
         cmds.addCommand( 'spf abr summary-lsa non-backbone ignore' )
      elif saveAll:
         cmds.addCommand( 'no spf abr summary-lsa non-backbone ignore' )

   prefixes = sorted( entity.networkArea, 
                key=lambda p: ( IpAddress( p.address ), int( p.len ) ) )
   for prefix in prefixes:
      areaConfig = entity.networkArea.get( prefix )
      if ( areaConfig is not None and \
            areaConfig == entity.areaConfig.get( areaConfig.area ) ):
         # TODO .stringValue should not be necessary, BUG25925
         cmds.addCommand( 'network %s area %s' \
                             % ( prefix.stringValue, areaConfig.area ) )

   # Always add the max-lsa command, to give its default value extra
   # visibility.
   cmd = 'max-lsa %d' % entity.maxLsa
   if entity.maxLsaThreshold != entity.maxLsaThresholdDefault or saveAll:
      cmd += ' %d' % entity.maxLsaThreshold
   if entity.maxLsaWarningOnly != entity.maxLsaWarningOnlyDefault:
      cmd += ' warning-only'
   else:
      if entity.maxLsaIgnoreTime != entity.maxLsaIgnoreTimeDefault or saveAll:
         cmd += ' ignore-time %d' % entity.maxLsaIgnoreTime
      if entity.maxLsaIgnoreCount != entity.maxLsaIgnoreCountDefault or saveAll:
         cmd += ' ignore-count %d' % entity.maxLsaIgnoreCount
      if entity.maxLsaResetTime != entity.maxLsaResetTimeDefault or saveAll:
         cmd += ' reset-time %d' % entity.maxLsaResetTime
   
   cmds.addCommand( cmd )

   if entity.simultaneousBringUp != entity.simultaneousBringUpDefault:
      cmds.addCommand( 'adjacency exchange-start threshold %d'
                       % entity.simultaneousBringUp )
   elif saveAll:
      cmds.addCommand( 'adjacency exchange-start threshold %d'
                       % entity.simultaneousBringUpDefault )
   if entity.lsaRetransmissionThreshold != entity.lsaRetransmissionThresholdDefault \
      or saveAll:
      cmds.addCommand( 'retransmission-threshold lsa %d'
                       % entity.lsaRetransmissionThreshold )
   if entity.adjacencyLogging != entity.adjacencyLoggingDefault or saveAll:
      cmd = 'log-adjacency-changes'
      if entity.adjacencyLogging == 'adjacencyLoggingTypeDetail':
         cmd += ' detail'
      elif entity.adjacencyLogging == 'adjacencyLoggingTypeNone':
         cmd = 'no ' + cmd
      cmds.addCommand( cmd )
 
   # print out command in the timers spf format if timerSpfCommandActive
   # else we print the command in timers [spf delay initial|throttle spf] format
   timerConfig = entity.spfThrottleTimerConfig
   if timerConfig.spfStartInt != entity.spfStartIntDefault or \
         timerConfig.spfHoldInt != entity.spfHoldIntDefault or \
         timerConfig.spfMaxWaitInt != entity.spfMaxWaitIntDefault:
      if entity.timerSpfCommandActive:
         holdIntSec = timerConfig.spfHoldInt // 1000 
         cmd = 'timers spf %d' % holdIntSec
      else:
         cmd = 'timers spf delay initial %d %d %d' % ( timerConfig.spfStartInt,
                                                       timerConfig.spfHoldInt,
                                                       timerConfig.spfMaxWaitInt )
      cmds.addCommand( cmd )
   elif saveAll:
      if entity.timerSpfCommandActive:
         holdIntSec = timerConfig.spfHoldInt // 1000 
         cmd = 'timers spf %d' % holdIntSec
      else:
         cmd = 'timers spf delay initial %d %d %d' % ( timerConfig.spfStartInt,
                                                       timerConfig.spfHoldInt,
                                                       timerConfig.spfMaxWaitInt )
      cmds.addCommand( cmd )

   # timers lsa rx min interval
   if entity.lsaArrivalInt != entity.lsaArrivalIntDefault:
      cmd = 'timers lsa rx min interval %d' % ( entity.lsaArrivalInt )
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'timers lsa rx min interval %d' % ( entity.lsaArrivalInt )
      cmds.addCommand( cmd )

   # timers lsa tx delay initial 
   timerConfig = entity.lsaThrottleTimerConfig
   if timerConfig.lsaStartInt != entity.lsaStartIntDefault or \
         timerConfig.lsaHoldInt != entity.lsaHoldIntDefault or \
         timerConfig.lsaMaxWaitInt != entity.lsaMaxWaitIntDefault:
      cmdStr = 'timers lsa tx delay initial %d %d %d'
      cmd = cmdStr % ( timerConfig.lsaStartInt, timerConfig.lsaHoldInt,
                       timerConfig.lsaMaxWaitInt )
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'timers lsa tx delay initial %d %d %d' % ( timerConfig.lsaStartInt,
                                 timerConfig.lsaHoldInt, timerConfig.lsaMaxWaitInt )
      cmds.addCommand( cmd )

   # timers out-delay <lsa out-delay>
   if entity.lsaOutDelayTimerConfig != entity.lsaOutDelayTimerDefault:
      cmd = "timers out-delay %d" % entity.lsaOutDelayTimerConfig
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = "no timers out-delay"
      cmds.addCommand( cmd )
 
   if entity.maxEcmp > 0:
      cmd = 'maximum-paths %d' % entity.maxEcmp
      cmds.addCommand( cmd )
   elif saveAll:
      routingHwStatus = requireMounts[ 'routing/hardware/status' ]
      if routingHwStatus and routingHwStatus.maxLogicalProtocolEcmp > 0:
         cmd = 'maximum-paths %d' % routingHwStatus.maxLogicalProtocolEcmp
         cmds.addCommand( cmd )

   if entity.floodPacing != entity.floodPacingDefault:
      cmds.addCommand( 'timers pacing flood %d' % entity.floodPacing )
   elif saveAll:
      cmd = "no timers pacing flood"
      cmds.addCommand( cmd )

   maxMetric = entity.maxMetricConfig
   cmd = ''
   if maxMetric.maxMetricFlag == entity.maxMetricFlagDefault and saveAll:
      cmd = 'no max-metric router-lsa'
   else:
      if maxMetric.maxMetricFlag != entity.maxMetricFlagDefault:
         cmd = 'max-metric router-lsa'
      if maxMetric.maxMetricExtMetric != entity.maxMetricExtMetricDefault:
         cmd += ' external-lsa'
         if maxMetric.maxMetricExtMetric != entity.maxMetricLsaMetricDefault or \
               saveAll:
            cmd += ' %d' % maxMetric.maxMetricExtMetric
      if maxMetric.maxMetricStubFlag != entity.maxMetricStubFlagDefault:
         cmd += ' include-stub'
      if maxMetric.maxMetricOnStartDuration != \
            entity.maxMetricOnStartDurationDefault:
         cmd += ' on-startup %d' % maxMetric.maxMetricOnStartDuration
      if maxMetric.maxMetricWaitBgpFlag != entity.maxMetricWaitBgpFlagDefault:
         cmd += ' on-startup wait-for-bgp'
      if maxMetric.maxMetricSumMetric != entity.maxMetricSumMetricDefault:
         cmd += ' summary-lsa'
         if maxMetric.maxMetricSumMetric != entity.maxMetricLsaMetricDefault or \
               saveAll:
            cmd += ' %d' % maxMetric.maxMetricSumMetric
   if cmd:
      cmds.addCommand( cmd )
   
   if entity.pointToPointRoutes is not True:
      cmds.addCommand( 'no point-to-point routes' )
   elif saveAll:
      cmds.addCommand( 'point-to-point routes' )

   if entity.tunnelRoutes:
      cmds.addCommand( 'tunnel routes' )
   elif saveAll:
      cmds.addCommand( 'no tunnel routes' )

   if entity.defInfoOrigin == 'ospfDefInfoOriginEnable':
      cmd = 'default-information originate'
      if entity.defInfoOriginAlways != entity.defInfoOriginAlwaysDefault:
         cmd += ' always'
      if entity.defInfoMetric != entity.defInfoMetricDefault or saveAll:
         cmd += ' metric %d' % entity.defInfoMetric
      if entity.defInfoMetricType != entity.defInfoMetricTypeDefault or saveAll:
         cmd += ' metric-type %d' % \
               int( entity.defInfoMetricType.lstrip( "extMetricType" ) )
      if ( ( entity.defInfoTag != entity.defInfoTagDefault or saveAll )
         and Toggles.OspfToggleLib.toggleOspfDIOTagSetEnabled() ):
         cmd += ' tag %d' % entity.defInfoTag
      if entity.defInfoRouteMap != entity.defInfoRouteMapDefault:
         cmd += ' route-map %s' % entity.defInfoRouteMap
      cmds.addCommand( cmd )
   elif entity.defInfoOrigin == 'ospfDefInfoOriginSuppress':
      cmds.addCommand( 'no default-information originate' )

   # summary-address
   for summAddr in sorted( entity.summaryAddrConfig,
                           key=lambda p: ( IpAddress( p.address ), int( p.len ) ) ):
      summCfg = entity.summaryAddrConfig.get( summAddr )
      if summCfg is not None:
         cmd = 'summary-address %s' % summCfg.prefix
         if summCfg.notAdvertise:
            cmd = cmd + ' not-advertise'
         elif summCfg.tag:
            cmd = cmd + ' tag %d' % summCfg.tag
         elif summCfg.attrMap:
            cmd = cmd + ' attribute-map %s' % summCfg.attrMap
         cmds.addCommand( cmd )

   if entity.gracefulRestart != entity.gracefulRestartDefault:
      cmdDisplayed = False
      if entity.grGracePeriod != entity.grGracePeriodDefault:
         cmds.addCommand( 'graceful-restart grace-period %d' %
                          ( entity.grGracePeriod ) )
         cmdDisplayed = True
      elif saveAll:
         cmds.addCommand( 'graceful-restart grace-period %d' %
                          ( entity.grGracePeriodDefault ) )
         cmdDisplayed = True
      if entity.grPlannedOnly != entity.grPlannedOnlyDefault:
         cmds.addCommand( 'graceful-restart planned-only' )
         cmdDisplayed = True
      if not cmdDisplayed:
         cmds.addCommand( 'graceful-restart' )
   elif saveAll:
      cmds.addCommand( 'no graceful-restart' )

   if entity.mplsLdpSync != entity.mplsLdpSyncDefault:
      cmds.addCommand( 'mpls ldp sync default' )
   if entity.grHelper != entity.grHelperDefault:
      cmds.addCommand( 'no graceful-restart-helper' )
   elif saveAll:
      cmds.addCommand( 'no mpls ldp sync default' )
      cmds.addCommand( 'graceful-restart-helper' )

   aclVrfName = entity.vrfName
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   serviceAclVrfConfig = aclCpConfig.cpConfig[ 'ip' ].serviceAcl.get( aclVrfName )
   if ( serviceAclVrfConfig and 
        OspfConsts.serviceName in serviceAclVrfConfig.service and
        serviceAclVrfConfig.service[ OspfConsts.serviceName ].aclName != '' ):
      cmds.addCommand( 'ip access-group %s' % 
            serviceAclVrfConfig.service[ OspfConsts.serviceName ].aclName )
   elif saveAll:
      cmds.addCommand( 'no ip access-group' )

   if Toggles.OspfToggleLib.toggleOspfDnBitHonorDefaultVrfEnabled():
      if entity.vrfName == DEFAULT_VRF:
         if entity.ignoreDnBit != entity.ignoreDnBitDefault:
            cmds.addCommand( 'dn-bit-ignore disabled' )
         elif saveAll:
            cmds.addCommand( 'no dn-bit-ignore disabled' )
      else:
         if entity.ignoreDnBit != entity.ignoreDnBitDefault:
            if entity.backwardCompatibilityIgnoreDnBitType5Type7 != \
               entity.backwardCompatibilityIgnoreDnBitType5Type7Default:
               cmds.addCommand( 'dn-bit-ignore lsa type-5 type-7' )
            else:
               cmds.addCommand( 'dn-bit-ignore' )
         elif saveAll:
            cmds.addCommand( 'no dn-bit-ignore' )
   else:
      if entity.ignoreDnBit != entity.ignoreDnBitDefault:
         if entity.backwardCompatibilityIgnoreDnBitType5Type7 != \
            entity.backwardCompatibilityIgnoreDnBitType5Type7Default:
            cmds.addCommand( 'dn-bit-ignore lsa type-5 type-7' )
         else:
            cmds.addCommand( 'dn-bit-ignore' )
      elif saveAll:
         cmds.addCommand( 'no dn-bit-ignore' )

   if entity.intraAreaRouteEcmp != entity.intraAreaRouteEcmpDefault:
      cmds.addCommand( 'ecmp intra-area route multi-area next-hops' )
   elif saveAll:
      cmds.addCommand( 'no ecmp intra-area route multi-area next-hops' )
   if entity.countersRatePeriodDecay != \
            entity.countersRatePeriodDecayDefault:
      cmd = "counters rate period decay %d" % entity.countersRatePeriodDecay
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = "no counters rate period decay"
      cmds.addCommand( cmd )

   if entity.ospfTeMode != entity.ospfTeModeDefault:
      saveInstanceTeConfig( entity, mode, saveAll )
   elif saveAll and entity.vrfName == DEFAULT_VRF:
      # Don't do this in the non default vrf because we have a guard error
      cmds.addCommand( 'no traffic-engineering' )

   srDataPlaneEnum = Tac.Type( 'Routing::Ospf::SrDataPlane' )
   srConfig = entity.srConfig
   if srConfig and srConfig.dataPlane == srDataPlaneEnum.srDataPlaneMpls:
      saveInstanceSrMplsConfig( entity, mode, saveAll )
   elif saveAll and entity.vrfName == DEFAULT_VRF:
      # Don't do this in the non default vrf because we have a guard error
      cmds.addCommand( 'no segment-routing mpls' )

   if entity.primaryDomainId != entity.domainIdDefault:
      cmds.addCommand( 'domain identifier %s primary' %
                       getDomainIdStr( entity.primaryDomainId ) )
   for secondaryId in sorted( entity.secondaryDomainId ):
      cmds.addCommand( 'domain identifier %s' % getDomainIdStr( secondaryId ) )
   
   if Toggles.gatedToggleLib.toggleOspfTilfaEnabled():   
      if entity.protectionModeTiLfa != entity.modeDefault:
         configProtection = entity.protectionModeTiLfa
         cmd = getProtectionModeCmd( configProtection )
         cmds.addCommand( cmd )
      elif saveAll:
         cmds.addCommand( 'no fast-reroute ti-lfa mode' )
      if entity.microLoopConvergenceDelay:
         cmd = 'timers local-convergence-delay'
         cmd += ' %d' % entity.microLoopConvergenceDelay
         cmd += ' protected-prefixes'
         cmds.addCommand( cmd )
      elif saveAll:
         cmds.addCommand( 'no timers local-convergence-delay' )

def saveGeneralConfig( ospfConfig, root, saveAll ):
   mode = root[ RouterOspfGeneralConfigMode ].getSingletonInstance()
   cmds = mode[ 'Ospf.general' ]

   generalConfig = ospfConfig.generalConfig
   if not generalConfig:
      return

   if generalConfig.dscp != generalConfig.dscpDefault or saveAll:
      cmds.addCommand( 'qos dscp %d' % generalConfig.dscp )

def saveInstanceTeConfig( instanceConfig, root, saveAll ):
   mode = root[ RouterOspfTeConfigMode ].getOrCreateModeInstance(
      instanceConfig.instance )
   cmds = mode[ 'Ospf.te' ]

   if instanceConfig.ospfTeEnabled != instanceConfig.ospfTeEnabledDefault:
      cmds.addCommand( 'no shutdown' )
   elif saveAll:
      cmds.addCommand( 'shutdown' )

   if ( instanceConfig.ospfTeConfiguredAllAreas !=
        instanceConfig.ospfTeConfiguredAllAreasDefault ):
      cmds.addCommand( 'no area all' )
   elif saveAll:
      cmds.addCommand( 'area all' )

   areaIdList = []
   for area in sorted( instanceConfig.areaConfig, key=IpAddress ):
      areaConfig = instanceConfig.areaConfig[ area ]
      if areaConfig.teConfigured != areaConfig.teConfiguredDefault:
         areaIdList.append( area )

   if areaIdList:
      cmds.addCommand( 'area %s' % ' '.join( areaIdList ) )

def saveInstanceSrMplsConfig( instanceConfig, root, saveAll ):
   mode = root[ RouterOspfSrMplsConfigMode ].getOrCreateModeInstance(
      instanceConfig.instance )
   cmds = mode[ 'Ospf.srMpls' ]

   srConfig = instanceConfig.srConfig
   if srConfig.shutdown != srConfig.shutdownDefault:
      cmds.addCommand( 'no shutdown' )
   elif saveAll:
      cmds.addCommand( 'shutdown' )

   proxyNodeSegmentList = []
   prefixSegmentList = []
   for prefixSegment in instanceConfig.srConfig.prefixSegments.values():
      if prefixSegment.isProxyNode is True:
         proxyNodeSegmentList.append( [ prefixSegment.prefix,
                                        prefixSegment.index ] )
      else:
         prefixSegmentList.append( [ prefixSegment.prefix,
                                     prefixSegment.index ] )

   for entry in proxyNodeSegmentList:
      cmds.addCommand( 'proxy-node-segment %s index %d' % ( entry[ 0 ],
                                                            entry[ 1 ] ) )

   for entry in prefixSegmentList:
      cmds.addCommand( 'prefix-segment %s index %d' % ( entry[ 0 ],
                                                        entry[ 1 ] ) )

   srAdjAllocationOptions = Tac.Type(
      "Routing::SegmentRoutingCli::SrAdjAllocationType" )
   if instanceConfig.srConfig.srAdjSegmentAlloc == \
          srAdjAllocationOptions.srAdjacencyAllocationNone:
      cmds.addCommand( 'adjacency-segment allocation none' )
   elif instanceConfig.srConfig.srAdjSegmentAlloc == \
          srAdjAllocationOptions.srAdjacencyAllocationAll:
      cmds.addCommand( 'adjacency-segment allocation all-interfaces' )
   elif saveAll:
      cmds.addCommand( 'adjacency-segment allocation sr-peers' )

#-------------------------------------------------------------------------------
# save config under global config mode
#-------------------------------------------------------------------------------

CliSave.GlobalConfigMode.addCommandSequence( 'Ip.Ospf',
                                             after=[ RouterOspfConfigMode ] )

@CliSave.saver( 'Routing::Ospf::Config', 'routing/ospf/config',
                requireMounts=( 'routing/hardware/statuscommon',
                                'routing/hardware/status',
                                'interface/config/all', 'interface/status/all',
                                'acl/cpconfig/cli', mgmtSecurityConfigPath ) )
def saveConfig( ospfConfig, root, requireMounts, options ):
   # Check if any Ospf configuration is made.
   ospfInstanceConfigured = len( ospfConfig.instanceConfig ) > 0
   ospfInterfacesConfigured = len( ospfConfig.intfConfig ) > 0
   ospfNameLookupConfigured = ospfConfig.nameLookup != ospfConfig.nameLookupDefault
   ospfGeneralConfigured = ospfConfig.generalConfig is not None

   ospfConfigEnabled = ( ospfInstanceConfigured or ospfInterfacesConfigured or \
                         ospfNameLookupConfigured or ospfGeneralConfigured )

   # No need to show ospf configs if nothing is configured
   if not ospfConfigEnabled and not options.saveAllDetail:
      return

   if ospfGeneralConfigured and ( options.saveAll or
                                  ospfConfig.generalConfig.dscp !=
                                    ospfConfig.generalConfig.dscpDefault ):
      saveGeneralConfig( ospfConfig, root, options.saveAll )

   if ospfConfig.nameLookup != ospfConfig.nameLookupDefault:
      root[ 'Ip.Ospf' ].addCommand( 'ip ospf router-id output-format hostnames' )
   elif options.saveAll:
      root[ 'Ip.Ospf' ].addCommand( 'no ip ospf router-id output-format hostnames' )
 
   for instanceId in sorted( ospfConfig.instanceConfig, key=int ):
      _saveInstanceConfig( ospfConfig.instanceConfig[ instanceId ],
                           root, ospfConfig, requireMounts, options.saveAll )

   if options.saveAllDetail:
      intfs = allIpIntfNames( requireMounts, includeEligible=True )
   elif options.saveAll:
      # Display Ospf configuration on routing protocol interfaces as well as 
      # switchports present in intfConfig collection.
      intfs = set( allRoutingProtocolIntfNames( requireMounts ) )
      intfs.update( ospfConfig.intfConfig )
   else:
      intfs = ospfConfig.intfConfig
   
   # save config for each interface
   for intfName in intfs:
      if isManagement(intfName): 
         continue

      intfConfig = ospfConfig.intfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Routing::Ospf::IntfConfig', intfName )
         else:
            continue
      saveIntfConfig( intfConfig, root, requireMounts, options )
