# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode.RsvpLer import (
      RsvpTeMode,
      RsvpP2mpLeafMode,
      RsvpPathSpecMode,
      RsvpTunnelProfileMode,
      RsvpTunnelSpecMode,
      RsvpP2mpTunnelProfileMode,
      RsvpP2mpTunnelSpecMode,
      )
import CliSave
from CliSavePlugin.TeCliSave import RouterGlobalTeMode
import Tac
from TypeFuture import TacLazyType

RsvpLerCliConfig = TacLazyType( 'Rsvp::RsvpLerCliConfig' )
RsvpLerConstants = TacLazyType( 'Rsvp::RsvpLerConstants' )
RsvpLerLspSelfPingMode = TacLazyType( 'Rsvp::RsvpLerLspSelfPingMode' )
RsvpLerPathSpecType = TacLazyType( 'Rsvp::RsvpLerPathSpecType' )
RsvpSessionType = TacLazyType( 'Rsvp::RsvpSessionType' )
lspSelfPingNone = RsvpLerLspSelfPingMode.lspSelfPingNone
lspSelfPingEnabled = RsvpLerLspSelfPingMode.lspSelfPingEnabled
pathSpecDynamicType = RsvpLerPathSpecType.pathSpecDynamicType
pathSpecExplicitType = RsvpLerPathSpecType.pathSpecExplicitType
p2mpLspTunnel = RsvpSessionType.p2mpLspTunnel
p2pLspTunnel = RsvpSessionType.p2pLspTunnel


class RsvpTeConfigMode( RsvpTeMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpTeMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterGlobalTeMode.addChildMode( RsvpTeConfigMode )
RsvpTeConfigMode.addCommandSequence( 'RsvpLer.config' )

# pylint: disable-msg=W0223, abstract-method
class RsvpP2pPathSpecConfigMode( RsvpPathSpecMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpPathSpecMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RsvpP2mpPathSpecConfigMode( RsvpPathSpecMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpPathSpecMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RsvpLeafConfigMode( RsvpP2mpLeafMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpP2mpLeafMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )


class RsvpP2pTunnelSpecConfigMode( RsvpTunnelSpecMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpTunnelSpecMode.__init__( self, param, )
      CliSave.Mode.__init__( self, param )

class RsvpP2mpTunnelSpecConfigMode( RsvpP2mpTunnelSpecMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpP2mpTunnelSpecMode.__init__( self, param, )
      CliSave.Mode.__init__( self, param )

class RsvpP2pTunnelProfileConfigMode( RsvpTunnelProfileMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpTunnelProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RsvpP2mpTunnelProfileConfigMode( RsvpP2mpTunnelProfileMode, CliSave.Mode ):
   def __init__( self, param ):
      RsvpP2mpTunnelProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )


RsvpTeConfigMode.addChildMode( RsvpP2mpPathSpecConfigMode,
                               before=[ RsvpP2pPathSpecConfigMode ] )
RsvpTeConfigMode.addChildMode( RsvpP2pPathSpecConfigMode,
                              after=[ RsvpP2mpPathSpecConfigMode ],
                               before=[ RsvpP2mpTunnelProfileConfigMode ] )
RsvpP2mpPathSpecConfigMode.addChildMode( RsvpLeafConfigMode )
RsvpTeConfigMode.addChildMode( RsvpP2mpTunnelProfileConfigMode,
                              after=[ RsvpP2pPathSpecConfigMode ],
                               before=[ RsvpP2pTunnelProfileConfigMode ] )
RsvpTeConfigMode.addChildMode( RsvpP2pTunnelProfileConfigMode,
                              after=[ RsvpP2mpTunnelProfileConfigMode ],
                               before=[ RsvpP2mpTunnelSpecConfigMode ] )
RsvpTeConfigMode.addChildMode( RsvpP2mpTunnelSpecConfigMode,
                              after=[ RsvpP2pTunnelProfileConfigMode ],
                               before=[ RsvpP2pTunnelSpecConfigMode ] )
RsvpTeConfigMode.addChildMode( RsvpP2pTunnelSpecConfigMode,
                              after=[ RsvpP2mpTunnelSpecConfigMode ] )

RsvpP2pPathSpecConfigMode.addCommandSequence( 'RsvpLer.p2pPathSpecConfig' )
RsvpP2mpPathSpecConfigMode.addCommandSequence( 'RsvpLer.p2mpPathSpecConfig' )
RsvpLeafConfigMode.addCommandSequence( 'RsvpLer.p2mpPathSpecConfig.leaf' )
RsvpP2pTunnelSpecConfigMode.addCommandSequence( 'RsvpLer.p2pTunnelSpecConfig' )
RsvpP2mpTunnelSpecConfigMode.addCommandSequence( 'RsvpLer.p2mpTunnelSpecConfig' )
RsvpP2pTunnelProfileConfigMode.addCommandSequence( 'RsvpLer.p2pTunnelProfileConfig' )
RsvpP2mpTunnelProfileConfigMode.addCommandSequence(
   'RsvpLer.p2mpTunnelProfileConfig' )

_cliSavers = []

def rsvpLerCliSaver( func ):
   _cliSavers.append( func )

@rsvpLerCliSaver
def saveLocalInterface( config, cmds, saveDefaultConfig, **kwargs ):
   if config.localIntf != '':
      cmds.addCommand( 'local-interface ' + config.localIntf )
   elif saveDefaultConfig:
      cmds.addCommand( 'no local-interface' )

@rsvpLerCliSaver
def saveOptimizationGlobal( config, cmds, saveDefaultConfig, **kwargs ):
   if config.optimizationInterval != RsvpLerConstants.optimizationIntervalDisabled:
      cmds.addCommand(
         'optimization interval ' + str( config.optimizationInterval ) + ' seconds' )
   elif saveDefaultConfig:
      cmds.addCommand( 'no optimization' )

@rsvpLerCliSaver
def saveAutoBandwidthTunnelFailure( config, cmds, saveDefaultConfig, **kwargs ):
   if not config.lastBwPreserved:
      cmds.addCommand( 'bandwidth auto tunnel failure computed-bandwidth reset' )
   elif saveDefaultConfig:
      cmds.addCommand( 'bandwidth auto tunnel failure computed-bandwidth preserved' )

@rsvpLerCliSaver
def saveLspSelfPing( config, cmds, saveDefaultConfig, **kwargs ):
   if config.lspSelfPingMode != lspSelfPingNone:
      if config.lspSelfPingMode == lspSelfPingEnabled:
         cmds.addCommand( 'lsp self-ping' )
   elif saveDefaultConfig:
      cmds.addCommand( 'no lsp self-ping' )

def saveRsvpLerLeaf( dstIp, rsvpPathSpec, parentMode, saveDefaultConfig ):
   params = ( parentMode, dstIp )
   mode = parentMode[ RsvpLeafConfigMode ].getOrCreateModeInstance( params )
   cmds = mode[ 'RsvpLer.p2mpPathSpecConfig.leaf' ]
   cliCmds = mode.leafToCliCmds( rsvpPathSpec.leaf.get( dstIp ),
                                 saveDefaultConfig )
   for cmd in cliCmds:
      cmds.addCommand( cmd )

def savePathSpecConfig( mode, pathSpec, cmds, saveDefaultConfig ):
   cliCmds = mode.pathSpecToCliCmds( pathSpec, saveDefaultConfig )
   for cmd in cliCmds:
      cmds.addCommand( cmd )

def saveRsvpLerPathSpec( rsvpPathSpec, root, saveDefaultConfig ):
   pathSpecName = rsvpPathSpec.pathSpecName
   pathSpecType = rsvpPathSpec.pathSpecType
   sessionType = rsvpPathSpec.sessionType
   teMode = root[ RouterGlobalTeMode ].getSingletonInstance()
   rsvpMode = teMode[ RsvpTeConfigMode ].getSingletonInstance()
   p2mp = sessionType == p2mpLspTunnel
   param = ( pathSpecName, pathSpecType, p2mp )
   if sessionType == p2mpLspTunnel:
      mode = rsvpMode[ RsvpP2mpPathSpecConfigMode ].getOrCreateModeInstance( param )
      cmds = mode[ 'RsvpLer.p2mpPathSpecConfig' ]
   elif sessionType == p2pLspTunnel:
      mode = rsvpMode[ RsvpP2pPathSpecConfigMode ].getOrCreateModeInstance( param )
      cmds = mode[ 'RsvpLer.p2pPathSpecConfig' ]
   else:
      return
   savePathSpecConfig( mode, rsvpPathSpec, cmds, saveDefaultConfig )
   for dstIp in rsvpPathSpec.leaf:
      saveRsvpLerLeaf( dstIp, rsvpPathSpec, mode, saveDefaultConfig )

def saveTunnelSpecConfig( mode, tunnelSpec, cmds, saveDefaultConfig,
                          rsvpLerCliConfig ):
   cliCmds = mode.tunnelSpecOrProfileToCliCmds(
      tunnelSpec, rsvpLerCliConfig, saveDefaultConfig )
   for cmd in cliCmds:
      cmds.addCommand( cmd )

def saveTunnelProfileConfig( mode, tunnelProfile, cmds, saveDefaultConfig,
                             rsvpLerCliConfig ):
   cliCmds = mode.tunnelSpecOrProfileToCliCmds(
      tunnelProfile, rsvpLerCliConfig, saveDefaultConfig )
   for cmd in cliCmds:
      cmds.addCommand( cmd )

def saveRsvpLerTunnelSpec( rsvpTunnelSpec, root, saveDefaultConfig,
                           rsvpLerCliConfig ):
   teMode = root[ RouterGlobalTeMode ].getSingletonInstance()
   rsvpMode = teMode[ RsvpTeConfigMode ].getSingletonInstance()
   param = rsvpTunnelSpec.tunnelName
   if rsvpTunnelSpec.sessionType == p2mpLspTunnel:
      mode = rsvpMode[ RsvpP2mpTunnelSpecConfigMode ].getOrCreateModeInstance(
                                                                              param )
      cmds = mode[ 'RsvpLer.p2mpTunnelSpecConfig' ]
   elif rsvpTunnelSpec.sessionType == p2pLspTunnel:
      mode = rsvpMode[ RsvpP2pTunnelSpecConfigMode ].getOrCreateModeInstance( param )
      cmds = mode[ 'RsvpLer.p2pTunnelSpecConfig' ]
   else:
      # Invalid session type, ignoring the command
      return
   saveTunnelSpecConfig( mode, rsvpTunnelSpec, cmds, saveDefaultConfig,
                         rsvpLerCliConfig )

def saveRsvpLerTunnelProfile( rsvpTunnelProfile, root, saveDefaultConfig,
                                  rsvpLerCliConfig ):
   teMode = root[ RouterGlobalTeMode ].getSingletonInstance()
   rsvpMode = teMode[ RsvpTeConfigMode ].getSingletonInstance()
   param = rsvpTunnelProfile.profileName
   if rsvpTunnelProfile.sessionType == p2mpLspTunnel:
      mode = rsvpMode[ RsvpP2mpTunnelProfileConfigMode ].getOrCreateModeInstance(
                                                                              param )
      cmds = mode[ 'RsvpLer.p2mpTunnelProfileConfig' ]
   elif rsvpTunnelProfile.sessionType == p2pLspTunnel:
      mode = rsvpMode[ RsvpP2pTunnelProfileConfigMode ].getOrCreateModeInstance(
                                                                              param )
      cmds = mode[ 'RsvpLer.p2pTunnelProfileConfig' ]
   else:
      # Invalid session type, ignoring the command
      return
   saveTunnelProfileConfig( mode, rsvpTunnelProfile, cmds, saveDefaultConfig,
                         rsvpLerCliConfig )

@CliSave.saver( 'Rsvp::RsvpLerCliConfig', RsvpLerCliConfig.mountPath,
                requireMounts=( 'routing/hardware/mpls/capability',
                                'routing/hardware/route/status',
                                'te/config' ) )
def saveRsvpLerCliConfig( rsvpLerCliConfig, root, requireMounts, options ):
   teConfig = requireMounts[ 'te/config' ]
   if teConfig.enabled == teConfig.enabledDefault:
      return

   hwCapability = requireMounts[ 'routing/hardware/mpls/capability' ]
   routingHwRouteStatus = requireMounts[ 'routing/hardware/route/status' ]

   rsvpLerSupported = ( hwCapability.mplsSupported and
                        routingHwRouteStatus.hierarchicalFecsEnabled )
   saveDefaultConfig = options.saveAll and rsvpLerSupported

   teMode = root[ RouterGlobalTeMode ].getSingletonInstance()
   mode = teMode[ RsvpTeConfigMode ].getSingletonInstance()
   cmds = mode[ 'RsvpLer.config' ]

   for saver in _cliSavers:
      saver( rsvpLerCliConfig, cmds, saveDefaultConfig,
             requireMounts=requireMounts )

@CliSave.saver( 'Rsvp::RsvpLerClient', 'mpls/rsvp/lerClient/cli',
                requireMounts=( RsvpLerCliConfig.mountPath,
                                'routing/hardware/mpls/capability',
                                'routing/hardware/route/status' ) )
def saveRsvpLerCliClient( lerClient, root, requireMounts, options ):
   hwCapability = requireMounts[ 'routing/hardware/mpls/capability' ]
   routingHwRouteStatus = requireMounts[ 'routing/hardware/route/status' ]
   rsvpLerCliConfig = requireMounts[ RsvpLerCliConfig.mountPath ]

   rsvpLerSupported = ( hwCapability.mplsSupported and
                        routingHwRouteStatus.hierarchicalFecsEnabled )
   saveDefaultConfig = options.saveAll and rsvpLerSupported

   # Save pathSpecs
   cliPathSpecColl = lerClient
   if cliPathSpecColl:
      for pathSpec in cliPathSpecColl.pathSpec.values():
         saveRsvpLerPathSpec( pathSpec, root, saveDefaultConfig )

   # Save tunnelSpecs
   if lerClient:
      for tunnelSpec in lerClient.tunnelSpec.values():
         saveRsvpLerTunnelSpec( tunnelSpec, root, saveDefaultConfig,
                                rsvpLerCliConfig )
      for tunnelProfile in lerClient.tunnelProfile.values():
         saveRsvpLerTunnelProfile( tunnelProfile, root, saveDefaultConfig,
                                   rsvpLerCliConfig )

@CliSave.saver( 'TrafficEngineering::Config', 'te/config',
                requireMounts=( 'routing/hardware/mpls/capability',
                                'routing/hardware/route/status',
                                'te/config' ) )
def saveEcmpTiebreakConfig( teConfig, root, requireMounts, options ):
   teConfig = requireMounts[ 'te/config' ]
   if teConfig.enabled == teConfig.enabledDefault:
      return
   hwCapability = requireMounts[ 'routing/hardware/mpls/capability' ]
   routingHwRouteStatus = requireMounts[ 'routing/hardware/route/status' ]

   rsvpLerSupported = ( hwCapability.mplsSupported and
                        routingHwRouteStatus.hierarchicalFecsEnabled )
   saveDefaultConfig = options.saveAll and rsvpLerSupported

   teMode = root[ RouterGlobalTeMode ].getSingletonInstance()
   mode = teMode[ RsvpTeConfigMode ].getSingletonInstance()
   cmds = mode[ 'RsvpLer.config' ]
   if not teConfig.rsvpEcmpLeastFill:
      cmds.addCommand( 'cspf ecmp tie-break random' )
   elif saveDefaultConfig:
      cmds.addCommand( 'cspf ecmp tie-break least-fill' )
