#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliSavePlugin.RoutingBgpCliSave import (
   registerAfPeerConfigCallback,
   RouterBgpBaseAfConfigMode,
   saveAfConfigCallbackDict,
   RouterBgpBaseConfigMode,
)
from RtMembershipLib import(
   nlriSupportEnumToCli,
)
from RouteMapLib import isAsdotConfigured

RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.rt-membership' )

def saveBgpRtMembershipPeerConfig( cmds, config, saveAll, addrOrPgName ):
   # neighbor PEER default-route-target [ only | disabled ]
   cmd = 'neighbor %s default-route-target' % ( addrOrPgName )
   if ( config.rtMembershipDefaultRouteTarget ==
        'rtMembershipDefaultRouteTargetEnabledWithOnly' ):
      cmd += ' only'
   elif ( config.rtMembershipDefaultRouteTarget ==
          'rtMembershipDefaultRouteTargetDisabled' ):
      cmd += ' disabled'
   if ( config.rtMembershipDefaultRouteTarget !=
        config.rtMembershipDefaultRouteTargetDefault ):
      cmds.append( cmd )
   elif saveAll:
      cmds.append( 'no ' + cmd )

   # neighbor PEER default-route-target encoding origin-as omit [ disabled ]
   cmd = 'neighbor %s default-route-target encoding origin-as omit' % (
      addrOrPgName )
   if ( config.rtMembershipDefaultRouteTargetEncoding ==
        'rtMembershipDefaultRouteTargetEncodingOriginAsSet' ):
      cmd += ' disabled'
   if ( config.rtMembershipDefaultRouteTargetEncoding !=
        config.rtMembershipDefaultRouteTargetEncodingDefault ):
      cmds.append( cmd )
   elif saveAll:
      cmds.append( 'no ' + cmd )

   # neighbor PEER nlri support length ( disabled | 0,96 | 0,32-96 | 32-96 )
   cmd = f"neighbor {addrOrPgName} nlri support length"
   # None is the default value
   if config.rtMembershipNlriSupport is not None:
      nlriSupport = nlriSupportEnumToCli[ config.rtMembershipNlriSupport ]
      cmds.append( f"{cmd} {nlriSupport}" )
   elif saveAll:
      cmds.append( f"no {cmd}" )

registerAfPeerConfigCallback( 'rt-membership', saveBgpRtMembershipPeerConfig )

def saveBgpRtMembershipConfig( bgpVrfConfig, addrFamily, saveAllOrDetail=False ):
   # neighbor default nlri support length ( disabled | 0,96 | 0,32-96 | 32-96 )
   cmds = []
   cmd = 'neighbor default nlri support length'
   nlriSupport = nlriSupportEnumToCli[ bgpVrfConfig.rtMembershipNlriSupport ]
   if bgpVrfConfig.rtMembershipNlriSupport != 'rtMembershipNlriSupportDisabled':
      # '-Disabled is not saved as it is the default for the command.
      cmds.append( f"{cmd} {nlriSupport}" )
   elif saveAllOrDetail:
      cmds.append( f"no {cmd}" )
   return cmds


# saveAfConfigCallbackDict would be used to display RtMembership
# specific config in "show bgp config active" output.
# For "show running" the config is displayed through
# saveRtMembershipAfConfig with CliSave.saver decorator.
saveAfConfigCallbackDict[ 'rt-membership' ] = saveBgpRtMembershipConfig

@CliSave.saver( 'Routing::Bgp::Config', 'routing/bgp/config',
                requireMounts=( 'routing/bgp/asn/config', ) )
def saveBgpRtMembershipAfConfig( bgpConfig, root, requireMounts, options ):
   if bgpConfig.asNumber == 0:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   af = "rt-membership"
   saveAllOrDetail = options.saveAll or options.saveAllDetail
   cmds = saveBgpRtMembershipConfig( bgpConfig, af, saveAllOrDetail )
   if cmds:
      bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
            ].getOrCreateModeInstance( af )
      for cmd in cmds:
         bgpAfMode[ 'Bgp.afConfig.%s' % af ].addCommand( cmd )
