# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from __future__ import absolute_import, division, print_function
import CliSave
from CliSavePlugin import Management
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliMode.PasswordPolicy import PasswordPolicyMode
from Toggles.MgmtSecurityToggleLib import toggleMinimumChangedCharactersEnabled
from Toggles.MgmtSecurityToggleLib import toggleDenyCommandsEnabled

mgmtSecurityConfigPath = 'mgmt/security/config'
class SecurityConfigMode( Management.MgmtConfigMode ):
   def __init__( self, param ):
      Management.MgmtConfigMode.__init__( self, "security" )

# 'Mgmt.security' CLI save block need to run before IntfConfigMode
# for password encryption configuration in IntfConfig submode
CliSave.GlobalConfigMode.addChildMode( SecurityConfigMode,
                                       before=[ IntfConfigMode ] )
SecurityConfigMode.addCommandSequence( 'Mgmt.security' )

class PasswordPolicyConfigMode( PasswordPolicyMode, CliSave.Mode ):
   def __init__( self, param ):
      PasswordPolicyMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

SecurityConfigMode.addChildMode( PasswordPolicyConfigMode )
PasswordPolicyConfigMode.addCommandSequence( 'Mgmt.security.passwdPolicy' )

@CliSave.saver( 'Mgmt::Security::Config', mgmtSecurityConfigPath )
def saveSecurity( securityConfig, root, requireMounts, options ):
   mode = root[ SecurityConfigMode ].getSingletonInstance()
   cmds = mode[ 'Mgmt.security' ]
   hwEntEnabled = securityConfig.entropySourceHardware
   havegeEnabled = securityConfig.entropySourceHaveged
   jitterEnabled = securityConfig.entropySourceJitter
   if ( hwEntEnabled == securityConfig.defaultEntropySourceHardware
         and havegeEnabled == securityConfig.defaultEntropySourceHaveged
         and jitterEnabled == securityConfig.defaultEntropySourceJitter ):
      if options.saveAll:
         cmds.addCommand( 'no entropy source' )
   else:
      sourcesList = []
      if hwEntEnabled:
         sourcesList.append( "hardware" )
      if havegeEnabled:
         sourcesList.append( "haveged" )
      if jitterEnabled:
         sourcesList.append( "cpu jitter" )
      cmd = "entropy source {entSources}"
      cmds.addCommand( cmd.format( entSources=" ".join( sourcesList ) ) )

   if ( securityConfig.useEntropyServer !=
        securityConfig.defaultUseEntropyServer ) or options.saveAll:
      cmd = 'entropy source hardware exclusive'
      if not securityConfig.useEntropyServer:
         cmd = 'no ' + cmd
      cmds.addCommand( cmd )

   if securityConfig.fipsLogging:
      cmds.addCommand( 'fips logging' )
   elif options.saveAll:
      cmds.addCommand( 'no fips logging' )

   if ( securityConfig.minPasswordLength != securityConfig.defaultMinPasswordLength
        or options.saveAll ):
      cmd = 'password minimum length'
      if securityConfig.minPasswordLength == 0:
         cmd = 'no ' + cmd
      else:
         cmd += ' %d' % securityConfig.minPasswordLength
      cmds.addCommand( cmd )

   if ( securityConfig.commonKeyEnabled != securityConfig.commonKeyEnabledDefault
        or options.saveAll ):
      cmd = 'password encryption-key common'
      # pylint: disable-next=singleton-comparison
      if securityConfig.commonKeyEnabled == False:
         cmd = 'no ' + cmd
      elif ( securityConfig.commonEncrytionKeyBytes !=
               securityConfig.commonEncrytionKeyDefault ):
         cmd += ' custom'
      cmds.addCommand( cmd )

   if securityConfig.enforceSignature or options.saveAll:
      cmd = "signature-verification extension"
      if not securityConfig.enforceSignature:
         cmd = "no " + cmd
      elif ( securityConfig.sslProfile != securityConfig.defaultSslProfile or
             options.saveAll ):
         cmd += " ssl profile %s" % securityConfig.sslProfile
      cmds.addCommand( cmd )

   for protocol in sorted( securityConfig.blockedNetworkProtocols ):
      cmds.addCommand( "network client protocol %s disabled" % protocol )

   if( securityConfig.commonEncrytionAESEnabled !=
       securityConfig.commonEncrytionAESEnabledDefault ) or options.saveAll:
      cmd = "password encryption reversible aes-256-gcm"
      if not securityConfig.commonEncrytionAESEnabled:
         cmd = "no " + cmd
      cmds.addCommand( cmd )

   for policyName, policy in securityConfig.passwordPolicies.items():
      policyMode = mode[ PasswordPolicyConfigMode
            ].getOrCreateModeInstance( policyName )
      policyCmds = policyMode[ 'Mgmt.security.passwdPolicy' ]
      if policy.minChanged != policy.defaultMinChanged or ( options.saveAll and
            toggleMinimumChangedCharactersEnabled() ):
         if policy.minChanged == policy.defaultMinChanged:
            policyCmds.addCommand( "no minimum changed" )
         else:
            policyCmds.addCommand( "minimum changed %d" % policy.minChanged )
      if policy.denyUsername != policy.defaultDenyUsername or ( options.saveAll and
            toggleDenyCommandsEnabled() ):
         if policy.denyUsername == policy.defaultDenyUsername:
            policyCmds.addCommand( "no deny username" )
         else:
            policyCmds.addCommand( "deny username" )
      if policy.denyLastPasswd != policy.defaultDenyLastPasswd or ( options.saveAll
            and toggleDenyCommandsEnabled() ):
         if policy.denyLastPasswd == policy.defaultDenyLastPasswd:
            policyCmds.addCommand( "no deny last" )
         else:
            policyCmds.addCommand( f"deny last {policy.denyLastPasswd}" )
      if policy.minDigits != policy.defaultMinDigits or options.saveAll:
         if policy.minDigits == policy.defaultMinDigits:
            policyCmds.addCommand( "no minimum digits" )
         else:
            policyCmds.addCommand( "minimum digits %d" % policy.minDigits )
      if ( policy.minPasswordLength != policy.defaultMinPasswordLength
           or options.saveAll ):
         if policy.minPasswordLength == policy.defaultMinPasswordLength:
            policyCmds.addCommand( "no minimum length" )
         else:
            policyCmds.addCommand( "minimum length %d" % policy.minPasswordLength )
      if policy.minLower != policy.defaultMinLower or options.saveAll:
         if policy.minLower == policy.defaultMinLower:
            policyCmds.addCommand( "no minimum lower" )
         else:
            policyCmds.addCommand( "minimum lower %d" % policy.minLower )
      if policy.minSpecial != policy.defaultMinSpecial or options.saveAll:
         if policy.minSpecial == policy.defaultMinSpecial:
            policyCmds.addCommand( "no minimum special" )
         else:
            policyCmds.addCommand( "minimum special %d" % policy.minSpecial )
      if policy.minUpper != policy.defaultMinUpper or options.saveAll:
         if policy.minUpper == policy.defaultMinUpper:
            policyCmds.addCommand( "no minimum upper" )
         else:
            policyCmds.addCommand( "minimum upper %d" % policy.minUpper )
      if policy.maxRepeated != policy.defaultMaxRepeated or options.saveAll:
         if policy.maxRepeated == policy.defaultMaxRepeated:
            policyCmds.addCommand( "no maximum repetitive" )
         else:
            policyCmds.addCommand( "maximum repetitive %d" % policy.maxRepeated )
      if policy.maxSequential != policy.defaultMaxSequential or options.saveAll:
         if policy.maxSequential == policy.defaultMaxSequential:
            policyCmds.addCommand( "no maximum sequential" )
         else:
            policyCmds.addCommand( "maximum sequential %d" % policy.maxSequential )
