# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import CliSave
from RoutingIntfUtils import allRoutingProtocolIntfNames
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.RouterGeneralCliSave import RouterGeneralCliSaveMode
import Tac
from CliMode.SegmentRoutingMode import SrModeBase
from Toggles.MplsToggleLib import (
   toggleReserveServiceLabelEnabled,
   toggleSrProxySegDefaultPopEnabled,
)

srSidInvalid = Tac.Type( "Routing::SegmentRoutingCli::Constants" ).srSidInvalid

class SrMode( SrModeBase, CliSave.Mode ):
   def __init__( self, param ):
      SrModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterGeneralCliSaveMode.addChildMode( SrMode )
SrMode.addCommandSequence( 'SegmentRouting.config' )

def getOrCreateSrModeInst( root ):
   generalMode = root[ RouterGeneralCliSaveMode ].getSingletonInstance()
   return generalMode[ SrMode ].getSingletonInstance()

# Node-segment configuration is shared by OSPF-SR and ISIS-SR, originally the
# configuration was stored in Routing::Isis::Config and was handled by the
# respective Isis plugins. Now with both OSPF and ISIS sharing the config,
# SegmentRoutingCli implements it. To maintain the order of the configuration
# statements as before we insert the node-segment config statements in between
# the isis intf config statements (so that customers who upgrade do not see any
# difference in the running config).
IntfConfigMode.addCommandSequence( 'Sr.intf', after=[ 'Ira.ipIntf' ] )

@CliSave.saver( 'Routing::SegmentRoutingCli::Config', 'routing/sr/config',
                requireMounts=( 'interface/config/all',
                                'interface/status/all' ) )
def saveConfig( srConfig, root, requireMounts, options ):

   if options.saveAllDetail:
      rpIntfs = allRoutingProtocolIntfNames( requireMounts, includeEligible=True )
      intfs = set( rpIntfs )
      intfs.update( srConfig.intfConfig )
   elif options.saveAll:
      # Display Isis configuration on routing protocol interfaces as well as
      # switchports present in intfConfig collection.
      rpIntfs = allRoutingProtocolIntfNames( requireMounts )
      intfs = set( rpIntfs )
      intfs.update( srConfig.intfConfig )
   else:
      intfs = srConfig.intfConfig

   for intfName in intfs:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      cmds = mode[ 'Sr.intf' ]

      srIntf = srConfig.intfConfig.get( intfName, None )
      if srIntf and ( srIntf.srNodeSegmentIndex != srSidInvalid or
                      srIntf.srV4NodeSegment ):
         if srIntf.srNodeSegmentIndex != srSidInvalid:
            explicitNull = ( ' explicit-null' if srIntf.srNodeSegmentExplicitNull
                              else '' )
            noPhp = ' no-php' if srIntf.srNodeSegmentNoPhp else ''
            cmds.addCommand( 'node-segment ipv4 index %d%s%s' %
                             ( srIntf.srNodeSegmentIndex, explicitNull, noPhp ) )
         for nodeSid in sorted( srIntf.srV4NodeSegment ):
            nodeSeg = srIntf.srV4NodeSegment[ nodeSid ]
            explicitNull = ' explicit-null' if nodeSeg.explicitNull else ''
            noPhp = ' no-php' if nodeSeg.noPhp else ''
            cmds.addCommand( 'node-segment ipv4 index %d flex-algo %s%s%s' % (
               nodeSid, nodeSeg.flexAlgoName, explicitNull, noPhp ) )
      elif options.saveAll:
         cmds.addCommand( 'no node-segment ipv4 index' )

      if srIntf and srIntf.srV4NodeSegmentLabel:
         for nodeSid in sorted( srIntf.srV4NodeSegmentLabel ):
            nodeSeg = srIntf.srV4NodeSegmentLabel[ nodeSid ]
            explicitNull = ' explicit-null' if nodeSeg.explicitNull else ''
            noPhp = ' no-php' if nodeSeg.noPhp else ''
            if nodeSeg.flexAlgoName:
               cmds.addCommand( 'node-segment ipv4 label %d flex-algo %s%s%s' % (
                  nodeSid, nodeSeg.flexAlgoName, explicitNull, noPhp ) )
            else:
               cmds.addCommand( 'node-segment ipv4 label %d%s%s' %
                                ( nodeSid, explicitNull, noPhp ) )
      elif options.saveAll:
         cmds.addCommand( 'no node-segment ipv4 label' )

      if srIntf and ( srIntf.srV6NodeSegmentIndex != srSidInvalid or
                      srIntf.srV6NodeSegment ):
         if srIntf.srV6NodeSegmentIndex != srSidInvalid:
            explicitNull = ( ' explicit-null' if srIntf.srV6NodeSegmentExplicitNull
                             else '' )
            noPhp = ' no-php' if srIntf.srV6NodeSegmentNoPhp else ''
            cmds.addCommand( 'node-segment ipv6 index %d%s%s' %
                            ( srIntf.srV6NodeSegmentIndex, explicitNull, noPhp ) )
         for nodeSid in sorted( srIntf.srV6NodeSegment ):
            nodeSeg = srIntf.srV6NodeSegment[ nodeSid ]
            explicitNull = ' explicit-null' if nodeSeg.explicitNull else ''
            noPhp = ' no-php' if nodeSeg.noPhp else ''
            cmds.addCommand( 'node-segment ipv6 index %d flex-algo %s%s%s' % (
               nodeSid, nodeSeg.flexAlgoName, explicitNull, noPhp ) )
      elif options.saveAll:
         cmds.addCommand( 'no node-segment ipv6 index' )

      if srIntf and srIntf.srV6NodeSegmentLabel:
         for nodeSid in sorted( srIntf.srV6NodeSegmentLabel ):
            nodeSeg = srIntf.srV6NodeSegmentLabel[ nodeSid ]
            explicitNull = ' explicit-null' if nodeSeg.explicitNull else ''
            noPhp = ' no-php' if nodeSeg.noPhp else ''
            if nodeSeg.flexAlgoName:
               cmds.addCommand( 'node-segment ipv6 label %d flex-algo %s%s%s' % (
                  nodeSid, nodeSeg.flexAlgoName, explicitNull, noPhp ) )
            else:
               cmds.addCommand( 'node-segment ipv6 label %d%s%s' %
                                ( nodeSid, explicitNull, noPhp ) )
      elif options.saveAll:
         cmds.addCommand( 'no node-segment ipv6 label' )

   if srConfig.enabled:
      _ = getOrCreateSrModeInst( root )

@CliSave.saver( 'Mpls::Config', 'routing/mpls/config' )
def saveTunnelIgpFecSharin( entity, root, requireMounts, options ):
   if not entity.tunnelIgpFecSharing:
      srMode = getOrCreateSrModeInst( root )
      cmds = srMode[ 'SegmentRouting.config' ]
      cmds.addCommand( 'fec sharing igp tunnel disabled' )

@CliSave.saver( 'Mpls::Config', 'routing/mpls/config' )
def saveReserveServiceLabel( entity, root, requireMounts, options ):
   if toggleReserveServiceLabelEnabled():
      if entity.reserveServiceLabel:
         srMode = getOrCreateSrModeInst( root )
         cmds = srMode[ 'SegmentRouting.config' ]
         cmds.addCommand( 'maximum-sid-depth reserve service-label' )

@CliSave.saver( 'Mpls::Config', 'routing/mpls/config' )
def saveProxyNodeSegmentFallbackPopFoward( entity, root, requireMounts, options ):
   if toggleSrProxySegDefaultPopEnabled():
      if entity.srProxySegFallbackPopEnabled:
         srMode = getOrCreateSrModeInst( root )
         cmds = srMode[ 'SegmentRouting.config' ]
         cmds.addCommand( 'proxy-node-segment mpls fallback pop forward' )
      elif options.saveAll:
         srMode = getOrCreateSrModeInst( root )
         cmds = srMode[ 'SegmentRouting.config' ]
         cmds.addCommand( 'no proxy-node-segment mpls fallback pop forward' )
