# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliMode.SfeServiceInsertion import (
        RouterServiceInsertionModeBase,
        RouterSiConnectionModeBase,
        RouterSiServiceGroupModeBase,
        RouterSiServiceGroupInstanceModeBase,
        )
from Toggles.WanTECommonToggleLib import (
      toggleAvtRemoteInternetExitEnabled,
)
from TypeFuture import TacLazyType

ServiceGroupType = TacLazyType( 'Avt::ServiceGroupType' )

class RouterServiceInsertionSaveMode( RouterServiceInsertionModeBase,
                                      CliSave.Mode ):
   def __init__( self, param ):
      RouterServiceInsertionModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

class RouterSiConnectionSaveMode( RouterSiConnectionModeBase,
                                  CliSave.Mode ):
   def __init__( self, param ):
      self.connectionName = param
      RouterSiConnectionModeBase.__init__( self, self.connectionName )
      CliSave.Mode.__init__( self, param )

class RouterSiServiceGroupSaveMode( RouterSiServiceGroupModeBase,
                                    CliSave.Mode ):
   def __init__( self, param ):
      groupName, groupId = param
      self.groupName = groupName
      self.groupId = groupId
      RouterSiServiceGroupModeBase.__init__( self, self.groupName, self.groupId )
      CliSave.Mode.__init__( self, param )

class RouterSiServiceGroupInstanceSaveMode( RouterSiServiceGroupInstanceModeBase,
                                            CliSave.Mode ):
   def __init__( self, param ):
      groupName, instanceName, instanceId = param
      self.groupName = groupName
      self.instanceName = instanceName
      self.instanceId = instanceId
      RouterSiServiceGroupInstanceModeBase.__init__( self, self.groupName,
            self.instanceName, self.instanceId )
      CliSave.Mode.__init__( self, param )

def registerAllCommands():
   CliSave.GlobalConfigMode.addChildMode( RouterServiceInsertionSaveMode )
   RouterServiceInsertionSaveMode.addCommandSequence(
                                    'SfeServiceInsertion.RouterServiceInsertion' )

   RouterServiceInsertionSaveMode.addChildMode( RouterSiConnectionSaveMode )
   RouterSiConnectionSaveMode.addCommandSequence(
                                    'SfeServiceInsertion.SiConnection' )

   if not toggleAvtRemoteInternetExitEnabled():
      return

   RouterServiceInsertionSaveMode.addChildMode( RouterSiServiceGroupSaveMode,
                                    after=[ RouterSiConnectionSaveMode ] )
   RouterSiServiceGroupSaveMode.addCommandSequence(
                                    'SfeServiceInsertion.SiServiceGroup' )
   RouterSiServiceGroupSaveMode.addCommandSequence(
         'SfeServiceInsertion.SiServiceGroup.SiServiceGroupType' )
   RouterSiServiceGroupSaveMode.addChildMode( RouterSiServiceGroupInstanceSaveMode,
         after=[ 'SfeServiceInsertion.SiServiceGroup.SiServiceGroupType' ] )
   RouterSiServiceGroupInstanceSaveMode.addCommandSequence(
         'SfeServiceInsertion.SiServiceGroup.SiServiceGroupInstance' )

registerAllCommands()

def getServiceGroupTypeStr( serviceGroupType ):
   typeStr = None
   if serviceGroupType == ServiceGroupType.internetExit:
      typeStr = 'internet-exit'
   return f'type {typeStr}' if typeStr else None

@CliSave.saver( 'SfeServiceInsertion::ServiceInsertionConfigDir', 'si/cli/config' )
def saveConfig( entity, root, requireMounts, options ):
   if not entity.enabled:
      return

   siMode = root[ RouterServiceInsertionSaveMode ].getOrCreateModeInstance(
                                    'SfeServiceInsertion.RouterServiceInsertion' )

   # handle connection configs
   for connName, connCfg in entity.connectionConfig.items():
      connMode = siMode[ RouterSiConnectionSaveMode ].getOrCreateModeInstance(
                                                     connName )
      connCmds = connMode[ 'SfeServiceInsertion.SiConnection' ]
      if connCfg.primaryIntf:
         if connCfg.primaryIntf.nextHop:
            connCmds.addCommand( f'interface {connCfg.primaryIntf.intfId} '
                                 f'next-hop {connCfg.primaryIntf.nextHop}' )
         else:
            connCmds.addCommand( f'interface {connCfg.primaryIntf.intfId} primary' )
      if connCfg.secondaryIntf:
         connCmds.addCommand( f'interface {connCfg.secondaryIntf.intfId} secondary' )
      if connCfg.monConnectivityHostName:
         connCmds.addCommand(
            f'monitor connectivity host {connCfg.monConnectivityHostName}' )

   if not toggleAvtRemoteInternetExitEnabled():
      return

   # handle service group configs
   for groupName, groupCfg in entity.serviceGroupConfig.items():
      groupMode = siMode[ RouterSiServiceGroupSaveMode ].getOrCreateModeInstance(
                          ( groupName, groupCfg.groupId ) )
      typeStr = None
      typeCmds = groupMode[ 'SfeServiceInsertion.SiServiceGroup.SiServiceGroupType' ]
      typeStr = getServiceGroupTypeStr( groupCfg.type )
      if typeStr:
         typeCmds.addCommand( typeStr )

      for instName, instanceCfg in groupCfg.instanceConfig.items():
         instanceMode = groupMode[ RouterSiServiceGroupInstanceSaveMode ].\
                getOrCreateModeInstance( ( groupName, instName,
                                           instanceCfg.instanceId ) )
         instanceCmds = instanceMode[
               'SfeServiceInsertion.SiServiceGroup.SiServiceGroupInstance' ]
         if instanceCfg.connectionName:
            connStr = f'connection {instanceCfg.connectionName}'
            instanceCmds.addCommand( connStr )
