#!/usr/bin/env python3
# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from collections import namedtuple
from itertools import chain
import CliSave
from CliSavePlugin import IntfCliSave
from SflowConst import defaultPort
from IpLibConsts import DEFAULT_VRF
import SflowUtil
import EthIntfUtil
import Tac
import Toggles.SflowLibToggleLib as ToggleSflow

CliSave.GlobalConfigMode.addCommandSequence( 'Sflow.config' )
CliSave.GlobalConfigMode.addCommandSequence( 'Sflow.extension' )
IntfCliSave.IntfConfigMode.addCommandSequence( 'Sflow.config' )

def findHwStatus( requireMounts ):
   return SflowUtil.sflowHwStatus( requireMounts[ 'sflow/hwstatus' ] )

@CliSave.saver( 'Sflow::Config', 'sflow/config',
                requireMounts=( 'sflow/hwstatus', ) )
def saveSwConfig( entity, root, requireMounts, options ):
   sflowHwStatus = findHwStatus( requireMounts )
   unmodifiedEgressKw = ''
   if sflowHwStatus.egressUnmodifiedSflowIntfSupported:
      unmodifiedEgressKw = ' unmodified'
   saveAll = options.saveAll
   vxlanHeaderStripSupported = True
   # Display sflow config/defaults only if sflow is supported

   if not sflowHwStatus.sflowSupported:
      saveAll = False
      vxlanHeaderStripSupported = False

   if not sflowHwStatus.sflowSampleVxlanHeaderStripSupported:
      vxlanHeaderStripSupported = False

   # First save the global configuration.
   cmds = root[ 'Sflow.config' ]

   if entity.sampleRate != entity.sampleRateDefault or saveAll:
      if not entity.sampleDangerous:
         cmds.addCommand( 'sflow sample %i' % entity.sampleRate )
      else:
         cmds.addCommand( 'sflow sample dangerous %i' % entity.sampleRate )

   if entity.sampleTruncateSize != entity.sampleTruncateSizeDefault or saveAll:
      cmds.addCommand( 'sflow sample truncate size %i' %
                        entity.sampleTruncateSize )

   if entity.maxDatagramSize != entity.maxDatagramSizeDefault or saveAll:
      cmds.addCommand( f'sflow datagram size maximum {entity.maxDatagramSize}' )

   if entity.pollingInterval != entity.pollingIntervalDefault or saveAll:
      if entity.pollingInterval != 0:
         cmds.addCommand( 'sflow polling-interval %i' % entity.pollingInterval )

   if entity.pollingInterval == 0:
      cmds.addCommand( 'no sflow polling-interval' )

   if not entity.samplingEnabled:
      cmds.addCommand( 'no sflow sample' )

   if entity.sampleIncludeDropReasonAcl:
      cmds.addCommand( 'sflow sample include drop reason acl' )

   if entity.sampleVxlanHeaderStrip:
      cmds.addCommand( 'sflow sample vxlan header strip' )
   elif saveAll and vxlanHeaderStripSupported:
      cmds.addCommand( 'no sflow sample vxlan header strip' )

   for vrfName, vrfConfig in sorted( entity.sflowVrfConfig.items() ):
      if vrfName != DEFAULT_VRF:
         vrfString = ' vrf %s' % vrfName
      else:
         vrfString = ''

      for hostAndPort in sorted( vrfConfig.collectorHostAndPort.values() ):
         if hostAndPort.port != defaultPort:
            cmds.addCommand( 'sflow%s destination %s %s' % (
               vrfString, hostAndPort.hostname, hostAndPort.port ) )
         else:
            cmds.addCommand( 'sflow%s destination %s' %
               ( vrfString, hostAndPort.hostname ) )

      if ( vrfConfig.addrConfigType == 'dependent' or
          vrfConfig.addrConfigType == 'none' or
          not ToggleSflow.toggleSflowIndependentAddressConfigEnabled() ):
         if vrfConfig.switchIpAddr:
            cmds.addCommand( 'sflow%s source %s' %
               ( vrfString, vrfConfig.switchIpAddr.stringValue ) )
         elif saveAll:
            cmds.addCommand( 'sflow%s source 0.0.0.0' % vrfString )

         if vrfConfig.switchIp6Addr:
            cmds.addCommand( 'sflow%s source %s' %
               ( vrfString, vrfConfig.switchIp6Addr.stringValue ) )
         elif saveAll:
            cmds.addCommand( 'sflow%s source ::' % vrfString )

      if vrfConfig.srcIntfName:
         cmds.addCommand( 'sflow%s source-interface %s' %
            ( vrfString, vrfConfig.srcIntfName ) )
      elif saveAll:
         cmds.addCommand( 'no sflow%s source-interface' % vrfString )

      if ( vrfConfig.addrConfigType == 'independent' and
            ToggleSflow.toggleSflowIndependentAddressConfigEnabled() ):
         if vrfConfig.switchIpAddr:
            cmds.addCommand( 'sflow%s source address %s' %
                  ( vrfString, vrfConfig.switchIpAddr.stringValue ) )
         elif saveAll:
            cmds.addCommand( 'sflow%s source address 0.0.0.0' % vrfString )

         if vrfConfig.switchIp6Addr:
            cmds.addCommand( 'sflow%s source address %s' %
                  ( vrfString, vrfConfig.switchIp6Addr.stringValue ) )
         elif saveAll:
            cmds.addCommand( 'sflow%s source address ::' % vrfString )

         if vrfConfig.agentIpAddr:
            cmds.addCommand( 'sflow%s agent address %s' %
                  ( vrfString, vrfConfig.agentIpAddr.stringValue ) )
         elif saveAll:
            cmds.addCommand( 'sflow%s agent address 0.0.0.0' %
                  vrfString )

         if vrfConfig.agentIp6Addr:
            cmds.addCommand( 'sflow%s agent address %s' %
                  ( vrfString, vrfConfig.agentIp6Addr.stringValue ) )
         elif saveAll:
            cmds.addCommand( 'sflow%s agent address ::' %
                  vrfString )


   if saveAll:
      if not entity.sflowVrfConfig:
         cmds.addCommand( 'no sflow source' )
         cmds.addCommand( 'no sflow source-interface' )

   if not entity.outputInterface:
      cmds.addCommand( 'no sflow sample output interface' )
   elif saveAll:
      cmds.addCommand( 'sflow sample output interface' )

   if entity.ingressSubintf:
      cmds.addCommand( 'sflow sample input subinterface' )
   elif saveAll:
      cmds.addCommand( 'no sflow sample input subinterface' )

   if entity.egressSubintf:
      cmds.addCommand( 'sflow sample output subinterface' )
   elif saveAll:
      cmds.addCommand( 'no sflow sample output subinterface' )

   if ToggleSflow.toggleSflowL2SubIntfVlanEnabled():
      if entity.updateSubIntfVlan:
         cmds.addCommand( 'sflow sample output vlan' )
      elif saveAll:
         cmds.addCommand( 'no sflow sample output vlan' )

   ExtensionInfo = namedtuple( 'ExtensionInfo', [ 'suffix', 'attrName' ] )
   sflowExtensions = [
      ExtensionInfo( 'bgp', 'bgpExtension', ),
      ExtensionInfo( 'router', 'routerExtension', ),
      ExtensionInfo( 'switch', 'switchExtension', ),
   ]

   # When implementing the MPLS Extension an ExtensionInfo
   # must be added to sflowExtensions (See BUG357099).

   if ToggleSflow.toggleSflowEvpnMplsEnabled():
      sflowExtensions.append( ExtensionInfo( 'evpn mpls', 'evpnMplsExtension' ) )

   if ToggleSflow.toggleSflowVplsExtensionEnabled():
      sflowExtensions.append( ExtensionInfo( 'vpls', 'vplsExtension' ) )

   if ToggleSflow.toggleSflowVxlanEnabled():
      sflowExtensions.append( ExtensionInfo( 'vxlan', 'vxlanExtension' ) )

   baseSflowExtenstionCmd = 'sflow extension '

   def addSflowExtensionCommand( curVal, suffix ):
      cmd = baseSflowExtenstionCmd + suffix
      if not curVal:
         cmd = 'no ' + cmd
      cmds.addCommand( cmd )

   for extension in sflowExtensions:
      suffix = extension.suffix
      attrName = extension.attrName
      curVal = getattr( entity, attrName )
      defaultVal = getattr( entity, attrName + 'Default' )
      if ( curVal != defaultVal ) or saveAll:
         addSflowExtensionCommand( curVal, suffix )

   if entity.tunnelIpv4EgrExtension:
      cmds.addCommand( 'sflow extension tunnel ipv4 egress' )
   elif saveAll:
      cmds.addCommand( 'no sflow extension tunnel ipv4 egress' )

   if entity.rewriteDscp:
      cmds.addCommand( 'sflow sample rewrite dscp' )
   elif saveAll:
      cmds.addCommand( 'no sflow sample rewrite dscp' )

   if entity.dscpValue != entity.dscpValueDefault:
      cmds.addCommand( 'sflow qos dscp %s' % entity.dscpValue )
   elif saveAll:
      cmds.addCommand( 'sflow qos dscp %s' % entity.dscpValueDefault )

   if entity.portChannelIfindex == 'portchannel':
      cmds.addCommand( 'sflow sample output portchannel ifindex portchannel' )
   elif entity.portChannelIfindex == 'member':
      cmds.addCommand( 'sflow sample output portchannel ifindex member' )

   if entity.egressSflowMulticastOutputIfindex:
      cmds.addCommand( 'sflow sample output multicast interface' )
   elif saveAll:
      cmds.addCommand( 'no sflow sample output multicast interface' )

   if entity.outputSviIfindex:
      cmds.addCommand( 'sflow sample output svi ifindex svi' )

   if entity.inputSviIfindex:
      cmds.addCommand( 'sflow sample input svi ifindex svi' )

   # Now save interface specific settings
   if saveAll:
      cfgIntfNames = EthIntfUtil.allSwitchportNames( requireMounts,
                                                     includeEligible=True )
   else:
      cfgIntfNames = chain( entity.enabledIngressIntf, entity.disabledIngressIntf,
                            entity.enabledEgressIntf, entity.disabledEgressIntf )
      cfgIntfNames = list( set( cfgIntfNames ) )

   for intfName in cfgIntfNames:
      mode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance( intfName )
      intfCmds = mode[ 'Sflow.config' ]

      enabledIngressIntfConfig = entity.enabledIngressIntf.get( intfName )
      disabledIngressIntfConfig = entity.disabledIngressIntf.get( intfName )
      if enabledIngressIntfConfig:
         intfCmds.addCommand( 'sflow enable' )
      elif disabledIngressIntfConfig:
         intfCmds.addCommand( 'no sflow enable' )
      elif saveAll:
         # interface has default config
         intfCmds.addCommand( 'default sflow enable' )

      enabledEgressIntfConfig = entity.enabledEgressIntf.get( intfName )
      disabledEgressIntfConfig = entity.disabledEgressIntf.get( intfName )
      if enabledEgressIntfConfig:
         intfCmds.addCommand( 'sflow egress%s enable' % unmodifiedEgressKw )
      elif disabledEgressIntfConfig:
         intfCmds.addCommand( 'no sflow egress%s enable' % unmodifiedEgressKw )
      elif saveAll:
         # interface has default config
         intfCmds.addCommand( 'default sflow egress%s enable' % unmodifiedEgressKw )


   # save global configuration for interface
   if not entity.ingressIntfsEnabledDefault:
      cmds.addCommand( 'sflow interface disable default' )
   elif saveAll:
      cmds.addCommand( 'no sflow interface disable default' )

   if entity.egressIntfsEnabledDefault:
      cmds.addCommand( 'sflow interface egress%s enable default'
                       % unmodifiedEgressKw )
   elif saveAll:
      cmds.addCommand( 'no sflow interface egress%s enable default'
                       % unmodifiedEgressKw )

   # Then decide to start Sflow if it was running before.
   if entity.enabled:
      cmds.addCommand( 'sflow run' )
   elif saveAll:
      cmds.addCommand( 'no sflow run' )
