# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
import Tac
from SrTePolicyLib import MplsLabel
from CliMode.SrTePolicy import SrTeModeBase
from CliMode.SrTePolicy import SrTePolicyModeBase
from CliMode.SrTePolicy import SrTeDynamicPolicyModeBase
from CliMode.SrTePolicy import SrTePolicyPathGroupModeBase
from CliMode.SrTePolicy import SrTePolicyPathLocalComputationModeBase
from CliMode.SrTePolicy import SrTePolicySegmentListModeBase
from CliSavePlugin.TeCliSave import RouterGlobalTeMode
from SrTePolicyCommonLib import tacEnlp, tacEnlpEnum
from SrTePolicy import CandidatePathType
from Toggles.SrTePolicyToggleLib import (
   toggleSrTePolicyIgpShortcutEnabled,
   toggleOptionalBsidEnabled,
   toggleDynamicallyAssignBsidEnabled,
   toggleSrTeTunnelHoldDownTimerEnabled,
)
from Toggles.SrTePolicyLibToggleLib import toggleDynamicSrTeEnabled
from TypeFuture import TacLazyType

SbfdHoldDownTime = TacLazyType( 'Bfd::HoldDownTime' )

class SrTeMode( SrTeModeBase, CliSave.Mode ):
   def __init__( self, param ):
      SrTeModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class SrTePolicyMode( SrTePolicyModeBase, CliSave.Mode ):
   def __init__( self, param ):
      SrTePolicyModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class SrTeDynamicPolicyMode( SrTeDynamicPolicyModeBase, CliSave.Mode ):
   def __init__( self, param ):
      SrTeDynamicPolicyModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class SrTePolicyPathGroupMode( SrTePolicyPathGroupModeBase, CliSave.Mode ):
   def __init__( self, param ):
      SrTePolicyPathGroupModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class SrTePolicyPathLocalComputationMode( SrTePolicyPathLocalComputationModeBase,
                                     CliSave.Mode ):
   def __init__( self, param ):
      SrTePolicyPathLocalComputationModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class SrTePolicyDynamicPathLocalComputationMode(
                                          SrTePolicyPathLocalComputationModeBase,
                                          CliSave.Mode ):
   def __init__( self, param ):
      SrTePolicyPathLocalComputationModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class SrTePolicySegmentListMode( SrTePolicySegmentListModeBase, CliSave.Mode ):
   def __init__( self, param ):
      SrTePolicySegmentListModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterGlobalTeMode.addChildMode( SrTeMode )
SrTeMode.addCommandSequence( 'SrTePolicy.config' )

SrTeMode.addChildMode( SrTePolicyMode )
SrTePolicyMode.addCommandSequence( 'SrTePolicy.policy.config' )

SrTeMode.addChildMode( SrTeDynamicPolicyMode )
SrTeDynamicPolicyMode.addCommandSequence( 'SrTePolicy.policy.dynamic.config' )

SrTePolicyMode.addChildMode( SrTePolicyPathGroupMode )
SrTePolicyPathGroupMode.addCommandSequence( 'SrTePolicy.policy.path.config' )

SrTePolicyPathGroupMode.addChildMode( SrTePolicySegmentListMode )
SrTePolicySegmentListMode.addCommandSequence(
   'SrTePolicy.policy.path.segmentList.config' )

SrTePolicyMode.addChildMode( SrTePolicyPathLocalComputationMode )
SrTePolicyPathLocalComputationMode.addCommandSequence(
   'SrTePolicy.policy.path.local.computation.config' )

if toggleDynamicSrTeEnabled():
   SrTeDynamicPolicyMode.addChildMode( SrTePolicyDynamicPathLocalComputationMode )
   SrTePolicyDynamicPathLocalComputationMode.addCommandSequence(
      'SrTePolicy.policy.dynamic.path.local.computation.config' )

def getOrCreateSrTeModeInst( root ):
   teMode = root[ RouterGlobalTeMode ].getSingletonInstance()
   return teMode[ SrTeMode ].getSingletonInstance()

def sfbdMultiDef():
   return Tac.Type( 'Bfd::BfdMultiplier' ).defval

def sbfdConfigDef():
   return Tac.Value( 'SrTePolicy::Sbfd::PolicySbfdConfig', 0, 0,
                                                        sfbdMultiDef() )

def _saveSbfdConfig( policy, policyCmds, saveAll ):
   if policy.sbfdConfig.sbfdRemoteDiscriminator != 0:
      bfdStr = "sbfd remote-discriminator "
      discriminator = policy.sbfdConfig.sbfdRemoteDiscriminator
      if policy.sbfdConfig.sbfdDiscIsU32:
         bfdStr += str( discriminator )
      else:
         ipDisc = Tac.Value( 'Arnet::IpAddr', discriminator ).stringValue
         bfdStr += ipDisc

      if policy.sbfdConfig.sbfdInterval != 0:
         bfdStr += " interval %u multiplier %u" % (
                                   policy.sbfdConfig.sbfdInterval,
                                   policy.sbfdConfig.sbfdMulti )

      if policy.sbfdConfig.sbfdProxy:
         bfdStr += " proxy"

      policyCmds.addCommand( bfdStr )
   elif saveAll:
      policyCmds.addCommand( 'no sbfd remote-discriminator' )

def saveInstanceConfig( srConfig, root, saveAll ):
   if not srConfig.enabled:
      return
   # Since we set srConfig.enabled=True the moment we enter the sr config submode,
   # we cannot have policy config unless srConfig.enabled is True
   srteMode = getOrCreateSrTeModeInst( root )

   def enlpBase( config, cmd, saveAll=False ):
      enlpCommand = 'explicit-null '
      if config.enlp != tacEnlp().valueDefault:
         if config.enlp == tacEnlpEnum.ipv4AndIpv6:
            enlpCommand += 'ipv4 ipv6'
         else:
            enlpCommand += config.enlp
         cmd.addCommand( enlpCommand )
      elif saveAll:
         enlpCommand = 'default ' + enlpCommand
         cmd.addCommand( enlpCommand )

   def pathSelectionTieBreakDiscriminator( config, cmd, saveAll=False ):
      # pstd stands for "path selection tie-break discriminator"
      pstdCommand = 'path selection tie-break discriminator'
      if config.useDiscriminatorInTieBreaking:
         cmd.addCommand( pstdCommand )
      elif saveAll:
         pstdCommand = 'default ' + pstdCommand
         cmd.addCommand( pstdCommand )

   def dynamicIgpMetricAddCfg( pCostMetric ):
      cfgStr = ''
      modType = Tac.Type( 'SrTePolicy::PolicyCostMetric::MetricModifierType' )
      if pCostMetric.dynamicMetricModifierType == modType.metricAdditive:
         cfgStr += ' + %d' % pCostMetric.dynamicMetricModifier
      elif pCostMetric.dynamicMetricModifierType == modType.metricSubtractive:
         cfgStr += ' - %d' % pCostMetric.dynamicMetricModifier
      if pCostMetric.dynamicMetricUnreachable != pCostMetric.defaultUnreachable:
         cfgStr += ' igp-unreachable %d' % pCostMetric.dynamicMetricUnreachable
      return cfgStr

   def enableGlobalSBFDEchoDefault( config, cmd, saveAll=False ):
      sbfdEchoDefaultCommand = 'sbfd echo default'
      if config.globalSBFDEcho:
         cmd.addCommand( sbfdEchoDefaultCommand )
      elif saveAll:
         sbfdEchoDefaultCommand = 'no ' + sbfdEchoDefaultCommand
         cmd.addCommand( sbfdEchoDefaultCommand )

   def saveSBFDHoldDownTime( config, cmd, saveAll=False ):
      cfgStr = None
      if config.sbfdHoldDownTime != SbfdHoldDownTime.invalid:
         cfgStr = 'sbfd hold down %d seconds' % config.sbfdHoldDownTime
      elif saveAll:
         cfgStr = 'no sbfd hold down'
      if cfgStr:
         cmd.addCommand( cfgStr )

   def savePathGroupLocalComputationModeCmds( config, cmd, saveAll ):
      cfgStr = None
      if pathGroup.candidatePathType == CandidatePathType.\
         segmentRoutingLabelResolution:
         cfgStr = "path segment-routing"
      elif pathGroup.candidatePathType == CandidatePathType.\
            segmentRoutingFlexAlgoLabelResolution:
         cfgStr = f"path segment-routing flex-algo {config.flexAlgoName}"
      elif saveAll:
         cfgStr = "path segment-routing"
      if cfgStr:
         cmd.addCommand( cfgStr )

   def savePathGroupModeCmds( pathGroup, pathMode, pathCmds, policy ):
      enlpBase( pathGroup, pathCmds )

      def saveSegmentListWithReturnPath( pathGroup, pathMode, pathCmds, policy,
                                       segmentList ):
         weight = pathGroup.staticSegmentList[ segmentList ].weight
         index = pathGroup.staticSegmentList[ segmentList ].index
         segListParam = pathGroup.staticSegmentList[ segmentList ]
         slMode = pathMode[ SrTePolicySegmentListMode ].\
            getOrCreateModeInstance( ( policy.key.endpoint,
                                       policy.key.color,
                                       pathGroup.preference,
                                       segmentList, weight, index
                                    ) )
         seglistCmds = slMode[ 'SrTePolicy.policy.path.segmentList.config' ]
         if segListParam.sbfdReturnPath:
            sbfdReturnPathCmd = "sbfd return-path label-stack"
            labelStackStr = ""
            for i in range( segListParam.sbfdReturnPath.stackSize ):
               labelStackStr += " "
               labelStackStr += \
                  str( segListParam.sbfdReturnPath.labelStack( i ) )
            seglistCmds.addCommand( sbfdReturnPathCmd + labelStackStr )

      for segmentList in sorted( pathGroup.staticSegmentList ):
         saveSegmentListWithReturnPath( pathGroup, pathMode, pathCmds, policy,
                                        segmentList )

   if srConfig.enabled:
      srTeConfigCmds = srteMode[ 'SrTePolicy.config' ]
      if srConfig.populateColoredTunnelRib:
         srTeConfigCmds.addCommand( 'rib system-colored-tunnel-rib' )
      elif saveAll:
         srTeConfigCmds.addCommand( 'no rib system-colored-tunnel-rib' )
      if toggleOptionalBsidEnabled():
         if not srConfig.specifiedBsidOnly:
            srTeConfigCmds.addCommand( 'binding-sid specified-only disabled' )
         elif saveAll:
            srTeConfigCmds.addCommand( 'no binding-sid specified-only disabled' )
      if toggleDynamicallyAssignBsidEnabled():
         if srConfig.dynamicallyAllocateBsid:
            srTeConfigCmds.addCommand( 'binding-sid allocation dynamic' )
         elif saveAll:
            srTeConfigCmds.addCommand( 'no binding-sid allocation dynamic' )
      cfgStr = None
      costMetric = srConfig.igpPrefCost
      if costMetric.dynamicMetric:
         if costMetric.costSet:
            cfgStr = "igp-cost preference %d metric dynamic" % costMetric.cost
         else:
            cfgStr = "igp-cost metric dynamic"
         cfgStr += dynamicIgpMetricAddCfg( costMetric )
      elif costMetric.costSet or saveAll:
         cfgStr = "igp-cost preference %d" % costMetric.cost
      if cfgStr:
         srTeConfigCmds.addCommand( cfgStr )
      enlpBase( srConfig, srTeConfigCmds, saveAll )
      pathSelectionTieBreakDiscriminator( srConfig, srTeConfigCmds, saveAll )
      enableGlobalSBFDEchoDefault( srConfig, srTeConfigCmds, saveAll )
      if toggleSrTeTunnelHoldDownTimerEnabled():
         saveSBFDHoldDownTime( srConfig, srTeConfigCmds, saveAll )
   for policyKey in sorted( srConfig.policy ):
      policy = srConfig.policy[ policyKey ]
      policyParams = policy.policyParams if toggleDynamicSrTeEnabled() else policy
      policyMode = srteMode[ SrTePolicyMode ].getOrCreateModeInstance( (
                                                           policy.key.endpoint,
                                                           policy.key.color ) )
      policyCmds = policyMode[ 'SrTePolicy.policy.config' ]
      if policyParams.bindingSid != MplsLabel.null:
         policyCmds.addCommand( "binding-sid %s" % policyParams.bindingSid )
      elif saveAll == True: # pylint: disable=singleton-comparison
         policyCmds.addCommand( "no binding-sid" )
      if policyParams.policyName != '':
         policyCmds.addCommand( "name %s" % policyParams.policyName )
      if policyParams.policyDescription != '':
         policyCmds.addCommand( "description %s" % policyParams.policyDescription )
      if toggleSrTePolicyIgpShortcutEnabled():
         if policyParams.configIgpShortcut:
            policyCmds.addCommand( "igp shortcut" )

      # In saveAll, we do not save the setting for preference if it is not
      # configured but the metric is. There is no way to specify to use
      # parent/default preference with a specific metric (other
      # than configuring just the metric). In this case the preference
      # is only saved at the global
      # (traffic-engineering/segment-routing) level.
      costMetric = policyParams.costMetric
      cfgStr = None
      if costMetric.dynamicMetric:
         if costMetric.costSet:
            cfgStr = "igp-cost preference %d metric dynamic" % costMetric.cost
         else:
            cfgStr = "igp-cost metric dynamic"
         cfgStr += dynamicIgpMetricAddCfg( costMetric )
      elif costMetric.costSet and costMetric.metric != costMetric.defaultIgpMetric:
         cfgStr = "igp-cost preference %d metric %d" % \
               ( costMetric.cost, costMetric.metric )
      elif costMetric.costSet:
         cfgStr = "igp-cost preference %d" % costMetric.cost
      elif costMetric.metric != costMetric.defaultIgpMetric:
         cfgStr = "igp-cost metric %d" % costMetric.metric
      if cfgStr:
         policyCmds.addCommand( cfgStr )
      for pathGroupPref in sorted( policyParams.candidatePath ):
         pathGroup = policyParams.candidatePath[ pathGroupPref ]
         candidateTypeToComputationTypeDict = {
            CandidatePathType.explicitSegmentLists: 'no',
            CandidatePathType.segmentRoutingLabelResolution: 'local',
            CandidatePathType.segmentRoutingFlexAlgoLabelResolution: 'local',
         }
         pathComputationType = candidateTypeToComputationTypeDict[
            pathGroup.candidatePathType ]

         computationTypeToMode = {
            'no': ( SrTePolicyPathGroupMode, 'SrTePolicy.policy.path.config' ),
            'local': ( SrTePolicyPathLocalComputationMode,
                        'SrTePolicy.policy.path.local.computation.config' )
         }
         pathMode = policyMode[ computationTypeToMode[ pathComputationType ][ 0 ] ].\
                                 getOrCreateModeInstance( ( policy.key.endpoint,
                                                            policy.key.color,
                                                            pathGroup.preference
                                                         ) )
         pathCmds = pathMode[ computationTypeToMode[ pathComputationType ][ 1 ] ]
         if pathComputationType == "local":
            savePathGroupLocalComputationModeCmds( pathGroup, pathCmds, saveAll )
         else:
            savePathGroupModeCmds( pathGroup, pathMode, pathCmds, policy )

      _saveSbfdConfig( policyParams, policyCmds, saveAll )

   if toggleDynamicSrTeEnabled():
      for color in sorted( srConfig.dynamicEpPolicy ):
         policy = srConfig.dynamicEpPolicy[ color ]
         policyParams = policy.policyParams
         policyMode = srteMode[ SrTeDynamicPolicyMode ].getOrCreateModeInstance( (
                                                               policy.key ) )
         policyCmds = policyMode[ 'SrTePolicy.policy.dynamic.config' ]
         for pathGroupPref in sorted( policyParams.candidatePath ):
            pathGroup = policyParams.candidatePath[ pathGroupPref ]
            pathMode = policyMode[ SrTePolicyDynamicPathLocalComputationMode ].\
                           getOrCreateModeInstance( ( None, policy.key,
                                                      pathGroup.preference ) )
            pathCmds = pathMode[
               'SrTePolicy.policy.dynamic.path.local.computation.config' ]
            savePathGroupLocalComputationModeCmds( pathGroup, pathCmds, saveAll )

@CliSave.saver( 'SrTePolicy::Config',
                'te/segmentrouting/srtepolicy/staticconfig' )
def saveConfig( srConfig, root, requireMounts, options ):
   saveAll = options.saveAll

   saveInstanceConfig( srConfig, root, saveAll )
