# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from __future__ import absolute_import, division, print_function
import CliSave
from CliSavePlugin.Security import SecurityConfigMode
from CliMode.Ssl import SslProfileMode
import DefaultSslLib
import DefaultSslProfile
import DefaultSslSelfSignedProfile
import six
from SslMonitorExpiry import getDisplayableTimeAndUnit
import Tac
from Toggles import MgmtSecurityToggleLib


Constants = Tac.Type( "Mgmt::Security::Ssl::Constants" )
NamedDhparams = Tac.Type( "Mgmt::Security::Ssl::NamedDhparams" )

class SslProfileConfigMode( SslProfileMode, CliSave.Mode ):
   def __init__( self, param ):
      SslProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

SecurityConfigMode.addChildMode( SslProfileConfigMode, 
                                 after=[ 'Mgmt.security' ] )
SslProfileConfigMode.addCommandSequence( 'Mgmt.security.ssl.profile' )
SslProfileConfigMode.addCommandSequence( 'Mgmt.security.ssl' )

@CliSave.saver( 'Mgmt::Security::Ssl::Config', 'mgmt/security/ssl/config' )
def saveSslProfileConfig( entity, root, requireMounts, options ):
   for profileName, profileConfig in six.iteritems( entity.profileConfig ):
      if not options.saveAll and (
            # pylint: disable-next=consider-using-in
            profileName == DefaultSslProfile.ARISTA_PROFILE or
            profileName == DefaultSslSelfSignedProfile.SELFSIGNED_PROFILE ):
         continue
      if profileConfig.profileType != "profileTypeSsl":
         continue
      secMode = root[ SecurityConfigMode ].getSingletonInstance()
      mode = secMode[ SslProfileConfigMode ].getOrCreateModeInstance( profileName )
      cmds = mode[ 'Mgmt.security.ssl.profile' ]

      if ( options.saveAll or not
           profileConfig.tlsVersion == Constants.allTlsVersion ):
         cmds.addCommand( 'tls versions %s' % ' '.join(
            DefaultSslLib.tlsVersionMaskToStrList( profileConfig.tlsVersion ) ) )

      if profileConfig.fipsMode:
         cmds.addCommand( 'fips restrictions' )
      elif options.saveAll:
         cmds.addCommand( 'no fips restrictions' )

      # CLI syntax transition in EL9
      # Change from 'cipher-list CIPHERS' used for TLSv1.2 and below cipher list
      # to 'cipher VERSION CIPHERS' used for all TLS versions configuration
      if entity.ciphersNewSyntax:
         if profileConfig.cipherSuite != Constants.defaultCipherSuite():
            cmds.addCommand( f'cipher v1.0 {profileConfig.cipherSuite}' )
         elif options.saveAll:
            cmds.addCommand( 'no cipher v1.0' )
         
         if profileConfig.cipherSuiteV1_3 != Constants.defaultCipherSuiteV1_3():
            cmds.addCommand( f'cipher v1.3 {profileConfig.cipherSuiteV1_3}' )
         elif options.saveAll:
            cmds.addCommand( 'no cipher v1.3' )
      else:
         if profileConfig.cipherSuite != Constants.defaultCipherSuite():
            cmds.addCommand( 'cipher-list %s' % profileConfig.cipherSuite )
         elif options.saveAll:
            cmds.addCommand( 'no cipher-list' )

      if profileConfig.certKeyPair.certFile != "":
         cmds.addCommand( 'certificate %s key %s' %
                          ( profileConfig.certKeyPair.certFile, 
                            profileConfig.certKeyPair.keyFile ) )

      for certName in sorted( profileConfig.trustedCert ):
         cmds.addCommand( 'trust certificate %s' % ( certName ) )

      for certName in sorted( profileConfig.chainedCert ):
         cmds.addCommand( 'chain certificate %s' % ( certName ) )

      for crlName in sorted( profileConfig.crl ):
         if entity.crlsNewSyntax:
            cmds.addCommand( f'revocation crl name {crlName}' )
         else:
            cmds.addCommand( f'crl {crlName}' )

      if profileConfig.ocspProfileName:
         cmds.addCommand( 'revocation ocsp profile'
                          f' {profileConfig.ocspProfileName}' )
      elif options.saveAll:
         cmds.addCommand( 'no revocation ocsp profile' )

      if profileConfig.verifyExtendedParameters:
         cmds.addCommand( 'certificate requirement extended-key-usage' )
      elif options.saveAll:
         cmds.addCommand( 'no certificate requirement extended-key-usage' )

      if profileConfig.verifyHostnameMatch:
         cmds.addCommand( 'certificate requirement hostname match' )
      elif options.saveAll:
         cmds.addCommand( 'no certificate requirement hostname match' )

      if profileConfig.verifyBasicConstraintTrust:
         cmds.addCommand( 'trust certificate requirement '
                          'basic-constraint ca true' )
      elif options.saveAll:
         cmds.addCommand( 'no trust certificate requirement '
                          'basic-constraint ca true' )

      if profileConfig.verifyTrustHostnameFqdn:
         cmds.addCommand( 'trust certificate requirement '
                          'hostname fqdn' )
      elif options.saveAll:
         cmds.addCommand( 'no trust certificate requirement '
                          'hostname fqdn' )

      if profileConfig.verifyBasicConstraintChain:
         cmds.addCommand( 'chain certificate requirement '
                          'basic-constraint ca true' )
      elif options.saveAll:
         cmds.addCommand( 'no chain certificate requirement '
                          'basic-constraint ca true' )

      if not profileConfig.verifyExpiryDateEndCert:
         cmds.addCommand( 'certificate policy expiry-date ignore' )
      elif options.saveAll:
         cmds.addCommand( 'no certificate policy expiry-date ignore' )

      if not profileConfig.verifyExpiryDateTrustCert:
         cmds.addCommand( 'trust certificate policy expiry-date ignore' )
      elif options.saveAll:
         cmds.addCommand( 'no trust certificate policy expiry-date ignore' )

      if not profileConfig.verifyExpiryDateCrl:
         if entity.ignExpDateCrlsNewSyntax:
            cmds.addCommand( 'revocation crl policy expiry-date ignore' )
         else:
            cmds.addCommand( 'crl policy expiry-date ignore' )
      elif options.saveAll:
         if entity.ignExpDateCrlsNewSyntax:
            cmds.addCommand( 'no revocation crl policy expiry-date ignore' )
         else:
            cmds.addCommand( 'no crl policy expiry-date ignore' )

      if profileConfig.verifyChainHasRootCA:
         cmds.addCommand( "chain certificate requirement include root-ca" )
      elif options.saveAll:
         cmds.addCommand( "no chain certificate requirement include root-ca" )

      if profileConfig.verifyPeerHostnameInSan:
         cmd = 'peer certificate requirement hostname match subject-alternative-name'
         if profileConfig.verifyPeerHostnameInCommonName:
            cmd += ' common-name'
         cmds.addCommand( cmd )
      elif options.saveAll:
         cmds.addCommand( 'no peer certificate requirement hostname match' )

      if MgmtSecurityToggleLib.toggleOCCertificateOIDValidationEnabled():
         if profileConfig.oidCheck:
            cmd = "peer certificate requirement oid"
            for oid, val in sorted( profileConfig.oidCheck.items() ):
               if oid == "ip":
                  cmd += ' ' + ' '.join( f'san ip {value}'
                        for value in sorted( val.split( ' ' ) ) )
               elif oid == "dns":
                  cmd += ' ' + ' '.join( f'san dns {value}'
                        for value in sorted( val.split( ' ' ) ) )
               elif oid == "uri":
                  cmd += ' ' + ' '.join( f'san uri {value}'
                        for value in sorted( val.split( ' ' ) ) )
               elif oid == "issuer":
                  cmd += f" issuer {val}"
               elif oid == "subject":
                  cmd += f" subject {val}"
               else:
                  cmd += f' san otherName {oid} {val}'

            cmds.addCommand( cmd )
         elif options.saveAll:
            cmds.addCommand( 'no peer certificate requirement oid' )

      for exp in profileConfig.commonNameRegex:
         cmds.addCommand( 'certificate common-name username regexp %s' % ( exp ) )

      if profileConfig.dhparam != NamedDhparams.generated or options.saveAll:
         cmds.addCommand( 'diffie-hellman parameters %s' % profileConfig.dhparam )

@CliSave.saver( 'Mgmt::Security::Ssl::Config', 'mgmt/security/ssl/config' )
def saveSslMonitorExpiryConfig( entity, root, requireMounts, options ):
   mode = root[ SecurityConfigMode ].getSingletonInstance()
   cmds = mode[ 'Mgmt.security' ]

   for monitor in sorted( entity.monitorConfig ):
      timeUntilExpiry, timeUntilExpiryUnit = getDisplayableTimeAndUnit(
         entity.monitorConfig[ monitor ].timeUntilExpiry )
      repeatInterval, repeatIntervalUnit = getDisplayableTimeAndUnit(
         entity.monitorConfig[ monitor ].repeatInterval )

      cmds.addCommand( "ssl monitor expiry "
                       f"{ timeUntilExpiry } { timeUntilExpiryUnit } "
                       "log interval "
                       f"{ repeatInterval } { repeatIntervalUnit }" )
