#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
import Arnet
from RoutingIntfUtils import allRoutingProtocolIntfNames
from McastCommonCliLib import mcastRoutingSupported
from McastCommonCliLib import mcast6RoutingSupported
from McastCommonCliLib import AddressFamily
from McastCommonCliLib import RoutePriority
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.MrouteCliSave import RouterMulticastAfConfigMode
from CliSavePlugin.MrouteCliSave import RouterMulticastBaseConfigMode
from CliSavePlugin.MrouteCliSave import RouterMulticastVrfConfigMode
from CliSavePlugin.MrouteCliSave import getCmdRoot
import Tac

IntfConfigMode.addCommandSequence( 'McastStatic.intf', after=[ 'Ira.ipIntf' ] )
LegacyConfig = Tac.Type( "Routing::Multicast::MulticastLegacyConfig" )

def saveIntfConfig( mcastStaticIntfConfig, intfId, root, saveAll,
                    saveAllDetail, af=AddressFamily.ipv4 ):
   if af == AddressFamily.ipunknown:
      afStr = ""
   else:
      afStr = "%s " % af
   cmd = 'multicast %sstatic' % afStr

   msic = False
   if intfId in mcastStaticIntfConfig:
      msic = True
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
   cmds = mode[ 'McastStatic.intf' ]

   if msic:
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( 'no ' + cmd )

CliSave.GlobalConfigMode.addCommandSequence( 'Ip.McastStaticRoute' )
# default-vrf "ip route multicast " commands must be saved before  
# non-default vrf's (RouterMulticastVrfConfigMode) commands.
# When there is no explicit order specifiecid b/w CliSave blocks( commandSequence
# and childMode), CliSave module saves them in ascending order of cliSave block
# names. For expample, StaticMcast block is saved after RouterMulticastVrfConfigMode.
# However, McastStatic is saved before RouterMulticastVrfConfigMode.
# Hence, "McastStaticRoute" is chosen instead of "StaticMcast" as block name for
# "ip route multicast" commands. Refer to Bug398324.
RouterMulticastBaseConfigMode.addCommandSequence( 'McastStaticRoute.config',
                                                  after=[ 'Multicast.config' ] )
RouterMulticastVrfConfigMode.addCommandSequence( 'McastStaticRoute.vrf.config',
                                               after=[ 'Multicast.vrf.config' ] )
RouterMulticastAfConfigMode.addCommandSequence( 'McastStaticFastdrop.vrf.af.config',
                                                after=[ 'Multicast.vrf.af.config' ] )
RouterMulticastAfConfigMode.addCommandSequence( 
   'McastStaticRoute.vrf.af.config', after=[ 'McastStaticFastdrop.vrf.af.config' ] )

@CliSave.saver( 'McastCommon::StaticMrouteConfigColl',
                'routing/multicast/routeconfig/static',
                requireMounts = ( 'routing/hardware/status',
                                  'interface/config/all', 'interface/status/all' ) )
def saveIpv4StaticMcastRouteConfigColl( config, root, requireMounts, options ):
   saveStaticMcastRouteConfigColl( AddressFamily.ipv4, config, root, requireMounts,
                                   options )

@CliSave.saver( 'McastCommon::StaticMrouteConfigColl',
                'routing6/multicast/routeconfig/static',
                requireMounts = ( 'routing6/hardware/status',
                                  'interface/config/all', 'interface/status/all' ) )
def saveIpv6StaticMcastRouteConfigColl( config, root, requireMounts, options ):
   saveStaticMcastRouteConfigColl( AddressFamily.ipv6, config, root, requireMounts,
                                   options )

def saveStaticMcastRouteConfigColl( af, mcastCommonConfig, root, requireMounts,
                                    options ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   # Save the default config only if the platform supports multicast routing

   if af == AddressFamily.ipv4:
      hwRoutingStatus = requireMounts[ 'routing/hardware/status' ]
      isMcastRoutingSupported = mcastRoutingSupported( None, hwRoutingStatus )
   else:
      hwRoutingStatus = requireMounts[ 'routing6/hardware/status' ]
      isMcastRoutingSupported = mcast6RoutingSupported( None, hwRoutingStatus )

   if not isMcastRoutingSupported:
      saveAll = False
      saveAllDetail = False

   if saveAllDetail:
      cfgIntfNames = allRoutingProtocolIntfNames( requireMounts,
                                                  includeEligible=True )
   elif saveAll:
      # Routing configuration is allowed on switchports as well.
      # Save configuration on all routing protocol interfaces and switchports
      # with non-default config.
      cfgIntfNames = set( allRoutingProtocolIntfNames( requireMounts ) )
      cfgIntfNames.update( mcastCommonConfig.intfConfig )
   else:
      cfgIntfNames = mcastCommonConfig.intfConfig

   for intfId in cfgIntfNames:
      intfConfig = mcastCommonConfig.intfConfig.get( intfId )
      if not intfConfig:
         if not saveAll:
            continue
      saveIntfConfig( mcastCommonConfig.intfConfig, intfId, root, saveAll,
                      saveAllDetail, af=af )

   for vrf in mcastCommonConfig.vrfConfig:
      cmds = None
      sgKeys = sorted( mcastCommonConfig.vrfConfig[ vrf ].staticMcastRoute.keys() )
      if sgKeys:
         cmds = getCmdRoot( root, vrf, af, version=LegacyConfig.ipMode,
                            prefix='McastStaticRoute' )
         for key in sgKeys:
            smr = mcastCommonConfig.vrfConfig[ vrf ].staticMcastRoute.get( key )
            saveStaticMcastRoute( smr, cmds, saveAll )
      # Saving fastdrop static status
      savedCmd = None
      if mcastCommonConfig.vrfConfig[ vrf ].fastdropEnabled:
         savedCmd = "fastdrop static"
      elif saveAllDetail:
         savedCmd = "no fastdrop static" 

      if savedCmd is not None:
         cmds = getCmdRoot( root, vrf, af, version=LegacyConfig.ipMode, 
                            prefix="McastStaticFastdrop" )
         cmds.addCommand( savedCmd )

def saveStaticMcastRoute( smr, cmds, saveAll ):
   group = smr.key.g.ipGenAddr
   source = smr.key.s.ipGenAddr
   if source.isAddrZero:
      source = ''


   cmd = "route %s %s%siif %s" % ( group, source, " " if source != "" else "",
                                   smr.iif )

   if smr.oifs:
      cmd += " oif %s" % ' '.join( Arnet.sortIntf( smr.oifs ) )

   if smr.toCpu:
      cmd += " cpu"

   if smr.iifFrr != "":
      cmd += " iifFrr %s" % smr.iifFrr

   if smr.routePriority != RoutePriority.staticPriority or saveAll:
      cmd += " priority %d" % smr.routePriority

   cmds.addCommand( cmd )
