#!/usr/bin/env python3
# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
import os, EthIntfUtil

IntfConfigMode.addCommandSequence( 'StormControl.intf' )
CliSave.GlobalConfigMode.addCommandSequence( 'Config.stormControlAggregate' )
CliSave.GlobalConfigMode.addCommandSequence( 'Config.stormControlBurst' )

@CliSave.saver( 'Bridging::StormControl::Config', 'bridging/stormcontrol/config',
                requireMounts=( 'bridging/stormcontrol/status', ) )
def saveConfig( entity, root, requireMounts, options ):
   statusDir = requireMounts[ 'bridging/stormcontrol/status' ]
   status = statusDir[ 'all' ]
   sliceDir = statusDir[ 'slice' ]
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   # Display the default config only if stormcontrol is supported by the platform.

   stormControlFlag = False
   stormControlSubIntfFlag = False
   for sliceStatus in sliceDir.values():
      if sliceStatus.stormControlSupported:
         stormControlFlag = True
      if sliceStatus.stormControlSubIntfSupported:
         stormControlSubIntfFlag = True

   if status.stormControlSubIntfSupported:
      stormControlSubIntfFlag = True

   if not status.stormControlSupported and \
      not stormControlFlag and \
      not os.environ.get( 'STORMCONTROL_CLITEST' ):
      saveAll = False
      saveAllDetail = False

   # Get the list of interfaces based on command type
   if saveAllDetail:
      cfgIntfNames = EthIntfUtil.allSwitchportNames( requireMounts,
                        includeEligible=True, 
                        includeSubIntf=stormControlSubIntfFlag )
   elif saveAll:
      # We allow L2 configuration on routed ports. List of cfgIntfNames are all
      # switchports AND routed ports for which non-default config exists.
      cfgIntfNames = set( entity.intfConfig )
      cfgIntfNames.update( EthIntfUtil.allSwitchportNames( requireMounts, 
                              includeSubIntf=stormControlSubIntfFlag ) )
   else:
      cfgIntfNames = entity.intfConfig

   defaultThreshold = Tac.Value( 'Bridging::StormControl::Threshold' )

   # Aggregate storm-control command for traffic-class
   # [ no ] storm-control bum aggregate traffic-class <tc> level pps <rate>
   if status.aggregateStormControlSupported: 
      aggregateStormControlCmd = 'storm-control bum aggregate traffic-class'
      cmds = root[ 'Config.stormControlAggregate' ]
      if saveAll:
         for tc in range( 8 ):
            aggregateTcConfig = entity.aggregateTcConfig.get( tc )
            cmd = aggregateStormControlCmd + ' %d' % tc
            if not aggregateTcConfig:
               cmd = 'no ' + cmd
            else:
               assert aggregateTcConfig.level != defaultThreshold
               cmd += ' %s' % ( aggregateTcConfig.level.formatForCli() )
            cmds.addCommand( cmd )
      else:
         for tc in entity.aggregateTcConfig:
            aggregateTcConfig = entity.aggregateTcConfig[ tc ]
            if aggregateTcConfig and aggregateTcConfig.level != defaultThreshold:
               cmd = aggregateStormControlCmd +' %d' % tc
               cmd += ' %s' % ( aggregateTcConfig.level.formatForCli() )
               cmds.addCommand( cmd )

   for intfName in cfgIntfNames:
      intfConfig = entity.intfConfig.get( intfName )
      if not intfConfig:
         if saveAll:
            intfConfig = Tac.newInstance( "Bridging::StormControl::IntfConfig",
                                          intfName )
         else:
            continue

      # display intfConfig
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfConfig.name )
      cmds = mode[ 'StormControl.intf' ]
      if intfConfig.broadcastLevel != defaultThreshold:
         cmds.addCommand(
            'storm-control broadcast %s'
            % intfConfig.broadcastLevel.formatForCli() )
      elif saveAll:
         cmds.addCommand( 'no storm-control broadcast' )

      if intfConfig.broadcastCpuLevel != defaultThreshold:
         cmds.addCommand(
            'storm-control broadcast cpu %s'
            % intfConfig.broadcastCpuLevel.formatForCli() )
      elif saveAll:
         cmds.addCommand( 'no storm-control broadcast cpu' )
      
      if intfConfig.multicastLevel != defaultThreshold:
         cmds.addCommand(
            'storm-control multicast %s'
            % intfConfig.multicastLevel.formatForCli() )
      elif saveAll:
         cmds.addCommand( 'no storm-control multicast' )

      if intfConfig.multicastCpuLevel != defaultThreshold:
         cmds.addCommand(
            'storm-control multicast cpu %s'
            % intfConfig.multicastCpuLevel.formatForCli() )
      elif saveAll:
         cmds.addCommand( 'no storm-control multicast cpu' )

      if intfConfig.uucastLevel != defaultThreshold:
         cmds.addCommand( 
            'storm-control unknown-unicast %s'
            % intfConfig.uucastLevel.formatForCli() )
      elif saveAll:
         cmds.addCommand( 'no storm-control unknown-unicast' )

      if intfConfig.uucastCpuLevel != defaultThreshold:
         cmds.addCommand(
            'storm-control unknown-unicast cpu %s'
            % intfConfig.uucastCpuLevel.formatForCli() )
      elif saveAll:
         cmds.addCommand( 'no storm-control unknown-unicast cpu' )

      if intfConfig.allLevel != defaultThreshold:
         cmds.addCommand(
            'storm-control all %s'
            % intfConfig.allLevel.formatForCli() )
      elif saveAll:
         cmds.addCommand( 'no storm-control all' )

      if intfConfig.allCpuLevel != defaultThreshold:
         cmds.addCommand(
            'storm-control all cpu %s'
            % intfConfig.allCpuLevel.formatForCli() )
      elif saveAll:
         cmds.addCommand( 'no storm-control all cpu' )

      if intfConfig.stormControlDropLoggingMode == 'on':
         cmds.addCommand( 'logging event storm-control discards' )
      elif intfConfig.stormControlDropLoggingMode == 'off':
         cmds.addCommand( 'no logging event storm-control discards' )
      elif saveAll:
         cmds.addCommand( 'logging event storm-control discards use-global' )

   # Global burst time
   # [ no ] storm-control burst time TIME ( milliseconds | microseconds )
   defaultBurst = Tac.Value( 'Bridging::StormControl::BurstInfo' )

   stormControlBurstFlag = False
   if status.stormControlBurstSupported:
      stormControlBurstFlag = True
   else :
      for sliceStatus in sliceDir.values():
         if sliceStatus.stormControlBurstSupported:
            stormControlBurstFlag = True

   if stormControlBurstFlag:
      burstStormControlCmd = 'storm-control burst time'
      cmds = root[ 'Config.stormControlBurst' ]
      name = "burstConfig"
      burstConfig = entity.burstConfig.get( name )
      if saveAll:
         if not burstConfig:
            cmd = 'no ' + burstStormControlCmd
         else:
            cmd = burstStormControlCmd + ' %d %s' % ( burstConfig.burst.burstTime,
                                                   burstConfig.burst.type )
         cmds.addCommand( cmd )
      else:
         if burstConfig and burstConfig.burst != defaultBurst:
            cmd = burstStormControlCmd + ' %d %s' % ( burstConfig.burst.burstTime,
                                                      burstConfig.burst.type )
            cmds.addCommand( cmd )

CliSave.GlobalConfigMode.addCommandSequence( 'Config.stormControlDropsLog' )

@CliSave.saver( 'Bridging::StormControl::Config', 'bridging/stormcontrol/config' )
def globalStormControlDropsLogcmds( entity, root, requireMounts, options ):
   cmds = root[ 'Config.stormControlDropsLog' ]
   if entity.stormControlDropLoggingMode == 'on':
      cmds.addCommand( 'logging event storm-control discards global' )
   elif options.saveAll:
      cmds.addCommand( 'no logging event storm-control discards global' )

   if entity.stormControlDropLogInterval:
      cmds.addCommand( 'logging event storm-control discards interval %d' % \
            entity.stormControlDropLogInterval )
   elif options.saveAll:
      cmds.addCommand( 'no logging event storm-control discards interval' )

CliSave.GlobalConfigMode.addCommandSequence( 'Config.policeKnownMcast' )

@CliSave.saver( 'Bridging::StormControl::Config', 'bridging/stormcontrol/config' )
def globalPoliceKnownMcast( entity, root, requireMounts, options ):
   cmds = root[ 'Config.policeKnownMcast' ]
   stormControlPoliceKnownMcastCmd = 'hardware storm-control known-multicast'
   if entity.policeKnownMulticast:
      if options.saveAll:
         cmds.addCommand( stormControlPoliceKnownMcastCmd )
   else:
      cmds.addCommand( 'no ' + stormControlPoliceKnownMcastCmd )
