#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliMode.StunCliMode import  StunServerProfileBaseMode
from CliSavePlugin.StunServerCliSave import StunCliSaveMode
from CliSavePlugin.StunClientCliSave import StunClientCliSaveMode
from CliSavePlugin.IntfCliSave import IntfConfigMode
from Arnet import IpGenAddr
from Toggles.StunToggleLib import toggleStunEndpointDependentNatEnabled

CliSave.GlobalConfigMode.addChildMode( StunCliSaveMode,
                                       after=[ IntfConfigMode ] )
StunCliSaveMode.addChildMode( StunClientCliSaveMode )

class StunServerProfileCliSaveMode( StunServerProfileBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      StunServerProfileBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

StunClientCliSaveMode.addChildMode( StunServerProfileCliSaveMode )

def isServerProfileDefault( profile ):
   return ( not profile.disabled and
            profile.serverAddress == IpGenAddr() and
            profile.serverPort == profile.serverPortDefault and
            profile.serverHostName == profile.serverHostNameDefault and
            profile.passwordProfile == profile.passwordProfileDefault and
            profile.sslProfile == profile.sslProfileDefault and
            profile.tunnelMode )

@CliSave.saver( 'Stun::ServerProfileConfig', 'stun/server-profile' )
def saveStunServerProfile( entity,
                                 root, requireMounts, options ):

   for name in  entity.serverProfile:
      profile = entity.serverProfile.get( name )
      stunMode = root[ StunCliSaveMode ].getSingletonInstance()
      stunClientMode = stunMode[ StunClientCliSaveMode ].getSingletonInstance()
      stunServerProfileMode = stunClientMode[
         StunServerProfileCliSaveMode ].getOrCreateModeInstance( name )
      cmds = stunServerProfileMode[ 'Stun.serverprofileConfig' ]

      # We shouldn't save any command if all the attributes in the config are set to
      # the default values and 'show running-config' wasn't called with saveAll or
      # saveAllDetail option
      if ( not ( options.saveAll or options.saveAllDetail ) and
           isServerProfileDefault( profile ) ):
         continue

      if profile.disabled:
         cmds.addCommand( 'disabled' )
      elif options.saveAll or options.saveAllDetail:
         # Display default
         cmds.addCommand( 'no disabled' )

      if profile.serverAddress != IpGenAddr():
         cmds.addCommand( "ip address %s" % profile.serverAddress.stringValue )
      elif options.saveAll or options.saveAllDetail:
         cmds.addCommand( "no ip address" )

      if profile.serverPort != profile.serverPortDefault:
         cmds.addCommand( "port %s" % profile.serverPort )
      elif options.saveAll or options.saveAllDetail:
         cmds.addCommand( "no port" )

      if profile.serverHostName != profile.serverHostNameDefault:
         cmds.addCommand( "hostname %s" % profile.serverHostName )
      # Do not save hidden commands in saveAll
      # elif options.saveAll or options.saveAllDetail:
      #    cmds.addCommand( "no hostname" )

      if profile.passwordProfile != profile.passwordProfileDefault:
         cmds.addCommand( "password profile " + profile.passwordProfile )
      elif options.saveAll or options.saveAllDetail:
         cmds.addCommand( "no password profile" )

      if profile.sslProfile != profile.sslProfileDefault:
         cmds.addCommand( "ssl profile " + profile.sslProfile )
      elif options.saveAll or options.saveAllDetail:
         cmds.addCommand( "no ssl profile" )

      if toggleStunEndpointDependentNatEnabled():
         if not profile.tunnelMode:
            cmds.addCommand( "tunnel disabled" )
         elif options.saveAll or options.saveAllDetail:
            cmds.addCommand( "no tunnel disabled" )

StunServerProfileCliSaveMode.addCommandSequence( 'Stun.serverprofileConfig' )
