#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
import Tac
import EthIntfUtil, Arnet
from CliSavePlugin import EthIntfCliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliMode.TapAgg import TapAggMode, IdentityMapMode
import Intf.IntfRange
from TypeFuture import TacLazyType
import six
from six.moves import filter

TapAggModeType = Tac.Type( "TapAgg::TapAggMode" )
isUnconnected = Tac.Type( "Arnet::EthIntfId" ).isUnconnected
FcsModeType = Tac.Type( 'TapAgg::TapAggFcsMode' )
RuntModeType = Tac.Type( 'TapAgg::TapAggRuntMode' )
TimestampHeaderFormatType = Tac.Type( 'TapAgg::TapAggTimestampHeaderFormat' )
TimestampHeaderPlacementType = Tac.Type( 'TapAgg::TapAggTimestampHeaderPlacement' )
ToolIdentityMode = Tac.Type( 'Bridging::Input::ToolIdentityMode' )
SwitchIntfConfigDefault = TacLazyType( 'Bridging::Input::SwitchIntfConfigDefault' )
GreProtocol = Tac.Type( 'TapAgg::GreProtocol' )
EthAddr = TacLazyType( "Arnet::EthAddr" )
DzGreVlanTaggingMode = Tac.Type( 'Bridging::Input::DzGreVlanTaggingMode' )
VlanId = Tac.Type( "Bridging::VlanId" )

dzGreVlanTagMapping = {
   DzGreVlanTaggingMode.dzGreDot1qPolicy: 'dot1q source dzgre policy',
   DzGreVlanTaggingMode.dzGreDot1qPort: 'dot1q source dzgre port',
   DzGreVlanTaggingMode.dzGreQinqPolicyInnerPort:
      'qinq source dzgre policy inner port',
   DzGreVlanTaggingMode.dzGreQinqPortInnerPolicy:
      'qinq source dzgre port inner policy',
}

class TapAggConfigMode( TapAggMode, CliSave.Mode ):

   def __init__( self, param ):
      self.modeKey = 'tap-agg'
      TapAggMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class IdentityMapDzGreMode( IdentityMapMode, CliSave.Mode ):

   def __init__( self, param ):
      IdentityMapMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( TapAggConfigMode, before=[ IntfConfigMode ] )
TapAggConfigMode.addCommandSequence( 'Tapagg.tapaggMode' )

TapAggConfigMode.addChildMode( IdentityMapDzGreMode )
IdentityMapDzGreMode.addCommandSequence( 'Tapagg.identityMapMode' )

IntfConfigMode.addCommandSequence( 'Tapagg.switchport' )

@CliSave.saver( 'TapAgg::CliConfig', 'tapagg/cliconfig',
                requireMounts=( 'tapagg/hwstatus', 'interface/hwstatus' ) )
def saveTapAggConfig( entity, root, requireMounts, options ):
   if not entity.enabled and not entity.noerrdisableIntf and \
          not entity.trafficSteeringMacAclMatchIp and \
          not entity.sliceProfileConfig and not options.saveAll and \
          entity.globalTruncationSize == 0 and entity.dzGreSwitchId == 0 and \
          len( entity.dzGreSwitchIdPortIdToVlanMap ) == 0 and \
          len( entity.dzGrePolicyIdToVlanMap ) == 0:
      return
   tapAggHwStatus = requireMounts[ 'tapagg/hwstatus' ]
   intfHwStatus = requireMounts[ 'interface/hwstatus' ]
   mode = root[ TapAggConfigMode ].getSingletonInstance()
   cmd = mode[ 'Tapagg.tapaggMode' ]
   profile = ''
   if entity.enabled:
      if entity.tcamProfile:
         profile = ' profile ' + entity.tcamProfile
      if entity.mode == TapAggModeType.tapAggModeExclusive:
         cmd.addCommand( 'mode exclusive' + profile )
      elif entity.mode == TapAggModeType.tapAggModeMixed:
         profileToCardMap = {}
         for ( card, profile ) in entity.sliceProfileConfig.items():
            if profile not in profileToCardMap:
               profileToCardMap[ profile ] = []
            lc = ''.join( filter( str.isdigit, card ) )
            if lc:
               profileToCardMap[ profile ].append( lc )
         for ( profile, cardList ) in sorted( six.iteritems( profileToCardMap ) ):
            command = 'mode mixed module linecard ' + ",".join( cardList )
            if profile:
               command += ' profile ' + profile
            cmd.addCommand( command )
      else:
         assert False, "unsupported mode"
   elif options.saveAll:
      cmd.addCommand( 'no mode' )

   if tapAggHwStatus.brVnTagStripSupported:
      if entity.vnTagStrip:
         cmd.addCommand( 'encapsulation vn-tag strip' )
      elif options.saveAll:
         cmd.addCommand( 'no encapsulation vn-tag strip' )

      if entity.brTagStrip:
         cmd.addCommand( 'encapsulation dot1br strip' )
      elif options.saveAll:
         cmd.addCommand( 'no encapsulation dot1br strip' )

   if tapAggHwStatus.lldpReceiveConfigurable:
      if entity.lldpReceive:
         cmd.addCommand( 'protocol lldp trap' )
      elif options.saveAll:
         cmd.addCommand( 'no protocol lldp trap' )

   for intf in Arnet.sortIntf( entity.noerrdisableIntf ):
      cmd.addCommand( 'mode exclusive no-errdisable %s' % intf )

   if tapAggHwStatus.macAclMatchIpConfigurable:
      if entity.trafficSteeringMacAclMatchIp:
         cmd.addCommand( 'service-policy type tapagg mac access-list match ip' )
      elif options.saveAll:
         cmd.addCommand( 'no service-policy type tapagg mac access-list match ip' )

   if tapAggHwStatus.dzGreEncodeSupported:
      if entity.dzGreSwitchId:
         cmd.addCommand( 'switch identity %d' % entity.dzGreSwitchId )
      elif options.saveAll:
         cmd.addCommand( 'default switch identity' )

   if tapAggHwStatus.globalIngressTruncationSizeSupported or \
      tapAggHwStatus.globalEgressTruncationSizeSupported:
      if entity.globalTruncationSize != 0:
         cmd.addCommand( 'truncation size {}'.format( entity.globalTruncationSize ) )
      elif options.saveAll:
         cmd.addCommand( 'default truncation size' )

   if intfHwStatus.timestampHeaderSupported:
      if entity.timestampReplaceSmac:
         cmd.addCommand( 'mac timestamp replace source-mac' )
      elif options.saveAll:
         cmd.addCommand( 'no mac timestamp replace source-mac' )

      if entity.timestampHeaderFormat == \
         TimestampHeaderFormatType.tapAggTimestamp48bit:
         cmd.addCommand( 'mac timestamp header format 48-bit' )
      elif entity.timestampHeaderFormat == \
           TimestampHeaderFormatType.tapAggTimestamp64bit:
         cmd.addCommand( 'mac timestamp header format 64-bit' )
      elif options.saveAll:
         cmd.addCommand( 'no mac timestamp header format' )

      if tapAggHwStatus.timestampPlacementSupported:
         if entity.timestampHeaderPlacement == \
            TimestampHeaderPlacementType.tapAggTimestampPlacementAfterL2:
            cmd.addCommand( 'mac timestamp header placement after l2' )
         elif options.saveAll:
            cmd.addCommand( 'mac timestamp header placement after source-mac' )

   if tapAggHwStatus.timestampEthTypeCustomizable:
      if entity.timestampHeaderEthType != Tac.enumValue( 'Arnet::EthType',
                                                         'ethTypeArista' ):
         cmd.addCommand( 'mac timestamp header eth-type '
                         + str( entity.timestampHeaderEthType ) )
      elif options.saveAll:
         cmd.addCommand( 'no mac timestamp header eth-type' )


   if tapAggHwStatus.fcsModeSupported:
      if entity.fcsMode == FcsModeType.tapAggFcsAppend:
         cmd.addCommand( 'mac fcs append' )
      elif options.saveAll:
         cmd.addCommand( 'default mac fcs' )

   if tapAggHwStatus.fcsErrorModeSupported:
      if entity.fcsErrorMode == FcsModeType.tapAggFcsErrorCorrect:
         cmd.addCommand( 'mac fcs-error correct' )
      elif entity.fcsErrorMode == FcsModeType.tapAggFcsErrorDiscard:
         cmd.addCommand( 'mac fcs-error discard' )
      elif entity.fcsErrorMode == FcsModeType.tapAggFcsErrorPassThrough:
         cmd.addCommand( 'mac fcs-error pass-through' )
      elif options.saveAll:
         cmd.addCommand( 'default mac fcs-error' )

   if tapAggHwStatus.runtModeSupported:
      if entity.runtMode == RuntModeType.tapAggRuntDrop:
         cmd.addCommand( 'mac runt drop' )
      elif entity.runtMode == RuntModeType.tapAggRuntPad:
         cmd.addCommand( 'mac runt pad' )
      elif entity.runtMode == RuntModeType.tapAggRuntPassThrough:
         cmd.addCommand( 'mac runt pass-through' )

   if entity.globalGreProtocol != GreProtocol.defaultProtocol:
      cmd.addCommand( 'nexthop-group encapsulation gre protocol {}'.format(
         entity.globalGreProtocol ) )
   elif options.saveAll and tapAggHwStatus.nexthopGroupSupported:
      cmd.addCommand( 'default nexthop-group encapsulation gre protocol' )

   identityMapMode = mode[ IdentityMapDzGreMode ].getSingletonInstance()

   identityMapCmd = identityMapMode[ 'Tapagg.identityMapMode' ]
   for switchIdPortId in entity.dzGreSwitchIdPortIdToVlanMap:
      identityMapCmd.addCommand( 'switch id %d port id %d vlan %d' %
                                 ( switchIdPortId.switchId, switchIdPortId.portId,
                                   entity.dzGreSwitchIdPortIdToVlanMap
                                   [ switchIdPortId ].vlanId ) )
   for policyId, ivmap in entity.dzGrePolicyIdToVlanMap.items():
      identityMapCmd.addCommand( 'policy id %d vlan %d' %
                                 ( policyId, ivmap.vlanId ) )
   if entity.dzGreDefaultSwitchIdPortIdVlan:
      identityMapCmd.addCommand(
         'switch id unmatched port id unmatched vlan %d' %
         entity.dzGreDefaultSwitchIdPortIdVlan )
   if entity.dzGreDefaultPolicyIdVlan:
      identityMapCmd.addCommand(
         'policy id unmatched vlan %d' %
         entity.dzGreDefaultPolicyIdVlan )
   if ( identityMapCmd.empty( None ) and options.saveAll and
        tapAggHwStatus.dzGreDecodeSupported ):
      cmd.addCommand( 'no identity map type dzgre' )

@CliSave.saver( 'Bridging::Input::CliConfig', 'bridging/input/config/cli',
                requireMounts=( 'tapagg/cliconfig',
                                  'tapagg/hwstatus' ) )
def saveTapAggBridgingConfig( entity, root, requireMounts, options ):
   # Display interface configs
   if options.saveAll:
      # Get all eligible switchports, since '[no] switchport' is always displayed.
      cfgIntfNames = EthIntfUtil.allSwitchportNames( requireMounts,
                                                     includeEligible=True )
   else:
      cfgIntfNames = entity.switchIntfConfig

   tapAggHwStatus = requireMounts[ 'tapagg/hwstatus' ]
   tapAggConfig = requireMounts[ 'tapagg/cliconfig' ]
   toolGroupNewFormat = tapAggConfig.toolGroupNewFormat
   for intfName in cfgIntfNames:
      # Tap agg not supported on these interfaces
      if isUnconnected( intfName ):
         continue

      intfConfig = entity.switchIntfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Bridging::Input::SwitchIntfConfig', 
                                          intfName, 'access' )
         else:
            continue  
      saveTapAggSwitchIntfConfig( intfConfig, root, options,
                                  tapAggHwStatus, toolGroupNewFormat )

#-------------------------------------------------------------------------------
# Saves the tap-agg specific state of an Bridging::SwitchIntfConfig object.
#-------------------------------------------------------------------------------
def saveTapAggSwitchIntfConfig( entity, root, options, tapAggHwStatus,
                                toolGroupNewFormat=False ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Tapagg.switchport' ]
   sicType = Tac.Type( "Bridging::Input::SwitchIntfConfig" )
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   # Display defaults on all ports for saveAllDetail and on L2 ports for saveAll. 
   saveDefault = saveAllDetail or ( saveAll and entity.enabled )
   
   if entity.switchportMode in ( 'tap', 'tool', 'tap-tool' ):
      cmds.addCommand( 'switchport mode %s' % entity.switchportMode )

   if entity.tapNativeVlan != 1 or saveDefault:
      cmds.addCommand( 'switchport tap native vlan %s'
                       % entity.tapNativeVlan )
   
   identityTuple = entity.tapIdentity
   if identityTuple.outer != 0:
      if identityTuple.inner != 0:
         assert identityTuple.outer <= VlanId.max
         cmds.addCommand( 'switchport tap identity %d inner %d'
                          % ( identityTuple.outer, identityTuple.inner ) )
      else:
         assert identityTuple.outer <= VlanId.max or \
                tapAggHwStatus.dzGreEncodeSupported
         cmds.addCommand( 'switchport tap identity %d'
                             % identityTuple.outer )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap identity' )

   macAddress = entity.tapMacAddress
   if macAddress.destMac != EthAddr.ethAddrZero:
      if macAddress.srcMac != EthAddr.ethAddrZero:
         cmds.addCommand( 'switchport tap mac-address dest %s src %s'
                          % ( macAddress.destMac, macAddress.srcMac ) )
      else:
         cmds.addCommand( 'switchport tap mac-address dest %s'
                          % macAddress.destMac )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap mac-address' )

   if entity.tapVxlanStrip:
      cmds.addCommand( 'switchport tap encapsulation vxlan strip' )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap encapsulation vxlan strip' )

   if entity.tapGreStrip:
      for key in sorted( entity.tapAllowedGreTunnel ):
         cmd = 'switchport tap encapsulation gre'
         if not key.dstIp.isAddrZero:
            cmd += ' destination %s' % key.dstIp
         if not key.srcIp.isAddrZero:
            cmd += ' source %s' % key.srcIp
         if key.protocol != key.protocolNull:
            cmd += ' protocol 0x%x' % key.protocol
         if key.featureHeaderLength:
            cmd += ' feature header length %d' % key.featureHeaderLength
         cmd += ' strip'
         if key.protocol != key.protocolNull and key.ethEncapEnabled:
            cmd += ' re-encapsulation ethernet'
         cmds.addCommand( cmd )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap encapsulation gre strip' )
      cmds.addCommand( 'no switchport tap encapsulation gre protocol strip' )

   if entity.tapGtpStrip:
      for key in sorted( entity.tapAllowedGtpTunnel ):
         dst = '' if key.dstIp.isAddrZero else f' destination { key.dstIp }'
         src = '' if key.srcIp.isAddrZero else f' source { key.srcIp }'
         cmd = f'switchport tap encapsulation gtp version 1{ dst }{ src } strip'
         cmds.addCommand( cmd )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap encapsulation gtp version 1 strip' )

   if entity.tapMplsPop:
      cmds.addCommand( 'switchport tap mpls pop all' )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap mpls pop all' )

   if entity.toolMplsPop:
      cmds.addCommand( 'switchport tool mpls pop all' )
   elif saveDefault:
      cmds.addCommand( 'no switchport tool mpls pop all' )

   if entity.toolVnStrip:
      cmds.addCommand( 'switchport tool encapsulation vn-tag strip' )
   elif saveDefault:
      cmds.addCommand( 'no switchport tool encapsulation vn-tag strip' )

   if entity.toolBrStrip:
      cmds.addCommand( 'switchport tool encapsulation dot1br strip' )
   elif saveDefault:
      cmds.addCommand( 'no switchport tool encapsulation dot1br strip' )

   if entity.tapAllowedVlans == '':
      cmds.addCommand( 'switchport tap allowed vlan none' )
   elif entity.tapAllowedVlans != SwitchIntfConfigDefault.defaultTapAllowedVlans \
         or saveDefault:
      cmds.addCommand( 'switchport tap allowed vlan %s' % 
                       entity.tapAllowedVlans )

   if entity.toolAllowedVlans == '':
      cmds.addCommand( 'switchport tool allowed vlan none' )
   elif entity.toolAllowedVlans != SwitchIntfConfigDefault.defaultToolAllowedVlans \
         or saveDefault:
      cmds.addCommand( 'switchport tool allowed vlan %s' % 
                       entity.toolAllowedVlans )

   if entity.toolIdentityTagging != 'none':
      cmds.addCommand( 'switchport tool identity %s' % entity.toolIdentityTagging )
   elif saveDefault:
      cmds.addCommand( 'no switchport tool identity' )

   if entity.toolDzGreIdentityTagging != 'dzGreNone':
      cmds.addCommand( 'switchport tool identity %s' %
                       dzGreVlanTagMapping[ entity.toolDzGreIdentityTagging ] )
   elif saveDefault:
      cmds.addCommand( 'no switchport tool identity source dzgre' )

   if tapAggHwStatus.tapHeaderRemoveSupported:
      if entity.tapHeaderRemove.size != 0:
         cmd = 'switchport tap header remove size %d' % entity.tapHeaderRemove.size
         if entity.tapHeaderRemove.preserveEth:
            cmd += ' preserve ethernet'
         cmds.addCommand( cmd )
      elif saveDefault:
         cmds.addCommand( 'no switchport tap header remove size' )

   if not entity.intfId.startswith( 'InternalRecirc' ):
      if entity.tapTruncationSize != 0:
         # Check if configured value is the special "tapTruncationUseGlobalSize"
         # value (1) and not print the size, this is to support downgrade to earlier
         # releases that didn't support configuring the size.
         if ( tapAggHwStatus.truncationSizePerIngressPortSupported and
              entity.tapTruncationSize != sicType.tapTruncationUseGlobalSize ):
            cmds.addCommand( 'switchport tap truncation %d' %
                             entity.tapTruncationSize )
         else:
            cmds.addCommand( 'switchport tap truncation' )
      elif saveDefault:
         cmds.addCommand( 'no switchport tap truncation' )

      if entity.toolTruncationSize != 0:
         if tapAggHwStatus.truncationSizePerEgressPortSupported:
            cmds.addCommand( 'switchport tool truncation %d' %
                             entity.toolTruncationSize )
         else:
            cmds.addCommand( 'switchport tool truncation' )
      elif saveDefault:
         cmds.addCommand( 'no switchport tool truncation' )

   tapGroups = list( entity.tapGroup )
   if tapGroups:
      cmds.addCommand( 'switchport tap default group %s'
                  % ' group '.join( sorted( tapGroups ) ) )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap default group' )
   
   tapNexthopGroups = list( entity.tapNexthopGroup )
   if tapNexthopGroups:
      cmds.addCommand( 'switchport tap default nexthop-group %s'
                  % ' '.join( sorted( tapNexthopGroups ) ) )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap default nexthop-group' )

   tapRawIntfs = list( entity.tapRawIntf )
   phyIntfList = [ i for i in tapRawIntfs if not i.startswith('Po') ]
   lagIntfList = [ i for i in tapRawIntfs if i.startswith('Po') ]
   if tapRawIntfs:
      if phyIntfList:
         # Due to our parser, we can not specify range for physical interfaces
         # in the startup config. For lag interfaces, it works fine
         for intf in Arnet.sortIntf( phyIntfList ):
            cmds.addCommand( 'switchport tap default interface %s' % intf )
      if lagIntfList:
         printLagList = Intf.IntfRange.intfListToCanonical( lagIntfList )
         cmds.addCommand( 'switchport tap default interface %s'
               % printLagList[ 0 ] )
   elif saveDefault:
      cmds.addCommand( 'no switchport tap default interface' )

   # For toolGroups, both comma and blank saparated list is supported
   # but only old format is generated by default. To generate new format,
   # user can run 'config convert new-syntax' command.
   toolGroups = list( entity.toolGroup )
   if toolGroups:
      if toolGroupNewFormat:
         cmds.addCommand( 'switchport tool group %s'
                  % ' group '.join( sorted( toolGroups ) ) )
      else:
         cmds.addCommand( 'switchport tool group set %s'
                       % ' '.join( sorted( toolGroups ) ) )
   elif saveDefault:
      cmds.addCommand( 'no switchport tool group' )

   if entity.toolDot1qRemoveVlans != '':
      cmds.addCommand( 'switchport tool dot1q remove outer %s'
                       % entity.toolDot1qRemoveVlans )
   elif saveDefault:
      cmds.addCommand( 'no switchport tool dot1q remove outer' )

IntfConfigMode.addCommandSequence( 'RecircIntf.timestamp',
                                   after=[ 'RecircIntf.intfConfig' ] )

@CliSave.saver( 'Interface::InternalRecircIntfConfig', 'interface/config/recirc',
                attrName='intfConfig',
                requireMounts=( 'interface/status/all', ) )
def saveRecircIntfTimestampConfig( entity, root, requireMounts, options ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'RecircIntf.timestamp' ]
   saveAll = options.saveAll

   if entity.timestampMode == 'timestampModeBeforeFcs':
      cmds.addCommand( 'mac timestamp before-fcs' )
   elif entity.timestampMode == 'timestampModeReplaceFcs':
      cmds.addCommand( 'mac timestamp replace-fcs' )
   elif entity.timestampMode == 'timestampModeHeader':
      cmds.addCommand( 'mac timestamp header' )
   else:
      assert entity.timestampMode == 'timestampModeDisabled'
      if saveAll:
         cmds.addCommand( 'no mac timestamp' )

IntfConfigMode.addCommandSequence( 'EthIntf.timestamp',
                                   after=[ EthIntfCliSave.EthIntfCmdSeq ] )

@CliSave.saver( 'Tac::Dir',
                'interface/config/eth/phy/slice',
                attrName=None,
                requireMounts=( 'interface/status/all',
                                 'interface/config/global',
                                 'interface/profile/config' ) )
def saveEthIntfTimestampConfig( baseDir, root, requireMounts, options ):
   for ethPhyIntfConfigDir in six.itervalues( baseDir ):
      for entity in six.itervalues( ethPhyIntfConfigDir.intfConfig ):
         if options.intfFilter and entity.intfId not in options.intfFilter:
            continue
         if entity.intfId.startswith( ( "Internal", "Downstream" ) ):
            # Interface type does not support timestamp config. Abort
            # BUG944 - need a more general way of doing this. Should
            # this be dictated by an attribute of the IntfConfig intead?
            continue

         mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
         cmds = mode[ 'EthIntf.timestamp' ]
         saveAll = options.saveAll

         if entity.timestampMode == 'timestampModeBeforeFcs':
            cmds.addCommand( 'mac timestamp before-fcs' )
         elif entity.timestampMode == 'timestampModeReplaceFcs':
            cmds.addCommand( 'mac timestamp replace-fcs' )
         elif entity.timestampMode == 'timestampModeHeader':
            cmds.addCommand( 'mac timestamp header' )
         else:
            assert entity.timestampMode == 'timestampModeDisabled'
            if saveAll:
               cmds.addCommand( 'no mac timestamp' )
