#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from CliMode.Classification import ( FieldSetPrefixModeBase,
                                     FieldSetL4PortModeBase,
                                     FieldSetVlanModeBase,
                                     FieldSetIntegerModeBase,
                                     FieldSetMacAddrModeBase,
                                     FieldSetServiceModeBase )

from CliMode.TrafficPolicy import ( TrafficPoliciesConfigMode,
      TrafficPoliciesModeBase,
      TrafficPoliciesVrfModeBase,
      TrafficPolicyModeBase,
      MatchRuleModeBase,
      EncapInnerModeBase,
      PacketTypeModeBase,
      EncapDzGreConfigModeBase,
      ActionsModeBase,
      ReplicateModeBase,
      TrafficPolicyConfigMode,
      FEATURE,
      FEATURE_SHORT,
      )
import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from TrafficPolicyLib import (
      actionsToCmds,
      structuredFilterToCmds,
      _protoToCmds,
)
from ClassificationLib import numericalRangeToRangeString
from Toggles.TrafficPolicyToggleLib import (
      toggleTrafficPolicyEnforceGtsmEnabled,
      toggleTrafficPolicyVxlanPacketTypeMatchEnabled
)
from Toggles.ClassificationToggleLib import (
      toggleTrafficPolicyFieldSetServiceEnabled,
)
from Arnet import EthAddr
import Tac
import six
from six.moves import map

FieldSetLimit = Tac.Type( "Classification::FieldSetLimit" )
tacMatchOption = Tac.Type( 'PolicyMap::ClassMapMatchOption' )
matchIpAccessGroup = tacMatchOption.matchIpAccessGroup
matchIpv6AccessGroup = tacMatchOption.matchIpv6AccessGroup
matchMacAccessGroup = tacMatchOption.matchMacAccessGroup
ContentsSource = Tac.Type( 'Classification::ContentsSource' )
fieldSetConfigPath = 'trafficPolicies/fieldset/input/cli'
UdfOffsetBase = Tac.Type( 'Classification::UdfOffsetBase' )

def allFieldSetSaveModes():
   return [ FSPrefixSaveMode,
            FSL4PortSaveMode,
            FSVlanSaveMode,
            FSIntegerSaveMode,
            FSMacAddrSaveMode,
            FSServiceSaveMode, ]

class TrafficPoliciesSaveMode( TrafficPoliciesModeBase, CliSave.Mode ):
   def __init__( self, param ):
      TrafficPoliciesModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( TrafficPoliciesSaveMode,
                                       after=[ IntfConfigMode ] )
TrafficPoliciesSaveMode.addCommandSequence( 'TrafficPolicy.TrafficPolicies' )
TrafficPoliciesSaveMode.addCommandSequence( 'TrafficPolicyParam' )
TrafficPoliciesSaveMode.addCommandSequence( 'LocationMatchAliasConfig' )

class TrafficPoliciesVrfSaveMode( TrafficPoliciesVrfModeBase, CliSave.Mode ):
   def __init__( self, param ):
      TrafficPoliciesVrfModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

TrafficPoliciesSaveMode.addChildMode( TrafficPoliciesVrfSaveMode,
                                      before=[ 'TrafficPolicyParam',
                                               'LocationMatchAliasConfig' ] )
TrafficPoliciesVrfSaveMode.addCommandSequence( 'TrafficPolicy.TrafficPoliciesVrf' )

class TrafficPolicySaveMode( TrafficPolicyModeBase, CliSave.Mode ):
   def __init__( self, param ):
      TrafficPolicyModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

TrafficPoliciesSaveMode.addChildMode( TrafficPolicySaveMode )
TrafficPolicySaveMode.addCommandSequence( 'TrafficPolicy.TrafficPolicy' )

# Global MatchRule used by applications which do not have matchType
class GlobalMatchRuleSaveMode( MatchRuleModeBase, CliSave.Mode ):
   def __init__( self, param ):
      MatchRuleModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )
      self.modeKey = "match-%s" % self.ruleName
      self.longModeKey = "%s-match-%s" % ( self.feature,
                                           self.ruleName )

class MatchRuleSaveMode( MatchRuleModeBase, CliSave.Mode ):
   def __init__( self, param ):
      MatchRuleModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def instanceKey( self ):
      return self.prio

   @classmethod
   def useInsertionOrder( cls ):
      # because `instanceKey` is overridden with prio
      return True

TrafficPolicySaveMode.addChildMode( MatchRuleSaveMode,
                                    after=[ 'TrafficPolicy.TrafficPolicy' ] )
MatchRuleSaveMode.addCommandSequence( 'TrafficPolicy.MatchRule' )

class ActionsSaveMode( ActionsModeBase, CliSave.Mode ):
   def __init__( self, param ):
      ActionsModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

MatchRuleSaveMode.addChildMode( ActionsSaveMode,
                                after=[ 'TrafficPolicy.MatchRule' ] )
ActionsSaveMode.addCommandSequence( 'TrafficPolicy.Actions' )

class ReplicateSaveMode( ReplicateModeBase, CliSave.Mode ):
   def __init__( self, param ):
      ReplicateModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

ActionsSaveMode.addChildMode( ReplicateSaveMode,
                              after=[ 'TrafficPolicy.Actions' ] )
ReplicateSaveMode.addCommandSequence( 'TrafficPolicy.ReplicateActions' )

class EncapInnerSaveMode( EncapInnerModeBase, CliSave.Mode ):
   def __init__( self, param ):
      EncapInnerModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

MatchRuleSaveMode.addChildMode( EncapInnerSaveMode,
                                before=[ ActionsSaveMode ],
                                after=[ 'TrafficPolicy.MatchRule' ] )
EncapInnerSaveMode.addCommandSequence( 'TrafficPolicy.EncapInner' )

class PacketTypeSaveMode( PacketTypeModeBase, CliSave.Mode ):
   def __init__( self, param ):
      PacketTypeModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

MatchRuleSaveMode.addChildMode( PacketTypeSaveMode,
                                before=[ ActionsSaveMode ],
                                after=[ 'TrafficPolicy.MatchRule' ] )

PacketTypeSaveMode.addCommandSequence( 'TrafficPolicy.PacketType' )

class EncapDzGreSaveMode( EncapDzGreConfigModeBase, CliSave.Mode ):
   def __init__( self, param ):
      EncapDzGreConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

MatchRuleSaveMode.addChildMode( EncapDzGreSaveMode,
                                before=[ ActionsSaveMode ],
                                after=[ 'TrafficPolicy.MatchRule' ] )
EncapDzGreSaveMode.addCommandSequence( 'TrafficPolicy.EncapDzGre' )

class FSPrefixSaveMode( FieldSetPrefixModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FieldSetPrefixModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

TrafficPoliciesSaveMode.addChildMode( FSPrefixSaveMode )
FSPrefixSaveMode.addCommandSequence( 'TrafficPolicy.FieldSetPrefix' )

class FSL4PortSaveMode( FieldSetL4PortModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FieldSetL4PortModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

TrafficPoliciesSaveMode.addChildMode( FSL4PortSaveMode )
FSL4PortSaveMode.addCommandSequence( 'TrafficPolicy.FieldSetL4Port' )

class FSServiceSaveMode( FieldSetServiceModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FieldSetServiceModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

TrafficPoliciesSaveMode.addChildMode( FSServiceSaveMode )
FSServiceSaveMode.addCommandSequence( 'TrafficPolicy.FieldSetService' )

class FSVlanSaveMode( FieldSetVlanModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FieldSetVlanModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

TrafficPoliciesSaveMode.addChildMode( FSVlanSaveMode )
FSVlanSaveMode.addCommandSequence( 'TrafficPolicy.FieldSetVlan' )

class FSIntegerSaveMode( FieldSetIntegerModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FieldSetIntegerModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

TrafficPoliciesSaveMode.addChildMode( FSIntegerSaveMode )
FSIntegerSaveMode.addCommandSequence( 'TrafficPolicy.FieldSetInteger' )

class FSMacAddrSaveMode( FieldSetMacAddrModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FieldSetMacAddrModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

TrafficPoliciesSaveMode.addChildMode( FSMacAddrSaveMode )
FSMacAddrSaveMode.addCommandSequence( 'TrafficPolicy.FieldSetMacAddr' )



class TrafficPolicySaver( object ):
   trafficPoliciesMode = TrafficPoliciesSaveMode
   trafficPolicyMode = TrafficPolicySaveMode
   defaultActionsMode = None # Unsupported
   matchRuleMode = MatchRuleSaveMode
   actionsRuleMode = ActionsSaveMode
   replicateMode = ReplicateSaveMode
   encapInnerMatchMode = EncapInnerSaveMode
   packetTypeMatchMode = PacketTypeSaveMode
   encapDzGreMatchMode = EncapDzGreSaveMode
   l4PortSaveAll = True

   def __init__( self, entity, root, requireMounts, options, feature,
                 shortFeature=None ):
      self.entity = entity
      self.root = root
      self.options = options
      self.trafficPolicy = None
      self.currPolicyName = None
      self.matchRule = None
      self.requireMounts = requireMounts
      self.feature = feature
      self.shortFeature = shortFeature
      if not self.shortFeature:
         self.shortFeature = feature
      self.policyMapType = 'traffic-policy'

   def trafficPolicyModeCmds( self, policyMode ):
      return policyMode[ 'TrafficPolicy.TrafficPolicy' ]

   def defaultActionsModeCmds( self, defaultActionsMode ):
      raise NotImplementedError

   def matchModeCmds( self, matchMode ):
      return matchMode[ 'TrafficPolicy.MatchRule' ]

   def innerMatchModeCmds( self, matchMode ):
      return matchMode[ 'TrafficPolicy.EncapInner' ]

   def packetTypeModeCmds( self, matchMode ):
      return matchMode[ 'TrafficPolicy.PacketType' ]

   def dzGreMatchModeCmds( self, matchMode ):
      return matchMode[ 'TrafficPolicy.EncapDzGre' ]

   def actionModeCmds( self, actionsMode ):
      return actionsMode[ 'TrafficPolicy.Actions' ]

   def replicateModeCmds( self, replicateMode ):
      return replicateMode[ 'TrafficPolicy.ReplicateActions' ]

   def _addrListToStr( self, addrs ):
      return " ".join( map( str, sorted( addrs ) ) )

   def saveTrafficPolicyAll( self ):
      if self.entity.pmapType is not None:
         allTrafficPolicy = sorted( self.entity.pmapType.pmap )
      else:
         allTrafficPolicy = []

      # we have some config, go to the right mode
      trafficPoliciesMode = \
         self.root[ self.trafficPoliciesMode ].getSingletonInstance()

      # Output any traffic policies
      for policy in allTrafficPolicy:
         self.saveTrafficPolicy( policy, trafficPoliciesMode )

   def saveTrafficPolicy( self, policyName, policiesMode ):
      self.trafficPolicy = self.entity.pmapType.pmap[ policyName ].currCfg
      self.currPolicyName = policyName
      if not self.trafficPolicy:
         return
      param = ( self.feature, self.trafficPoliciesMode, policyName )
      trafficPolicyMode = \
         policiesMode[ self.trafficPolicyMode ].getOrCreateModeInstance( param )
      self.saveDefaultActions( trafficPolicyMode )
      self.saveNamedCounters( trafficPolicyMode )
      self.savePolicyDescriptions( trafficPolicyMode )
      self.saveMatchRules( trafficPolicyMode )

   def saveDefaultActions( self, policyMode ):
      if not self.defaultActionsMode:
         return

      for matchOption, defaultActions in self.trafficPolicy.defaultAction.items():
         if matchOption == matchIpAccessGroup:
            matchType = "ipv4"
         elif matchOption == matchIpv6AccessGroup:
            matchType = "ipv6"
         elif matchOption == matchMacAccessGroup:
            matchType = "mac"
         else:
            raise NotImplementedError

         param = ( self.feature, self.trafficPolicyMode,
                   self.currPolicyName, matchType, )
         actionCmds = sorted( actionsToCmds( defaultActions.action ) )
         if actionCmds:
            defaultActionsMode = policyMode[
               self.defaultActionsMode ].getOrCreateModeInstance( param )
            cmds = self.defaultActionsModeCmds( defaultActionsMode )
            for cmd in actionCmds:
               cmds.addCommand( cmd )

   def saveNamedCounters( self, policyMode ):
      if not self.trafficPolicy.namedCounter:
         return
      cmds = self.trafficPolicyModeCmds( policyMode )
      cmd = "counter %s" % ( " ".join( sorted( self.trafficPolicy.namedCounter ) ) )
      cmds.addCommand( cmd )

   def savePolicyDescriptions( self, policyMode ):
      if not self.trafficPolicy.policyDesc:
         return
      cmds = self.trafficPolicyModeCmds( policyMode )
      cmd = f'description {self.trafficPolicy.policyDesc}'
      cmds.addCommand( cmd )

   def actionsToCmds( self, actions ):
      # This method allows a derived Saver class to save custom actions.
      # The mapping done in structuredFilterToCmds assumes all actions are
      # defined in TrafficPolicy or PolicyMap.
      actionCmds = None
      return actionCmds

   def saveMatchRules( self, policyMode ):
      for prio, ruleName in six.iteritems( self.trafficPolicy.classPrio ):
         matchRule = self.trafficPolicy.rawClassMap[ ruleName ]
         matchOption = list( matchRule.match )[ 0 ]
         matchType = ""
         if matchOption == matchIpAccessGroup:
            matchType = "ipv4"
         elif matchOption == matchIpv6AccessGroup:
            matchType = "ipv6"
         elif matchOption == matchMacAccessGroup:
            matchType = "mac"
         else:
            raise NotImplementedError
         param = ( self.feature, self.trafficPolicyMode,
                   self.currPolicyName, ruleName, matchType, prio )
         matchMode = \
               policyMode[ self.matchRuleMode ].getOrCreateModeInstance( param )
         self.saveMatchDesc( matchMode, matchRule.matchDesc )
         structuredFilter = matchRule.match[ matchOption ].structuredFilter
         classAction = self.trafficPolicy.classAction.get( ruleName )
         actionCmds = self.actionsToCmds( classAction.policyAction )
         # If a derived instance of this Saver class provides the actionCmds
         # then use those in the saved output. Otherwise, obtain the commands
         # from structuredFilterToCmds.
         if actionCmds is not None:
            structuredFilterCmds = structuredFilterToCmds( structuredFilter,
                                                           None,
                                                           matchType,
                                                           self.l4PortSaveAll )
            structuredFilterCmds[ 'actions' ] = actionCmds

         else:
            structuredFilterCmds = structuredFilterToCmds( structuredFilter,
                                                           classAction.policyAction,
                                                           matchType,
                                                           self.l4PortSaveAll )
         if structuredFilter is not None:
            self.saveEthType( structuredFilterCmds, matchMode )
            self.saveMacAddr( structuredFilterCmds, matchMode )
            self.saveVlanTag( structuredFilterCmds, matchMode )
            self.saveVlan( structuredFilterCmds, matchMode )
            self.saveSip( structuredFilterCmds, matchMode )
            self.saveFieldSet( structuredFilterCmds, 'srcPrefixSet', matchMode )
            self.saveSip( structuredFilterCmds, matchMode, lpm=True )
            self.saveFieldSet( structuredFilterCmds, 'srcLpmPrefixSet', matchMode )
            self.saveDip( structuredFilterCmds, matchMode )
            self.saveFieldSet( structuredFilterCmds, 'dstPrefixSet', matchMode )
            self.saveDip( structuredFilterCmds, matchMode, lpm=True )
            self.saveFieldSet( structuredFilterCmds, 'dstLpmPrefixSet', matchMode )
            self.saveProtoNeighborsBgp( structuredFilterCmds, matchMode )
            if toggleTrafficPolicyEnforceGtsmEnabled():
               self.saveProtoNeighborsBgpTtlSec( structuredFilterCmds, matchMode )
            self.saveProtoMatchBgp( structuredFilterCmds, matchMode )
            self.saveProtocol( structuredFilterCmds, matchMode )
            if toggleTrafficPolicyFieldSetServiceEnabled():
               self.saveServiceSet( structuredFilterCmds, matchMode )
            self.saveFragment( structuredFilterCmds, matchMode )
            self.saveOptions( structuredFilterCmds, matchMode )
            self.saveMatchTracked( structuredFilterCmds, matchMode )
            self.saveActions( structuredFilterCmds, ruleName, matchMode )
            self.saveNumericalRangeOption( structuredFilterCmds, 'ttl', matchMode )
            self.saveNumericalRangeOption( structuredFilterCmds, 'cos', matchMode )
            self.saveNumericalRangeOption( structuredFilterCmds, 'dscp', matchMode )
            self.saveNumericalRangeOption( structuredFilterCmds, 'length',
                                           matchMode )
            self.saveNexthopGroup( structuredFilterCmds, matchMode )
            self.saveLocationMatch( structuredFilterCmds, matchMode )
            self.saveAppProfileName( structuredFilterCmds, matchMode )
            self.saveDestSelf( structuredFilterCmds, matchMode )
            self.savePacketTypeMatch( structuredFilterCmds, ruleName, matchMode )
            if structuredFilter.encapField and \
               structuredFilter.encapField.encapType == 'dzgre':
               self.saveDzGreMatch( structuredFilter, structuredFilterCmds,
                                    ruleName, matchMode )
         if structuredFilter and structuredFilter.innerMatch:
            self.saveInnerMatch( structuredFilter.innerMatch,
                                 structuredFilter.encapField,
                                 ruleName, matchType, matchMode )

   def saveMatchDesc( self, matchMode, matchDesc ):
      if not matchDesc:
         return
      cmds = self.matchModeCmds( matchMode )
      cmd = f'description {matchDesc}'
      cmds.addCommand( cmd )

   def saveNexthopGroup( self, structuredFilterCmds, matchMode ):
      cmds = self.matchModeCmds( matchMode )
      command = structuredFilterCmds.get( 'nexthopGroup' )
      if command:
         cmds.addCommand( command )

   def saveAppProfileName( self, structuredFilterCmds, matchMode ):
      cmds = self.matchModeCmds( matchMode )
      command = structuredFilterCmds.get( 'appProfile' )
      if command:
         cmds.addCommand( command )

   def saveDestSelf( self, structuredFilterCmds, matchMode ):
      cmds = self.matchModeCmds( matchMode )
      command = structuredFilterCmds.get( 'dstSelf' )
      if command:
         cmds.addCommand( command )

   def saveLocationMatch( self, structuredFilterCmds, matchMode ):
      cmds = self.matchModeCmds( matchMode )
      commands = structuredFilterCmds.get( 'location', [] )
      for cmd in commands:
         cmds.addCommand( cmd )

   def saveVlanTag( self, structuredFilterCmds, matchMode ):
      cmds = self.matchModeCmds( matchMode )
      commands = structuredFilterCmds.get( 'vlanTag', [] )
      for cmd in commands:
         cmds.addCommand( cmd )

   def saveVlan( self, structuredFilterCmds, matchMode ):
      cmds = self.matchModeCmds( matchMode )
      commands = structuredFilterCmds.get( 'vlan', [] )
      for cmd in commands:
         cmds.addCommand( cmd )

   def saveMacAddr( self, structuredFilterCmds, matchMode, innerMatch=False ):
      cmds = self.matchModeCmds( matchMode )
      for attrName in [ 'srcMacAddr', 'dstMacAddr',
                        'srcMacAddrSet', 'dstMacAddrSet' ]:
         cmd = structuredFilterCmds.get( attrName )
         if cmd:
            cmds.addCommand( cmd )

   def savePacketTypeMatch( self, structuredFilterCmds, outerRuleName, matchMode ):
      if self.packetTypeMatchMode is None:
         return
      packetTypeCmds = structuredFilterCmds.get( 'packetTypes', [] )
      param = ( self.shortFeature, self.matchRuleMode, self.currPolicyName,
                outerRuleName )
      packetTypeMode = \
         matchMode[ self.packetTypeMatchMode ].getOrCreateModeInstance( param )
      cmds = self.packetTypeModeCmds( packetTypeMode )
      for pktType in packetTypeCmds:
         cmds.addCommand( pktType )

   def saveTeid( self, structuredFilterCmds, matchMode ):
      cmds = self.innerMatchModeCmds( matchMode )
      teidCmds = structuredFilterCmds.get( 'teid', [] )
      for cmd in teidCmds:
         cmds.addCommand( cmd )

   def saveEthType( self, structuredFilterCmds, matchMode ):
      cmds = self.matchModeCmds( matchMode )
      ethTypeCmds = structuredFilterCmds.get( 'ethType', [] )
      for cmd in ethTypeCmds:
         cmds.addCommand( cmd )

   def saveDzGreMatch( self, structuredFilter, structuredFilterCmds,
                       ruleName, matchMode ):
      param = ( self.shortFeature, self.matchRuleMode, ruleName )
      encapDzGreMatchMode = \
         matchMode[ self.encapDzGreMatchMode ].getOrCreateModeInstance( param )
      self.saveDzGreNumericalRangeOption( structuredFilterCmds, 'dzGreSwitchId',
                                          encapDzGreMatchMode )
      self.saveDzGreNumericalRangeOption( structuredFilterCmds, 'dzGrePortId',
                                          encapDzGreMatchMode )
      self.saveDzGreNumericalRangeOption( structuredFilterCmds, 'dzGrePolicyId',
                                          encapDzGreMatchMode )

   def saveDzGreNumericalRangeOption( self, structuredFilterCmds,
                                      optionCmd, matchMode ):
      cmds = self.dzGreMatchModeCmds( matchMode )
      cmd = structuredFilterCmds.get( optionCmd )
      if cmd:
         cmds.addCommand( cmd )

   def saveInnerMatch( self, innerMatchSf, encapField, outerRuleName, outerMatchType,
                       matchMode ):
      param = ( self.shortFeature, self.matchRuleMode, self.currPolicyName,
                outerRuleName, encapField.encapType,
                'mac' if innerMatchSf.af == 'ipunknown' else innerMatchSf.af )
      encapInnerMode = \
         matchMode[ self.encapInnerMatchMode ].getOrCreateModeInstance( param )
      innerMatchCmds = structuredFilterToCmds( innerMatchSf, None, innerMatchSf.af )
      if encapField.encapType == 'gtpv1':
         self.saveTeid( innerMatchCmds, encapInnerMode )
         self.saveSip( innerMatchCmds, encapInnerMode, innerMatch=True )
         self.saveDip( innerMatchCmds, encapInnerMode, innerMatch=True )
         self.saveFieldSet( innerMatchCmds, 'srcPrefixSet',
                            encapInnerMode, innerMatch=True )
         self.saveFieldSet( innerMatchCmds, 'dstPrefixSet',
                            encapInnerMode, innerMatch=True )
         self.saveProtocol( innerMatchCmds, encapInnerMode, innerMatch=True )
      elif toggleTrafficPolicyVxlanPacketTypeMatchEnabled() and \
            encapField.encapType == 'vxlan':
         self.saveSip( innerMatchCmds, encapInnerMode, innerMatch=True )
         self.saveDip( innerMatchCmds, encapInnerMode, innerMatch=True )
         self.saveFieldSet( innerMatchCmds, 'srcPrefixSet',
                            encapInnerMode, innerMatch=True )
         self.saveFieldSet( innerMatchCmds, 'dstPrefixSet',
                            encapInnerMode, innerMatch=True )
         self.saveProtocol( innerMatchCmds, encapInnerMode, innerMatch=True )

   def saveSip( self, structuredFilterCmds, matchMode, innerMatch=False, lpm=False ):
      if innerMatch:
         cmds = self.innerMatchModeCmds( matchMode )
      else:
         cmds = self.matchModeCmds( matchMode )
      cmd = structuredFilterCmds.get( 'sourceLpm' if lpm else 'source' )
      if cmd:
         cmds.addCommand( cmd )

   def saveDip( self, structuredFilterCmds, matchMode, innerMatch=False, lpm=False ):
      if innerMatch:
         cmds = self.innerMatchModeCmds( matchMode )
      else:
         cmds = self.matchModeCmds( matchMode )
      cmd = structuredFilterCmds.get( 'destinationLpm' if lpm else 'destination' )
      if cmd:
         cmds.addCommand( cmd )

   def saveFieldSet( self, structuredFilterCmds, prefixSet, matchMode,
                     innerMatch=False ):
      if innerMatch:
         cmds = self.innerMatchModeCmds( matchMode )
      else:
         cmds = self.matchModeCmds( matchMode )
      cmd = structuredFilterCmds.get( prefixSet )
      if cmd:
         cmds.addCommand( cmd )

   def saveActions( self, structuredFilterCmds, ruleName, matchMode ):
      actionCmds = structuredFilterCmds.get( 'actions' )
      param = ( self.feature, self.currPolicyName, ruleName,
                TrafficPolicyConfigMode )
      actionsMode = \
         matchMode[ self.actionsRuleMode ].getOrCreateModeInstance( param )
      if actionCmds:
         cmds = self.actionModeCmds( actionsMode )
         for cmd in actionCmds:
            if isinstance( cmd, tuple ):
               self.saveActionSets( cmd, ruleName, actionsMode )
            else:
               cmds.addCommand( cmd )

   def saveActionSets( self, actionCmds, ruleName, actionMode ):
      if actionCmds:
         asetName = actionCmds[ 0 ]
         param = ( self.shortFeature, self.currPolicyName, ruleName, asetName,
                   TrafficPolicyConfigMode )
         replicateMode = \
            actionMode[ self.replicateMode ].getOrCreateModeInstance( param )
         cmds = self.replicateModeCmds( replicateMode )
         for cmd in actionCmds[ 1 ]:
            cmds.addCommand( cmd )

   def saveProtocol( self, structuredFilterCmds, matchMode, innerMatch=False ):
      if innerMatch:
         cmds = self.innerMatchModeCmds( matchMode )
      else:
         cmds = self.matchModeCmds( matchMode )
      commands = structuredFilterCmds.get( 'protocol', [] )
      for cmd in commands:
         cmds.addCommand( cmd )

   def saveServiceSet( self, structuredFilterCmds, matchMode, innerMatch=False ):
      cmds = self.matchModeCmds( matchMode )
      command = structuredFilterCmds.get( 'serviceSet', [] )
      if command:
         cmds.addCommand( command )

   def saveFragment( self, structuredFilterCmds, matchMode ):
      cmds = self.matchModeCmds( matchMode )

      # Handle fragment
      cmd = structuredFilterCmds.get( 'fragment' )
      if cmd:
         cmds.addCommand( cmd )

   def saveOptions( self, structuredFilterCmds, matchMode ):
      cmds = self.matchModeCmds( matchMode )

      # Handle match options
      cmd = structuredFilterCmds.get( 'matchIpOptions' )
      if cmd:
         cmds.addCommand( cmd )

   def saveMatchTracked( self, structuredFilterCmds, matchMode ):
      cmds = self.matchModeCmds( matchMode )

      # Handle match on 'tracked' packets
      cmd = structuredFilterCmds.get( 'matchTracked' )
      if cmd:
         cmds.addCommand( cmd )

   def saveNumericalRangeOption( self, structuredFilterCmds, optionCmd, matchMode,
                                 innerMatch=False ):
      if innerMatch:
         cmds = self.innerMatchModeCmds( matchMode )
      else:
         cmds = self.matchModeCmds( matchMode )
      cmd = structuredFilterCmds.get( optionCmd )
      if cmd:
         cmds.addCommand( cmd )

   def saveProtoNeighborsBgp( self, structuredFilterCmds, matchMode ):
      cmds = self.matchModeCmds( matchMode )
      cmd = structuredFilterCmds.get( 'protoNeighborsBgp' )
      if cmd:
         cmds.addCommand( cmd )

   def saveProtoNeighborsBgpTtlSec( self, structuredFilterCmds, matchMode ):
      cmds = self.matchModeCmds( matchMode )
      cmd = structuredFilterCmds.get( 'protoNeighborsBgpTtlSec' )
      if cmd:
         cmds.addCommand( cmd )

   def saveProtoMatchBgp( self, structuredFilterCmds, matchMode ):
      cmds = self.matchModeCmds( matchMode )
      cmd = structuredFilterCmds.get( 'protoBgp' )
      if cmd:
         cmds.addCommand( cmd )

   def save( self ):
      self.saveTrafficPolicyAll()

class FieldSetSaver( object ):
   trafficPoliciesMode = TrafficPoliciesSaveMode
   trafficPoliciesConfigMode = TrafficPoliciesConfigMode
   feature = FEATURE_SHORT
   fieldSetPrefixCmdSeq = 'TrafficPolicy.FieldSetPrefix'
   fieldSetL4PortCmdSeq = 'TrafficPolicy.FieldSetL4Port'
   fieldSetVlanCmdSeq = 'TrafficPolicy.FieldSetVlan'
   fieldSetIntegerCmdSeq = 'TrafficPolicy.FieldSetInteger'
   fieldSetMacAddrCmdSeq = 'TrafficPolicy.FieldSetMacAddr'
   fieldSetServiceCmdSeq = 'TrafficPolicy.FieldSetService'

   def __init__( self, entity, root, requireMounts, options ):
      self.entity = entity
      self.root = root
      self.options = options
      self.trafficPolicy = None
      self.currPolicyName = None
      self.matchRule = None
      self.requireMounts = requireMounts
      self.urlSrcType = ContentsSource.url

   def _addrListToStr( self, addrs ):
      return " ".join( map( str, sorted( addrs ) ) )

   def saveFieldSetAll( self ):
      fieldSetIpv4Prefix = sorted( self.entity.fieldSetIpPrefix.keys() )
      fieldSetIpv6Prefix = sorted( self.entity.fieldSetIpv6Prefix.keys() )
      fieldSetL4Port = sorted( self.entity.fieldSetL4Port )
      fieldSetVlan = sorted( self.entity.fieldSetVlan )
      fieldSetInteger = sorted( self.entity.fieldSetInteger )
      fieldSetMacAddr = sorted( self.entity.fieldSetMacAddr )
      fieldSetService = sorted( self.entity.fieldSetService )
      if not fieldSetIpv4Prefix and not fieldSetIpv6Prefix and not fieldSetL4Port \
         and not fieldSetVlan and not fieldSetInteger and not fieldSetMacAddr \
         and not fieldSetService:
         # No field-sets defined
         return
      # Only adding trafficPoliciesMode if at least 1 field-set has a valid currCfg.
      if not self.hasValidCurrCfg( fieldSetIpv4Prefix, fieldSetIpv6Prefix,
                                   fieldSetL4Port, fieldSetVlan, fieldSetInteger,
                                   fieldSetMacAddr, fieldSetService ):
         return
      trafficPoliciesMode = \
         self.root[ self.trafficPoliciesMode ].getOrCreateModeInstance( None )
      self.saveAllFieldSetPrefix( fieldSetIpv4Prefix, 'ipv4', trafficPoliciesMode )
      self.saveAllFieldSetPrefix( fieldSetIpv6Prefix, 'ipv6', trafficPoliciesMode )
      self.saveAllFieldSetL4Port( fieldSetL4Port, trafficPoliciesMode )
      self.saveAllFieldSetVlan( fieldSetVlan, trafficPoliciesMode )
      self.saveAllFieldSetInteger( fieldSetInteger, trafficPoliciesMode )
      self.saveAllFieldSetMacAddr( fieldSetMacAddr, trafficPoliciesMode )
      self.saveAllFieldSetService( fieldSetService, trafficPoliciesMode )

   def hasValidCurrCfg( self, fieldSetIpv4PrefixNames, fieldSetIpv6PrefixNames,
                        fieldSetL4PortNames, fieldSetVlanNames,
                        fieldSetIntegerNames,
                        fieldSetMacAddrNames,
                        fieldSetServiceNames ):
      for fieldSetName in fieldSetIpv4PrefixNames:
         if self.entity.fieldSetIpPrefix[ fieldSetName ].currCfg:
            return True
      for fieldSetName in fieldSetIpv6PrefixNames:
         if self.entity.fieldSetIpv6Prefix[ fieldSetName ].currCfg:
            return True
      for fieldSetName in fieldSetL4PortNames:
         if self.entity.fieldSetL4Port[ fieldSetName ].currCfg:
            return True
      for fieldSetName in fieldSetVlanNames:
         if self.entity.fieldSetVlan[ fieldSetName ].currCfg:
            return True
      for fieldSetName in fieldSetIntegerNames:
         if self.entity.fieldSetInteger[ fieldSetName ].currCfg:
            return True
      for fieldSetName in fieldSetMacAddrNames:
         if self.entity.fieldSetMacAddr[ fieldSetName ].currCfg:
            return True
      for fieldSetName in fieldSetServiceNames:
         if self.entity.fieldSetService[ fieldSetName ].currCfg:
            return True
      return False

   def maybeGetExternalFieldSetSource( self, currCfg ):
      src = ''
      if self.options.saveAll and currCfg.source == ContentsSource.cli and \
         not currCfg.urlConfig:
         # No need to add source of 'static' when using CLI non-saveAll
         src = 'static'
      elif currCfg.source == self.urlSrcType and currCfg.urlConfig.url:
         src = currCfg.urlConfig.url
         if currCfg.urlConfig.vrfName:
            src += ' vrf %s' % currCfg.urlConfig.vrfName
      elif currCfg.source == ContentsSource.bgp:
         src = 'bgp'
      elif currCfg.source == ContentsSource.controller:
         src = 'controller'
      elif currCfg.source == ContentsSource.intersect:
         fieldSetNames = self._addrListToStr( currCfg.intersectFieldSets )
         src = 'and-results %s' % fieldSetNames
      return src

   def saveAllFieldSetPrefix( self, fieldSetNames, af, policiesMode ):
      for fieldSetName in fieldSetNames:
         if af == 'ipv4':
            fieldSet = self.entity.fieldSetIpPrefix[ fieldSetName ]
         else:
            fieldSet = self.entity.fieldSetIpv6Prefix[ fieldSetName ]
         # fieldSet.currCfg can be None if we try to print save config while a
         # field-set is in the process of being committed.
         if not fieldSet.currCfg:
            continue
         param = ( self.feature, fieldSet.af,
                   fieldSetName, self.trafficPoliciesConfigMode )
         fieldSetPrefixMode = \
            policiesMode[ FSPrefixSaveMode ].getOrCreateModeInstance( param )
         cmds = fieldSetPrefixMode[ self.fieldSetPrefixCmdSeq ]
         staticSource = True
         src = self.maybeGetExternalFieldSetSource( fieldSet.currCfg )
         if src:
            cmds.addCommand( 'source %s' % src )
            # We skip here because we do not want to print out
            # externally populated field-sets
            if src != 'static':
               staticSource = False

         if staticSource:
            prefixes = [ prefix.stringValue for prefix in
                         sorted( fieldSet.currCfg.prefixes ) ]
            cmd = " ".join( prefixes )
            if cmd:
               cmds.addCommand( cmd )
            if fieldSet.currCfg.exceptPrefix:
               cmd = "except "
               exceptPrefixes = \
                  [ p.stringValue for p in sorted( fieldSet.currCfg.exceptPrefix ) ]
               cmd += " ".join( exceptPrefixes )
               if cmd:
                  cmds.addCommand( cmd )

         if fieldSet.currCfg.limit != FieldSetLimit.null:
            cmds.addCommand( 'limit entries %d' % fieldSet.currCfg.limit )

   def saveAllFieldSetL4Port( self, fieldSetNames, policiesMode ):
      for fieldSetName in fieldSetNames:
         fieldSet = self.entity.fieldSetL4Port[ fieldSetName ]
         # fieldSet.currCfg can be None if we try to print save config while a
         # field-set is in the process of being committed.
         if not fieldSet.currCfg:
            continue
         param = ( self.feature, 'l4-port',
                   fieldSetName, self.trafficPoliciesConfigMode )
         fieldSetL4Port = \
            policiesMode[ FSL4PortSaveMode ].getOrCreateModeInstance( param )
         cmds = fieldSetL4Port[ self.fieldSetL4PortCmdSeq ]

         staticSource = True
         src = self.maybeGetExternalFieldSetSource( fieldSet.currCfg )
         if src:
            cmds.addCommand( 'source %s' % src )
            # We skip here because we do not want to print out
            # externally populated field-sets
            if src != 'static':
               staticSource = False

         if staticSource:
            if fieldSet.currCfg.ports:
               cmds.addCommand(
                     numericalRangeToRangeString( fieldSet.currCfg.ports ) )

         if fieldSet.currCfg.limit != FieldSetLimit.null:
            cmds.addCommand( 'limit entries expanded %d' % fieldSet.currCfg.limit )

   def saveAllFieldSetService( self, fieldSetNames, policiesMode ):
      for fieldSetName in fieldSetNames:
         fieldSet = self.entity.fieldSetService[ fieldSetName ]
         # fieldSet.currCfg can be None if we try to print save config while a
         # field-set is in the process of being committed.
         if not fieldSet.currCfg:
            continue
         param = ( self.feature, 'service',
                   fieldSetName, self.trafficPoliciesConfigMode )
         fieldSetService = \
            policiesMode[ FSServiceSaveMode ].getOrCreateModeInstance( param )
         cmds = fieldSetService[ self.fieldSetServiceCmdSeq ]
         staticSource = True
         src = self.maybeGetExternalFieldSetSource( fieldSet.currCfg )
         if src and src != 'static':
            staticSource = False
         if staticSource:
            if fieldSet.currCfg.proto:
               protoCmds = _protoToCmds( fieldSet.currCfg.proto, 'protocol', "ipv4",
                                        l4PortSaveAll=True )
               for protoCmd in protoCmds:
                  cmds.addCommand( protoCmd )
         if fieldSet.currCfg.limit != FieldSetLimit.null:
            cmds.addCommand( 'limit entries expanded %d' % fieldSet.currCfg.limit )

   def saveAllFieldSetVlan( self, fieldSetNames, policiesMode ):
      for fieldSetName in fieldSetNames:
         fieldSet = self.entity.fieldSetVlan[ fieldSetName ]
         if not fieldSet.currCfg:
            return
         param = ( self.feature, 'vlan',
                   fieldSetName, self.trafficPoliciesConfigMode )
         fieldSetVlan = \
                     policiesMode[ FSVlanSaveMode ].getOrCreateModeInstance( param )
         cmds = fieldSetVlan[ self.fieldSetVlanCmdSeq ]

         staticSource = True
         src = self.maybeGetExternalFieldSetSource( fieldSet.currCfg )
         if src:
            cmds.addCommand( 'source %s' % src )
            # We skip here because we do not want to print out
            # externally populated field-sets
            if src != 'static':
               staticSource = False

         if staticSource:
            if fieldSet.currCfg.vlans:
               cmds.addCommand(
                     numericalRangeToRangeString( fieldSet.currCfg.vlans ) )

         if fieldSet.currCfg.limit != FieldSetLimit.null:
            cmds.addCommand( 'limit entries expanded %d' % fieldSet.currCfg.limit )

   def saveAllFieldSetInteger( self, fieldSetNames, policiesMode ):
      for fieldSetName in fieldSetNames:
         fieldSet = self.entity.fieldSetInteger[ fieldSetName ]
         if not fieldSet.currCfg:
            return
         param = ( self.feature, 'integer',
                   fieldSetName, self.trafficPoliciesConfigMode )
         fieldSetInteger = \
                  policiesMode[ FSIntegerSaveMode ].getOrCreateModeInstance( param )
         cmds = fieldSetInteger[ self.fieldSetIntegerCmdSeq ]

         staticSource = True
         src = self.maybeGetExternalFieldSetSource( fieldSet.currCfg )
         if src:
            cmds.addCommand( 'source %s' % src )
            # We skip here because we do not want to print out
            # externally populated field-sets
            if src != 'static':
               staticSource = False

         if staticSource:
            if fieldSet.currCfg.integerRange:
               cmds.addCommand( numericalRangeToRangeString(
                  fieldSet.currCfg.integerRange ) )

         if fieldSet.currCfg.limit != FieldSetLimit.null:
            cmds.addCommand( 'limit entries expanded %d' % fieldSet.currCfg.limit )

   def saveAllFieldSetMacAddr( self, fieldSetNames, policiesMode ):
      for fieldSetName in fieldSetNames:
         fieldSet = self.entity.fieldSetMacAddr[ fieldSetName ]
         if not fieldSet.currCfg:
            return
         param = ( self.feature, 'mac',
                   fieldSetName, self.trafficPoliciesConfigMode )
         fieldSetMacAddr = \
               policiesMode[ FSMacAddrSaveMode ].getOrCreateModeInstance( param )
         cmds = fieldSetMacAddr[ self.fieldSetMacAddrCmdSeq ]
         src = self.maybeGetExternalFieldSetSource( fieldSet.currCfg )
         if src:
            cmds.addCommand( 'source %s' % src )
            # We skip here because we do not want to print out
            # externally populated field-sets
            if src != 'static':
               continue
         if fieldSet.currCfg.macAddrs:
            macAddresses = [ EthAddr( macAddr ).displayString for macAddr in
                             sorted( fieldSet.currCfg.macAddrs ) ]
            cmd = " ".join( macAddresses )
            cmds.addCommand( cmd )

   def save( self ):
      self.saveFieldSetAll()

class TrafficPolicyParamSaver( object ):
   trafficPoliciesMode = TrafficPoliciesSaveMode

   def __init__( self, entity, root, requireMounts, options ):
      self.entity = entity
      self.root = root
      self.options = options
      self.requireMounts = requireMounts

   def save( self ):
      def commonTrieEnabled( commonTrieInfo ):
         return commonTrieInfo.commonOuterIpv4XformTrie and \
            commonTrieInfo.commonOuterIpv6XformTrie and \
            commonTrieInfo.commonInnerIpv4XformTrie and \
            commonTrieInfo.commonInnerIpv6XformTrie

      commonTrieInfo = self.entity.commonTrieConfig

      @Tac.memoize
      def lazyCmds():
         trafficPoliciesMode = self.root[ self.trafficPoliciesMode
            ].getOrCreateModeInstance( None )
         cmds = trafficPoliciesMode[ 'TrafficPolicyParam' ]
         return cmds

      if self.entity.actionDuringUpdate == 'drop':
         lazyCmds().addCommand( "update interface default action drop" )

      if not self.entity.allowHitfulUpdate:
         lazyCmds().addCommand( "update interface hitless strict" )

      if self.entity.ingressCounterGranularity == 'counterPerInterface':
         lazyCmds().addCommand( "counter interface per-interface ingress" )

      if self.entity.counterPollInterval is not None:
         # NOTE: The CLI only accepts integer poll intervals
         lazyCmds().addCommand( "counter interface poll interval "
                                f"{int( self.entity.counterPollInterval )} seconds" )

      if self.entity.proportionalLagPolicerEnabled:
         lazyCmds().addCommand(
            "policer interface port-channel distribution proportional" )

      if commonTrieEnabled( commonTrieInfo ):
         lazyCmds().addCommand(
            "transforms interface prefix common source-destination" )

      if self.entity.prefixScaleOuter:
         lazyCmds().addCommand( "transforms interface prefix scale outer" )

class LocationMatchAliasSaver( object ):
   trafficPoliciesMode = TrafficPoliciesSaveMode

   def __init__( self, entity, root, requireMounts, options ):
      self.entity = entity
      self.root = root
      self.options = options
      self.requireMounts = requireMounts

   def save( self ):
      aliasConfig = self.entity.udf.items()

      if not aliasConfig:
         return

      trafficPoliciesMode = \
            self.root[ self.trafficPoliciesMode ].getOrCreateModeInstance( None )
      cmds = trafficPoliciesMode[ 'LocationMatchAliasConfig' ]
      for aliasName, config in sorted( aliasConfig ):
         udfAnchorConfig = self.entity.udf[ aliasName ]
         aliasCmd = 'location ' + udfAnchorConfig.alias + ' '
         if udfAnchorConfig.base == UdfOffsetBase.UdfOffsetBaseL3HeaderStart:
            udfBase = 'ip-header-start'
         elif udfAnchorConfig.base == UdfOffsetBase.UdfOffsetBaseL3HeaderEnd:
            udfBase = 'ip-header-end'
         elif udfAnchorConfig.base == UdfOffsetBase.UdfOffsetBasePacketStart:
            udfBase = 'packet-start'
         elif udfAnchorConfig.base == UdfOffsetBase.UdfOffsetBaseEncapGtpIpHeaderEnd:
            udfBase = 'encapsulation gtp ip-header-end'
         elif udfAnchorConfig.base == \
                                    UdfOffsetBase.UdfOffsetBaseEncapGtpIpHeaderStart:
            udfBase = 'encapsulation gtp ip-header-start'
         elif udfAnchorConfig.base == \
                                    UdfOffsetBase.UdfOffsetBaseEncapVxlanIpHeaderEnd:
            udfBase = 'encapsulation vxlan ip-header-end'
         elif udfAnchorConfig.base == \
                                  UdfOffsetBase.UdfOffsetBaseEncapVxlanIpHeaderStart:
            udfBase = 'encapsulation vxlan ip-header-start'
         elif udfAnchorConfig.base == UdfOffsetBase.UdfOffsetBaseMetadata:
            udfBase = 'metadata'
         aliasCmd += udfBase + ' offset ' + str( udfAnchorConfig.offset )
         udfLength = 16
         if udfAnchorConfig.length == 'udf32b':
            udfLength = 32
         aliasCmd += ' bytes length ' + str( udfLength ) + ' bits'
         cmds.addCommand( aliasCmd )

class TrafficPoliciesVrfSaver:
   trafficPoliciesMode = TrafficPoliciesSaveMode
   trafficPoliciesVrfMode = TrafficPoliciesVrfSaveMode
   vrfModeToken = 'TrafficPolicy.TrafficPoliciesVrf'

   def __init__( self, entity, root, requireMounts, options, feature ):
      self.entity = entity
      self.root = root
      self.options = options
      self.requireMounts = requireMounts
      self.feature = feature

   def save( self ):
      trafficPoliciesMode = self.root[ self.trafficPoliciesMode ].\
               getSingletonInstance()
      vrfConfigs = dict( self.entity.vrfConfig.trafficPolicies.items() )
      for vrf in sorted( self.entity.vrf ):
         param = ( vrf, TrafficPoliciesConfigMode, self.feature, self.entity, None )
         vrfMode = trafficPoliciesMode[
               self.trafficPoliciesVrfMode ].getOrCreateModeInstance( param )
         if vrf not in vrfConfigs:
            continue
         cmds = vrfMode[ self.vrfModeToken ]
         vrfPolicyName = vrfConfigs[ vrf ].policyName
         cmd = f'traffic-policy input {vrfPolicyName} physical'
         cmds.addCommand( cmd )

@CliSave.saver( 'TrafficPolicy::TrafficPolicyVrfConfig',
                'trafficPolicies/vrf/input/cli' )
def saveVrfConfig( entity, root, requireMounts, options ):
   cliDumper = TrafficPoliciesVrfSaver( entity, root, requireMounts, options,
                                        FEATURE )
   cliDumper.save()

# Save all field-set
@CliSave.saver( 'Classification::FieldSetConfig', fieldSetConfigPath )
def saveFieldSetConfig( entity, root, requireMounts, options ):
   cliDumper = FieldSetSaver( entity, root, requireMounts, options )
   cliDumper.save()

# Save all traffic-policy
@CliSave.saver( 'TrafficPolicy::TrafficPolicyConfig', 'trafficPolicies/input/cli' )
def saveTrafficPolicyConfig( entity, root, requireMounts, options ):
   cliDumper = TrafficPolicySaver( entity, root, requireMounts, options,
                                   FEATURE, FEATURE_SHORT )
   cliDumper.save()

# Save update default action
@CliSave.saver( 'TrafficPolicy::TrafficPolicyIntfParamConfig',
                'trafficPolicies/param/config/interface' )
def saveTrafficPolicyParamConfig( entity, root, requireMounts, options ):
   cliDumper = TrafficPolicyParamSaver( entity, root, requireMounts, options )
   cliDumper.save()

# Location Alias anchor config (udf anchor config )
@CliSave.saver( 'Classification::UdfConfig',
                'trafficPolicies/udfconfig/cli' )
def saveLocationMatchAliasConfig( entity, root, requireMounts, options ):
   cliDumper = LocationMatchAliasSaver( entity, root, requireMounts, options )
   cliDumper.save()

# -------------------------------------------------------------------------------
# Saves the state of an PolicyMap::IntfConfig object vni field.
# -------------------------------------------------------------------------------
@CliSave.saver( 'PolicyMap::IntfConfig',
                'trafficPolicies/intf/input/config/vxlan',
                requireMounts=( 'interface/config/eth/vxlan', ) )
def saveTrafficPolicyOnVniConfig( entity, root, requireMounts, options ):

   vtiConfigDir = requireMounts[ 'interface/config/eth/vxlan' ]

   for vti in vtiConfigDir.vtiConfig:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( vti )
      cmds = mode[ 'VxlanIntf.config' ]
      for vni in sorted( entity.vni ):
         pmapName = entity.vni[ vni ]
         cmds.addCommand( f'vxlan vni {vni} traffic-policy input {pmapName}' )
