# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliSave
from CliMode.TunnelProfile import (
      TunnelProfilesBaseMode,
      TunnelProfilesEncapTypeBaseMode,
      TunnelProfilesTemplateBaseMode,
      TunnelProfilePolicyBaseMode,
      TunnelProfilePolicySeqBaseMode,
      )
import Toggles.TunnelToggleLib

class TunnelProfilesConfigMode( TunnelProfilesBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      TunnelProfilesBaseMode.__init__( self )
      CliSave.Mode.__init__( self, param )

class TunnelProfilesEncapTypeConfigMode( TunnelProfilesEncapTypeBaseMode,
                                         CliSave.Mode ):
   def __init__( self, encapType ):
      TunnelProfilesEncapTypeBaseMode.__init__( self, encapType )
      CliSave.Mode.__init__( self, encapType )

class TunnelProfilesTemplateConfigMode( TunnelProfilesTemplateBaseMode,
                                        CliSave.Mode ):
   def __init__( self, encapTypeAndTemplateName ):
      TunnelProfilesTemplateBaseMode.__init__( self, encapTypeAndTemplateName )
      CliSave.Mode.__init__( self, encapTypeAndTemplateName )

class TunnelProfilePolicyConfigMode( TunnelProfilePolicyBaseMode,
                                     CliSave.Mode ):
   def __init__( self, encapTypeAndPolicyName ):
      TunnelProfilePolicyBaseMode.__init__( self, encapTypeAndPolicyName )
      CliSave.Mode.__init__( self, encapTypeAndPolicyName )

class TunnelProfilePolicySeqConfigMode( TunnelProfilePolicySeqBaseMode,
                                        CliSave.Mode ):
   def __init__( self, policyNameAndSeqno ):
      TunnelProfilePolicySeqBaseMode.__init__( self, policyNameAndSeqno )
      CliSave.Mode.__init__( self, policyNameAndSeqno )

CliSave.GlobalConfigMode.addChildMode( TunnelProfilesConfigMode )
TunnelProfilesConfigMode.addChildMode( TunnelProfilesEncapTypeConfigMode )
TunnelProfilesEncapTypeConfigMode.addChildMode( TunnelProfilePolicyConfigMode )
TunnelProfilePolicyConfigMode.addChildMode( TunnelProfilePolicySeqConfigMode )
TunnelProfilePolicySeqConfigMode.addCommandSequence(
      "TunnelProfilePolicySeq.config" )
if Toggles.TunnelToggleLib.toggleDynamicGueTunnelsTemplatesEnabled():
   TunnelProfilesEncapTypeConfigMode.addChildMode(
      TunnelProfilesTemplateConfigMode, before=[ TunnelProfilePolicyConfigMode ] )
   TunnelProfilesTemplateConfigMode.addCommandSequence(
      'TunnelProfileTemplate.config' )

@CliSave.saver( "Tunnel::TunnelProfile::TunnelProfilesConfig",
                "tunnel/tunnelProfiles/config" )
def saveTunnelProfileCliConfig( entity, root, requireMounts, options ):
   saveAllOrDetail = options.saveAll or options.saveAllDetail
   if not Toggles.TunnelToggleLib.toggleDynamicGueTunnelsEnabled() or \
         ( not entity.encapTypeConfig and not saveAllOrDetail ):
      return
   tunnelProfilesMode = root[ TunnelProfilesConfigMode
         ].getOrCreateModeInstance( None )
   for encapType, encapTypeConfig in entity.encapTypeConfig.items():
      if not encapTypeConfig.templateConfig and \
         not encapTypeConfig.policyConfig and not saveAllOrDetail:
         continue
      encapTypeMode = tunnelProfilesMode[ TunnelProfilesEncapTypeConfigMode
            ].getOrCreateModeInstance( encapType )
      if Toggles.TunnelToggleLib.toggleDynamicGueTunnelsTemplatesEnabled():
         for templateName, templateConfig in encapTypeConfig.templateConfig.items():
            templateMode = encapTypeMode[
               TunnelProfilesTemplateConfigMode ].getOrCreateModeInstance(
                  ( encapType, templateName ) )
            cmds = templateMode[ 'TunnelProfileTemplate.config' ]

            sourceIp = templateConfig.sourceIp
            if sourceIp:
               cmds.addCommand( f'source {sourceIp}' )
            elif saveAllOrDetail:
               cmds.addCommand( 'no source' )
            tos = templateConfig.tos
            if tos.valid():
               cmds.addCommand( f"tos { tos.value }" )
            elif saveAllOrDetail:
               cmds.addCommand( "no tos" )
            if templateConfig.ttl.isSet:
               cmds.addCommand( f"ttl { templateConfig.ttl.value }" )
            elif saveAllOrDetail:
               cmds.addCommand( "no ttl" )

      for policyName, policyConfig in encapTypeConfig.policyConfig.items():
         policyMode = encapTypeMode[ TunnelProfilePolicyConfigMode
               ].getOrCreateModeInstance( ( encapType, policyName ) )
         for seqno, policyConfigEntry in policyConfig.policyConfigEntry.items():
            policySeqMode = policyMode[ TunnelProfilePolicySeqConfigMode
                  ].getOrCreateModeInstance( ( policyName, seqno ) )
            cmds = policySeqMode[ "TunnelProfilePolicySeq.config" ]
            prefix = policyConfigEntry.matchDestinationIpPrefix
            prefixList = policyConfigEntry.matchDestinationIpPrefixList
            templateName = policyConfigEntry.templateName
            sourceIp = policyConfigEntry.sourceIp
            tos = policyConfigEntry.tos
            if prefix:
               cmds.addCommand( f"match destination ip {prefix}" )
            elif prefixList:
               cmds.addCommand( f"match destination ip prefix-list {prefixList}" )
            elif saveAllOrDetail:
               cmds.addCommand( "no match destination ip" )

            if templateName:
               cmds.addCommand( f'template {templateName}' )
            elif saveAllOrDetail:
               cmds.addCommand( 'no template' )

            if sourceIp:
               cmds.addCommand( f"source {sourceIp}" )
            elif saveAllOrDetail:
               cmds.addCommand( "no source" )

            if tos.valid():
               cmds.addCommand( f"tos { tos.value }" )
            elif saveAllOrDetail:
               cmds.addCommand( "no tos" )

            if policyConfigEntry.ttl.isSet or saveAllOrDetail:
               cmds.addCommand( f"ttl { policyConfigEntry.ttl.value }" )
