# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing
import CliSave
from CliMode.TwampLight import (
   MonitorTwampModeBase,
   TwampLightModeBase,
   TwampLightProfileMode,
   TwampLightDefaultsMode
)

t0 = Tracing.trace0

class MonitorTwampCliSaveMode( MonitorTwampModeBase, CliSave.Mode ):
   def __init__( self, param ):
      MonitorTwampModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class TwampLightCliSaveMode( TwampLightModeBase, CliSave.Mode ):
   def __init__( self, param ):
      TwampLightModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class TwampLightReflectorCliSaveMode( TwampLightProfileMode, CliSave.Mode ):
   def __init__( self, param ):
      TwampLightProfileMode.__init__( self, ( param, "reflector" ) )
      CliSave.Mode.__init__( self, ( param, "reflector" ) )

class TwampLightSenderCliSaveMode( TwampLightProfileMode, CliSave.Mode ):
   def __init__( self, param ):
      TwampLightProfileMode.__init__( self, ( param, "sender" ) )
      CliSave.Mode.__init__( self, ( param, "sender" ) )

class TwampLightReflectorDefaultsCliSaveMode( TwampLightDefaultsMode, CliSave.Mode ):
   def __init__( self, param ):
      TwampLightDefaultsMode.__init__( self, "reflector" )
      CliSave.Mode.__init__( self, "reflector" )

   def skipIfEmpty( self ):
      return True

class TwampLightSenderDefaultsCliSaveMode( TwampLightDefaultsMode, CliSave.Mode ):
   def __init__( self, param ):
      TwampLightDefaultsMode.__init__( self, "sender" )
      CliSave.Mode.__init__( self, "sender" )

   def skipIfEmpty( self ):
      return True

def registerAllCommands():
   CliSave.GlobalConfigMode.addChildMode( MonitorTwampCliSaveMode )
   MonitorTwampCliSaveMode.addCommandSequence( 'MonitorTwamp' )

   MonitorTwampCliSaveMode.addChildMode( TwampLightCliSaveMode,
                                         after=[ 'MonitorTwamp' ] )
   TwampLightCliSaveMode.addCommandSequence( 'TwampLight' )

   TwampLightCliSaveMode.addChildMode( TwampLightReflectorDefaultsCliSaveMode,
                                       after=[ 'TwampLight' ] )
   TwampLightReflectorDefaultsCliSaveMode.addCommandSequence( 'ReflectorDefaults' )

   TwampLightCliSaveMode.addChildMode( TwampLightSenderDefaultsCliSaveMode,
                                       after=[ 'TwampLight' ] )
   TwampLightSenderDefaultsCliSaveMode.addCommandSequence( 'SenderDefaults' )

   TwampLightCliSaveMode.addChildMode(
      TwampLightSenderCliSaveMode, after=[ TwampLightSenderDefaultsCliSaveMode ] )
   TwampLightSenderCliSaveMode.addCommandSequence( 'SenderProfile' )

   TwampLightCliSaveMode.addChildMode(
      TwampLightReflectorCliSaveMode, after=[ TwampLightSenderDefaultsCliSaveMode ] )
   TwampLightReflectorCliSaveMode.addCommandSequence( 'ReflectorProfile' )

registerAllCommands()

@CliSave.saver( 'Twamp::Light::Config', 'monitor/twamp/twampLight/config' )
def twampLightCliConfig( config, root, requireMounts, options ):
   monitorTwamp = root[ MonitorTwampCliSaveMode ].getSingletonInstance()

   if not config.reflectorActive:
      return

   twampLight = monitorTwamp[ TwampLightCliSaveMode ].getSingletonInstance()
   saveAll = options.saveAll

   reflectorDefaults = twampLight[ TwampLightReflectorDefaultsCliSaveMode ]\
      .getSingletonInstance()
   reflectorDefaultsCmds = reflectorDefaults[ 'ReflectorDefaults' ]

   if config.reflectorListenPort != config.reflectorListenPortDefault or saveAll:
      reflectorDefaultsCmds.addCommand(
         f'listen port {config.reflectorListenPort}' )

   senderDefaults = twampLight[ TwampLightSenderDefaultsCliSaveMode ]\
      .getSingletonInstance()
   senderDefaultsCmds = senderDefaults[ 'SenderDefaults' ]

   if ( config.senderDestinationPort != config.senderDestinationPortDefault or
        saveAll ):
      senderDefaultsCmds.addCommand(
         f'destination port {config.senderDestinationPort}' )

   if config.senderSourcePort != config.senderSourcePortDefault or saveAll:
      senderDefaultsCmds.addCommand( f'source port {config.senderSourcePort}' )

   for profileName in config.senderProfile:
      senderProfile = twampLight[ TwampLightSenderCliSaveMode ]\
         .getOrCreateModeInstance( profileName )
      senderCmds = senderProfile[ 'SenderProfile' ]
      profile = config.senderProfile[ profileName ]

      if profile.interval != profile.intervalDefault or saveAll:
         senderCmds.addCommand( f'measurement interval {profile.interval} seconds' )

      if profile.samples != profile.samplesDefault or saveAll:
         senderCmds.addCommand( f'measurement samples {profile.samples}' )

      if profile.significance != profile.significanceDefault \
         or profile.offset != profile.offsetDefault \
         or saveAll:
         cmd = ( f'significance {profile.significance} microseconds' +
                 f' offset {profile.offset} microseconds' )
         senderCmds.addCommand( cmd )
