# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliSave import (
   Mode,
   GlobalConfigMode,
   saver,
)
from CliMode.Vpls import (
   RouterVplsMode,
   VplsInstanceMode,
   VplsVlanOrBundleMode,
   VplsGroupMode,
)
from CliSavePlugin.PseudowireCliSave import PatchPanelConfigMode
from RouteMapLib import (
   CommunityType,
   commValueToPrint,
)
from Toggles.EvpnLibToggleLib import toggleEvpnVplsInteropEnabled
from Toggles.PseudowireToggleLib import toggleVplsBgpSignalingEnabled

import Tac

class RouterVplsConfigSaveMode( RouterVplsMode, Mode ):
   def __init__( self, param ):
      RouterVplsMode.__init__( self )
      Mode.__init__( self, param )

class VplsInstanceConfigSaveMode( VplsInstanceMode, Mode ):
   def __init__( self, vplsName ):
      VplsInstanceMode.__init__( self, vplsName )
      Mode.__init__( self, vplsName )

class VplsVlanOrBundleConfigSaveMode( VplsVlanOrBundleMode, Mode ):
   def __init__( self, param ):
      vplsName, bundle, vlanId = param
      VplsVlanOrBundleMode.__init__( self, vplsName, bundle, vlanId )
      Mode.__init__( self, vlanId )

class VplsGroupConfigSaveMode( VplsGroupMode, Mode ):
   def __init__( self, param ):
      vplsName, groupName, signalingProtocol, splitHorizon = param
      VplsGroupMode.__init__( self, vplsName, groupName, signalingProtocol,
                              splitHorizon )
      # Use signaling+groupName as the key, so VPLS groups are sorted by
      # signaling type when displayed
      Mode.__init__( self, signalingProtocol + groupName )

GlobalConfigMode.addChildMode( RouterVplsConfigSaveMode,
                               after=[ PatchPanelConfigMode ] )
RouterVplsConfigSaveMode.addChildMode( VplsInstanceConfigSaveMode )
VplsInstanceConfigSaveMode.addCommandSequence( 'Vpls.macWithdrawInterface' )
VplsInstanceConfigSaveMode.addChildMode( VplsVlanOrBundleConfigSaveMode )
VplsVlanOrBundleConfigSaveMode.addCommandSequence( 'Vpls.vlanOrBundleConfig' )
VplsInstanceConfigSaveMode.addChildMode(
                      VplsGroupConfigSaveMode,
                      after=[ VplsVlanOrBundleConfigSaveMode ] )
VplsGroupConfigSaveMode.addCommandSequence( 'Vpls.vplsGroupConfig' )

def saveVlanOrBundleConfig( instance, vplsInstance, instMode ):
   if not vplsInstance.vlan:
      return
   isBundle = vplsInstance.vlanBundleStrForCliSave != ''
   vlanId = None
   if isBundle:
      vlanId = vplsInstance.vlanBundleStrForCliSave
   else:
      vlanId = next( iter( vplsInstance.vlan ) )
   vlanMode = instMode[ VplsVlanOrBundleConfigSaveMode ].\
      getOrCreateModeInstance( ( instance, isBundle, vlanId ) )
   cmds = vlanMode[ 'Vpls.vlanOrBundleConfig' ]

   tagForFormat = None
   if isBundle:
      tagForFormat = vplsInstance.normalizedTagStrForCliSave
   else:
      firstVlanTag = list( vplsInstance.vlan.values() )[ 0 ].normalizedTag
      if firstVlanTag.isSet:
         tagForFormat = firstVlanTag.value
   if tagForFormat:
      cmds.addCommand( f'encapsulation dot1q {tagForFormat}' )

def saveGroupConfig( instance, vplsInstance, instMode, saveAll ):
   for group in vplsInstance.ldpPseudowireGroup:
      groupConfig = vplsInstance.ldpPseudowireGroup[ group ]
      groupMode = instMode[ VplsGroupConfigSaveMode ].\
         getOrCreateModeInstance( ( instance, group, 'ldp',
                                    groupConfig.splitHorizon ) )
      cmds = groupMode[ 'Vpls.vplsGroupConfig' ]
      for pwName in groupConfig.pseudowire:
         pwConfig = groupConfig.pseudowire[ pwName ]
         cmd = [ 'pseudowire', pwName ]
         if pwConfig.secondary:
            cmd.append( 'secondary' )
         if pwConfig.precedence:
            cmd.append( str( pwConfig.precedence ) )
         cmds.addCommand( ' '.join( cmd ) )

      # Save pseudowire profile
      profileCmdPrefix = "bgp auto-discovery profile"
      if groupConfig.pseudowireProfile:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( profileCmdPrefix + " {} id {}".format(
            groupConfig.pseudowireProfile.name,
            groupConfig.pseudowireProfile.rawId or commValueToPrint(
               groupConfig.pseudowireProfile.l2VpnId,
               commType=CommunityType.communityTypeExtended,
               withTypeStr=False ) ) )
      elif saveAll:
         cmds.addCommand( "no " + profileCmdPrefix )

      if groupConfig.macWithdrawalTriggerPseudowire:
         cmds.addCommand( 'mac withdrawal trigger pseudowire' )
      elif saveAll:
         cmds.addCommand( 'no mac withdrawal trigger pseudowire' )
      prop = groupConfig.macWithdrawalPropagation
      if prop != 'locallyTriggered':
         propStr = { 'locallyTriggeredAndReceived': 'received',
                     'suppress': 'suppressed' }[ prop ]
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'mac withdrawal propagation {prop}'.format(
                                            prop=propStr ) )
      elif saveAll:
         cmds.addCommand( 'no mac withdrawal propagation' )

      if toggleEvpnVplsInteropEnabled():
         if groupConfig.forwardingPreferenceEvpn:
            cmds.addCommand( 'forwarding preference evpn' )
         elif saveAll and groupConfig.splitHorizon:
            cmds.addCommand( 'no forwarding preference evpn' )

   # Generate the commands for all the configured BGP VPLS groups
   if toggleVplsBgpSignalingEnabled():
      for group in vplsInstance.bgpPseudowireGroup:
         groupConfig = vplsInstance.bgpPseudowireGroup[ group ]
         groupMode = instMode[ VplsGroupConfigSaveMode ].\
            getOrCreateModeInstance( ( instance, group, 'bgp',
                                       groupConfig.splitHorizon ) )

@saver( 'Pseudowire::VplsConfig', 'vpls/config' )
def saveConfig( vplsConfig, root, requireMounts, options ):
   saveAll = options.saveAll
   if not vplsConfig.instance and not saveAll:
      return
   mode = root[ RouterVplsConfigSaveMode ].getOrCreateModeInstance( None )
   for instance in vplsConfig.instance:
      instMode = mode[ VplsInstanceConfigSaveMode ].\
                   getOrCreateModeInstance( instance )
      cmds = instMode[ 'Vpls.macWithdrawInterface' ]
      vplsInstance = vplsConfig.instance[ instance ]
      if vplsInstance.macWithdrawalTriggerInterface:
         cmds.addCommand( 'ldp mac withdrawal trigger interface' )
      elif saveAll:
         cmds.addCommand( 'no ldp mac withdrawal trigger interface' )

      saveVlanOrBundleConfig( instance, vplsInstance, instMode )
      saveGroupConfig( instance, vplsInstance, instMode, saveAll )
