#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Tac
import CliSave
import Arnet
import Toggles.VxlanToggleLib
import collections
from CliSavePlugin.IntfCliSave import IntfConfigMode, saveIntfConfig
from Ethernet import convertMacAddrCanonicalToDisplay
from VxlanVniLib import VniFormat
from VxlanCliLib import canonicalVtepString
from VxlanCliLib import icmpHelperInputConfigClientName
import six
import MultiRangeRule

IntfConfigMode.addCommandSequence( 'VxlanIntf.config' )
IntfConfigMode.addCommandSequence( 'Vxlan.ecnConfig',
      after=[ 'VxlanIntf.config' ] )
IntfConfigMode.addCommandSequence( 'Vxlan.outerDscpFromInnerDscpConfig',
      after=[ 'VxlanIntf.config' ] )
IntfConfigMode.addCommandSequence( 'Vxlan.floodsetConfig',
      after=[ 'VxlanIntf.config' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'VxlanGlobal.config' )
CliSave.GlobalConfigMode.addCommandSequence( 'Vxlan.staticMac',
                                             after=[ IntfConfigMode ] )

#-------------------------------------------------------------------------------
# Saves the state of an Interface::VtiConfig object.
#-------------------------------------------------------------------------------
# pylint: disable-msg=C0322
# XXX Workaround for a pylint bug with decorators.
@CliSave.saver( 'Vxlan::VtiConfig', 'interface/config/eth/vxlan',
                attrName='vtiConfig',
                requireMounts=( 'vxlancontroller/config',
                                  'bridging/hwcapabilities',
                                  'interface/status/all',
                                  'vxlan/config',
                                  'interface/config/eth/vxlan',
                                  'vxlan/clientDir/cli',
                                  'vxlan/vtepValidationStatus/cli',
                                  'tepicmphelper/input/config' ),
                commandTagSupported=True )
def saveVtiConfig( entity, root, requireMounts, options ):
   # Cleanup and bail out on inconsistent state where only one of VxlanConfig and
   # VtiConfig exists
   vxlanConfigDir = requireMounts[ 'vxlan/config' ]
   vtiConfigDir = requireMounts[ 'interface/config/eth/vxlan' ]
   if entity.intfId not in vxlanConfigDir.vxlanConfig:
      del vtiConfigDir.vtiConfig[ entity.intfId ]
      return

   # handle baseclass (Arnet::IntfConfig) first
   saveIntfConfig( entity, root, requireMounts, options )

   saveAll = options.saveAll

   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'VxlanIntf.config' ]

   if entity.srcIpIntf != '':
      cmds.addCommand( 'vxlan source-interface %s' %
                       entity.srcIpIntf )
   elif saveAll:
      cmds.addCommand( 'no vxlan source-interface' )

   vtepValidationStatusCli = requireMounts[ 'vxlan/vtepValidationStatus/cli' ]
   if vtepValidationStatusCli.remoteVtepAddr:
      sipVtepsStrList = \
            [ v.stringValue for v in vtepValidationStatusCli.remoteVtepAddr ]

      vtepValidationStatusCmd = 'vxlan source-vtep-validation '
      vtepValidationStatusCmd += ' '.join( sorted( sipVtepsStrList ) )

      cmds.addCommand( vtepValidationStatusCmd )

   if entity.cliVccImportVlans == '':
      cmds.addCommand( 'vxlan controller-client import vlan none' )
   elif entity.cliVccImportVlans != '1-4094':
      cmds.addCommand( 'vxlan controller-client import vlan %s' %
                       entity.cliVccImportVlans )
   elif saveAll:
      cmds.addCommand( 'vxlan controller-client import vlan 1-4094' )

   if entity.controllerClientMode:
      cmds.addCommand( 'vxlan controller-client' )
   elif saveAll:
      cmds.addCommand( 'no vxlan controller-client' )

   if Toggles.VxlanToggleLib.toggleVxlanVccArpProxySupportedEnabled():
      if entity.vccArpProxy:
         cmds.addCommand( 'vxlan controller-client arp proxy' )
      elif saveAll:
         cmds.addCommand( 'no vxlan controller-client arp proxy' )
   if Toggles.VxlanControllerToggleLib.toggleVxlanVccNdProxySupportedEnabled():
      if entity.vccNdProxy:
         cmds.addCommand( 'vxlan controller-client nd proxy' )
      elif saveAll:
         cmds.addCommand( 'no vxlan controller-client nd proxy' )

   if Toggles.VxlanToggleLib.toggleVxlanArpProxyCliCommandEnabled():
      encapsulatedArpSnooping = \
            requireMounts[ 'vxlan/clientDir/cli' ].vxlanEncapsulatedArpSnooping
      if encapsulatedArpSnooping.isSet and encapsulatedArpSnooping.value:
         cmds.addCommand( 'vxlan arp proxy' )
      elif saveAll:
         cmds.addCommand( 'no vxlan arp proxy' )

   if Toggles.VxlanToggleLib.toggleVxlanNdProxyCliCommandEnabled():
      encapsulatedNdSnooping = \
            requireMounts[ 'vxlan/clientDir/cli' ].vxlanEncapsulatedNdSnooping
      if encapsulatedNdSnooping.isSet and encapsulatedNdSnooping.value:
         cmds.addCommand( 'vxlan nd proxy' )
      elif saveAll:
         cmds.addCommand( 'no vxlan nd proxy' )

   if entity.arpReplyRelay:
      cmds.addCommand( 'arp reply relay' )
   elif saveAll:
      cmds.addCommand( 'no arp reply relay' )

   if entity.mlagSharedRouterMacConfig == 'autoGenerated':
      cmds.addCommand( 'vxlan virtual-router encapsulation mac-address '
                       'mlag-system-id' )
   elif entity.mlagSharedRouterMacConfig == 'explicitConfig':
      cmds.addCommand( 'vxlan virtual-router encapsulation mac-address %s' %
                       entity.mlagSharedRouterMacAddr )
   elif saveAll:
      cmds.addCommand( 'no vxlan virtual-router encapsulation mac-address' )

   if entity.arpLocalAddress:
      cmds.addCommand( 'arp source ip address local' )
   elif saveAll:
      cmds.addCommand( 'no arp source ip address local' )

   # BUG48350
   # if entity.ttl != entity.defaultTtl:
   #   cmds.addCommand( 'vxlan ttl %d' % entity.ttl )
   # elif saveAll:
   #   cmds.addCommand( 'no vxlan ttl' )

   # if entity.udpPort != entity.vxlanWellKnownPort:
   #    cmds.addCommand( 'vxlan udp-port %d' % entity.udpPort )
   # elif saveAll:
   #    cmds.addCommand( 'no vxlan udp-port' )

   # As per discussion, we decided to generate udp-port running config always
   # till a consensus reached for default port in the industry.
   brHwCap = requireMounts[ 'bridging/hwcapabilities' ]
   if entity.intfId == 'Vxlan1' or brHwCap.vxlanMultiVtiMultiPortSupported:
      # Only Vxlan1 accepts a UDP port config unless
      # brHwCap.vxlanMultiVtiMultiPortSupported
      cmds.addCommand( 'vxlan udp-port %d' % entity.udpPort )

   if entity.srcPortRange != Tac.Value( "Vxlan::VxlanSrcPortRange" ):
      if brHwCap.vxlanSrcPortSupported == 'offsetAllValuesRangePowerOfTwo':
         cmds.addCommand( 'vxlan udp-port source offset %d length %d' %
                  ( entity.srcPortRange.offset, entity.srcPortRange.length ) )
      elif brHwCap.vxlanSrcPortSupported == 'anySingleValue':
         cmds.addCommand( 'vxlan udp-port source %d' % entity.srcPortRange.offset )
   elif saveAll:
      cmds.addCommand( 'no vxlan udp-port source offset' )

   # Generate secure udp-port running config only if any vtep is configured
   # as secure or if the secure udp-port is configured.
   if entity.vtepToSecProfile or \
         entity.secUdpPort != entity.vxlanSecWellKnownPort:
      cmds.addCommand( 'vxlan security udp-port %d' % entity.secUdpPort )

   if entity.secSrcPortRange.offset:
      cmds.addCommand( 'vxlan security udp-port source %d' %
                        entity.secSrcPortRange.offset )

   if entity.use32BitVni:
      cmds.addCommand( 'vxlan header vni 32-bit' )
   elif saveAll:
      cmds.addCommand( 'no vxlan header vni 32-bit' )

   # VTEP-to-VTEP bridging is only meaningful on Vxlan1, not on other VTIs
   if entity.intfId == 'Vxlan1':
      vtepToVtepBridging = \
            requireMounts[ 'vxlan/clientDir/cli' ].vxlanVtepToVtepBridgingEnabled
      if entity.vtepSourcePruningAll:
         cmds.addCommand( 'vxlan bridging vtep-to-vtep source-vtep tx disabled' )
      elif entity.vtepSetForSourcePruning:
         vteps = ' '.join( sorted( entity.vtepSetForSourcePruning,
                                   key=lambda x: Arnet.IpAddress( x ).value ) )
         cmd = 'vxlan bridging vtep-to-vtep source-vtep tx disabled %s' % vteps
         cmds.addCommand( cmd )
      elif vtepToVtepBridging.isSet and vtepToVtepBridging.value:
         cmds.addCommand( 'vxlan bridging vtep-to-vtep' )
         if saveAll:
            cmds.addCommand(
                  'no vxlan bridging vtep-to-vtep source-vtep tx disabled' )
      elif saveAll:
         cmds.addCommand( 'no vxlan bridging vtep-to-vtep' )
         cmds.addCommand( 'no vxlan bridging vtep-to-vtep source-vtep tx disabled' )

   if entity.floodLearnedAll:
      cmds.addCommand( 'vxlan flood vtep learned data-plane' )
   elif saveAll:
      cmds.addCommand( 'no vxlan flood vtep learned data-plane' )

   if entity.mcastRouting:
      cmds.addCommand( 'vxlan multicast routing ipv4' )
   elif saveAll:
      cmds.addCommand( 'no vxlan multicast routing ipv4' )

   vxlanEncapType = Tac.Type( "Vxlan::VxlanEncap" )
   brHwCap = requireMounts[ 'bridging/hwcapabilities' ]
   if brHwCap.vxlanEncapIpv6ConfigSupported:
      if entity.vxlanEncap == vxlanEncapType.vxlanEncapIp6:
         cmds.addCommand( 'vxlan encapsulation ipv6' )
      elif entity.vxlanEncap == vxlanEncapType.vxlanEncapDual:
         cmds.addCommand( 'vxlan encapsulation ipv4 ipv6' )
      elif saveAll:
         cmds.addCommand( 'vxlan encapsulation ipv4' )

   vxlanCntrlConfig = requireMounts[ 'vxlancontroller/config' ]

   sortedVlans = sorted( entity.vlanToVniMap )
   # Save VLAN-VNI map using range when new syntax is enabled
   if vtiConfigDir.vlanVniRangeSyntax and sortedVlans:
      vlanStr, vniStr = "", ""
      prevVlan = startVlan = None
      for vlan in sortedVlans:
         vlan = int( vlan )
         # Skip first iteration
         if prevVlan is None:
            prevVlan = startVlan = vlan
            continue
         vni = entity.vlanToVniMap[ vlan ]
         prevVni = entity.vlanToVniMap[ prevVlan ]
         # VLANs and VNIs must be adjacent to collapse into range
         if vlan != prevVlan + 1 or vni != prevVni + 1:
            startVni = VniFormat( entity.vlanToVniMap[ startVlan ],
                                  vxlanCntrlConfig.vniInDottedNotation )
            if startVlan == prevVlan:
               vlanStr += "%d," % startVlan
               vniStr += "%s," % startVni
            else:
               endVni = VniFormat( entity.vlanToVniMap[ prevVlan ],
                                   vxlanCntrlConfig.vniInDottedNotation )
               vlanStr += "%d-%d," % ( startVlan, prevVlan )
               vniStr += "%s-%s," % ( startVni, endVni )
            startVlan = vlan
         prevVlan = vlan
      # Add final mapping
      startVni = VniFormat( entity.vlanToVniMap[ startVlan ],
                            vxlanCntrlConfig.vniInDottedNotation )
      if prevVlan == startVlan:
         vlanStr += "%d" % startVlan
         vniStr += "%s" % startVni
      else:
         endVni = VniFormat( entity.vlanToVniMap[ prevVlan ],
                             vxlanCntrlConfig.vniInDottedNotation )
         vlanStr += "%d-%d" % ( startVlan, prevVlan )
         vniStr += "%s-%s" % ( startVni, endVni )
      cmds.addCommand( 'vxlan vlan %s vni %s' % ( vlanStr, vniStr ) )
   else:
      for vlan in sortedVlans:
         vniStr = VniFormat( entity.vlanToVniMap[ vlan ],
               vxlanCntrlConfig.vniInDottedNotation )
         cmds.addCommand( 'vxlan vlan %s vni %s' %
               ( vlan, vniStr ) )

   if entity.mcastGrpDecap:
      grps = ' '.join( sorted( entity.mcastGrpDecap,
                        key=lambda x: Arnet.IpAddress( x ).value ) )
      cmds.addCommand( 'vxlan multicast-group decap %s' % grps )
   elif saveAll:
      cmds.addCommand( 'no vxlan multicast-group decap' )

   if Toggles.VxlanToggleLib.toggleVxlanTcpMssClampingSupportedEnabled():
      if entity.tcpMssV4 or entity.tcpMssV6:
         cmdStr = 'tcp mss ceiling'
         if entity.tcpMssV4:
            cmdStr += ' ipv4 ' + str( entity.tcpMssV4 )
         if entity.tcpMssV6:
            cmdStr += ' ipv6 ' + str( entity.tcpMssV6 )
         cmdStr += ' egress'
         cmds.addCommand( cmdStr )
      elif saveAll:
         cmds.addCommand( 'no tcp mss ceiling' )

   vrfSet = set( entity.vrfToVniMap )
   # vrfToDecapVniMap is reverse map of entity.alternateDecapVniToVrfMap
   vrfToDecapVniMap = collections.defaultdict( set )
   for vni, vrf in six.iteritems( entity.alternateDecapVniToVrfMap ):
      vrfToDecapVniMap[ vrf ].add( vni )

   # Generate NAT profile configuration for the VNI
   if brHwCap.natWithVxlanSupported:
      for vni in sorted( entity.vniNatProfile ):
         profileName = entity.vniNatProfile[ vni ]
         cmds.addCommand(
               f"vxlan vni {vni} routed ip nat service-profile {profileName}" )

   if Toggles.VxlanToggleLib.toggleVxlanVirtualArpOptimizedFloodingEnabled():
      if not entity.virtualArpOptimizedFloodingEnabled:
         cmds.addCommand( 'virtual-router arp optimize flooding disabled' )
      elif saveAll:
         cmds.addCommand( 'no virtual-router arp optimize flooding disabled' )

   icmpHelperInputConfigDir = requireMounts[ 'tepicmphelper/input/config' ]
   if Toggles.VxlanToggleLib.toggleVtepIcmpHelperEnabled():
      if icmpHelperInputConfigClientName in icmpHelperInputConfigDir.entityPtr:
         cmds.addCommand( 'icmp helper' )
         for v in icmpHelperInputConfigDir[ icmpHelperInputConfigClientName ].vrf:
            cmds.addCommand( 'icmp helper vrf %s' % v )
      elif saveAll:
         cmds.addCommand( 'no icmp helper' )

   def displayDecapVniForVrf( vrf ):
      # Decap VNIs
      if vrf not in vrfToDecapVniMap:
         return
      dotN = vxlanCntrlConfig.vniInDottedNotation
      vrfToDecapVniMapTagged = collections.defaultdict( set )
      vrfToDecapVni = set()
      for vni in vrfToDecapVniMap[ vrf ]:
         vrfVni = Tac.const( Tac.newInstance( 'Vxlan::VrfVniConfigTagKey',
            vrf, vni ) )
         configTag = entity.vrfVniConfigTagConfig.get( vrfVni, None )
         if configTag:
            vrfToDecapVniMapTagged[ configTag ].add( vni )
         else:
            vrfToDecapVni.add( vni )

      def addCommands( vniList, configTag=None ):
         decapVniStr = ",".join( sorted( VniFormat( vni, dotN ).__str__()
            for vni in vniList ) )
         if decapVniStr:
            if configTag:
               # we can have multiple config tag associations for a given
               # vrf and decap vniList. Hence we 'add' the vniList for configTags.
               cmds.addCommand(
                     'vxlan vrf %s vni decapsulation add %s command-tag %s'
                     % ( vrf, decapVniStr, configTag ) )
            else:
               cmds.addCommand( 'vxlan vrf %s vni decapsulation %s'
                     % ( vrf, decapVniStr ) )

      # Tagged vniList for vrf ( with 'add' keyword ) must appear after the
      # non tagged vniList to ensure correctness in order.
      addCommands( vrfToDecapVni )
      for configTag, vniList in six.iteritems( vrfToDecapVniMapTagged ):
         addCommands( vniList, configTag )

   decapVrfs = set( entity.alternateDecapVniToVrfMap.values() )
   ipVrfs = set( entity.vrfToIpAddr )
   ip6Vrfs = set( entity.vrfToIp6Addr )
   for vrf in sorted( vrfSet | decapVrfs | ipVrfs | ip6Vrfs ):
      if vrf in vrfSet:
         vni = entity.vrfToVniMap[ vrf ]
         vrfVni = Tac.const( Tac.newInstance( 'Vxlan::VrfVniConfigTagKey',
            vrf, vni ) )
         vniStr = VniFormat( vni, vxlanCntrlConfig.vniInDottedNotation )
         configTag = entity.vrfVniConfigTagConfig.get( vrfVni, None )
         if configTag:
            cmds.addCommand( 'vxlan vrf %s vni %s command-tag %s'
                  % ( vrf, vniStr, configTag ) )
         else:
            cmds.addCommand( 'vxlan vrf %s vni %s' % ( vrf, vniStr ) )
      displayDecapVniForVrf( vrf )
      if vrf in ipVrfs:
         ipStr = str( entity.vrfToIpAddr[ vrf ] )
         cmds.addCommand( 'vxlan vrf %s ip address %s' % ( vrf, ipStr ) )
      if vrf in ip6Vrfs:
         ip6Str = str( entity.vrfToIp6Addr[ vrf ] )
         cmds.addCommand( 'vxlan vrf %s ipv6 address %s' % ( vrf, ip6Str ) )

   if entity.vtepAddrMask != '255.255.255.255':
      cmds.addCommand( 'vxlan vtep ipv4 address-mask %s' %
                       entity.vtepAddrMask )
   elif saveAll:
      cmds.addCommand( 'no vxlan vtep ipv4 address-mask' )

   if entity.mlagSrcIpIntf != '':
      cmds.addCommand( 'vxlan mlag source-interface %s' %
                       entity.mlagSrcIpIntf )
   elif saveAll:
      cmds.addCommand( 'no vxlan mlag source-interface' )

   if entity.varpVtepSrcIpIntf != '':
      cmds.addCommand( 'vxlan virtual-vtep local-interface %s' %
                       entity.varpVtepSrcIpIntf )
   elif saveAll:
      cmds.addCommand( 'no vxlan virtual-vtep local-interface' )

   if entity.decapFilterMode != 'filterEnabled':
      cmdStr = 'vxlan decapsulation filter'
      if entity.decapFilterMode == 'filterDisabled':
         cmdStr += ' disabled'
         cmds.addCommand( cmdStr )
      else:
         if entity.decapFilterMode == 'filterEnabledIntf':
            cmdStr += ' vrf default'
         else:
            cmdStr += ' interface multiple-vrf disabled'
         # 1. decapFilterIntf will be empty when the mode is filterRelaxedAll
         # 2. decapFilterIntf is shared between modes filterRelaxedIntf and
         #    filterEnabledIntf
         for intf in Arnet.sortIntf( entity.decapFilterIntf ):
            cmdStr += ( ' %s' % intf )
         cmds.addCommand( cmdStr )
   elif saveAll:
      cmds.addCommand( 'no vxlan decapsulation filter' )

   if entity.decapFilterNonDefaultVrf:
      cmds.addCommand( 'vxlan decapsulation filter vrf non-default ipv4' )
   elif saveAll:
      cmds.addCommand( 'no vxlan decapsulation filter vrf non-default ipv4' )

   if Toggles.VxlanToggleLib.toggleVxlanDecapRouteAllEnabled():
      if entity.decapRouteAll:
         cmds.addCommand( 'vxlan decapsulation route all' )
      elif saveAll:
         cmds.addCommand( 'no vxlan decapsulation route all' )

   profileToVtepMap = dict()
   for ( vtep, profile ) in six.iteritems( entity.vtepToSecProfile ):
      profileToVtepMap.setdefault( profile, list() ).append( vtep )
   for profile, vteps in sorted( six.iteritems( profileToVtepMap ) ):
      secVteps = ' '.join( sorted( ip.v4Addr for ip in vteps ) )
      cmds.addCommand( 'vxlan vtep %s ip security profile %s' %
                        ( secVteps, profile ) )

   if entity.bfdParams.bfdEnabled:
      cmds.addCommand( 'bfd vtep evpn interval %d min-rx %d multiplier %d' %
                       ( entity.bfdParams.bfdIntervalParams.minTx,
                         entity.bfdParams.bfdIntervalParams.minRx,
                         entity.bfdParams.bfdIntervalParams.mult ) )
   elif saveAll:
      cmds.addCommand( 'no bfd vtep evpn interval' )

   if entity.bfdPrefixList:
      cmds.addCommand( 'bfd vtep evpn prefix-list %s' % entity.bfdPrefixList )
   elif saveAll:
      cmds.addCommand( 'no bfd vtep evpn prefix-list' )

@CliSave.saver( 'Vxlan::VxlanConfigDir', 'vxlan/config' )
def saveVxlanFdbConfig( entity, root, requireMounts, options ):
   cmds = root[ 'Vxlan.staticMac' ]
   configuredHosts = entity.fdbConfig.configuredHost
   sortedConfiguredHosts = sorted( configuredHosts.values(),
                                   key=lambda h: ( h.macVlanPair.vlanId,
                                                   h.macVlanPair.macAddr ) )
   for host in sortedConfiguredHosts:
      macAddr = convertMacAddrCanonicalToDisplay( host.macVlanPair.macAddr )
      cmds.addCommand(
         'mac address-table static %s vlan %s interface %s vtep %s' %
         ( macAddr, host.macVlanPair.vlanId, host.vtiIntfId, host.remoteVtepAddr ) )

@CliSave.saver( 'Vxlan::VxlanConfig', 'vxlan/config',
                attrName='vxlanConfig',
                requireMounts=( 'interface/config/eth/vxlan',
                                'vxlan/config' ) )
def saveVxlanConfig( entity, root, requireMounts, options ):
   # Cleanup and bail out on inconsistent state where only one of VxlanConfig and
   # VtiConfig exists
   vtiConfigDir = requireMounts[ 'interface/config/eth/vxlan' ]
   vxlanConfigDir = requireMounts[ 'vxlan/config' ]
   if entity.intfId not in vtiConfigDir.vtiConfig:
      del vxlanConfigDir.vxlanConfig[ entity.intfId ]
      return

   saveAll = options.saveAll
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'VxlanIntf.config' ]

   if entity.floodVtepList or entity.floodVtepList6:
      vteps = canonicalVtepString( entity.floodVtepList, entity.floodVtepList6 )
      cmd = f'vxlan flood vtep {vteps}'
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no vxlan flood vtep'
      cmds.addCommand( cmd )

   if entity.vlanToVtepList and vxlanConfigDir.vlanFloodVtepRangeSyntax:
      # Accumulate all vlans for each vtepList to save commands with range syntax.
      vtepsToVlanList = collections.defaultdict( list )
      for vlan, vl in sorted( entity.vlanToVtepList.items() ):
         vteps = canonicalVtepString( vl.remoteVtepAddr, vl.remoteVtepAddr6 )
         vtepsToVlanList[ vteps ].append( vlan )
      # Save the commands.
      for vteps, vlanList in vtepsToVlanList.items():
         vlans = MultiRangeRule.multiRangeToCanonicalString( vlanList )
         cmd = f'vxlan vlan {vlans} flood vtep {vteps}'
         cmds.addCommand( cmd )
   elif entity.vlanToVtepList:
      for vlan, vl in sorted( entity.vlanToVtepList.items() ):
         vteps = canonicalVtepString( vl.remoteVtepAddr, vl.remoteVtepAddr6 )
         cmd = f'vxlan vlan {vlan} flood vtep {vteps}'
         cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no vxlan vlan flood vtep'
      cmds.addCommand( cmd )

   if entity.learnFrom == 'learnFromFloodList':
      cmd = 'vxlan learn-restrict flood'
      cmds.addCommand( cmd )
   elif entity.learnFrom == 'learnFromAny':
      cmd = 'vxlan learn-restrict any'
      cmds.addCommand( cmd )
   elif entity.learnFrom == 'learnFromList':
      if entity.learnPrefixList:
         prefs = sorted( entity.learnPrefixList,
                         key=lambda x: x.ipPrefix.sortKey )
         prefixes = ' '.join( [ x.ipPrefix.stringValue for x in prefs ] )
         cmd = 'vxlan learn-restrict vtep %s' % prefixes
      else:
         # Learn from nobody
         cmd = 'vxlan learn-restrict vtep'
      cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no vxlan learn-restrict'
      cmds.addCommand( cmd )

   if entity.vlanToLearnRestrict:
      for ( vlan, vl ) in sorted( six.iteritems( entity.vlanToLearnRestrict ),
                                  key=lambda k__1: k__1[ 0 ] ):
         if vl.learnFrom == 'learnFromFloodList':
            cmd = 'vxlan vlan %d learn-restrict flood' % vlan
            cmds.addCommand( cmd )
         elif vl.learnFrom == 'learnFromAny':
            cmd = 'vxlan vlan %d learn-restrict any' % vlan
            cmds.addCommand( cmd )
         elif vl.learnFrom == 'learnFromList':
            if vl.prefixList:
               prefs = sorted( vl.prefixList,
                               key=lambda x: x.ipPrefix.sortKey )
               prefixes = ' '.join( [ x.ipPrefix.stringValue for x in prefs ] )
               cmd = 'vxlan vlan %d learn-restrict vtep %s' % ( vlan, prefixes )
            else:
               # Learn from nobody
               cmd = 'vxlan vlan %d learn-restrict vtep' % vlan
            cmds.addCommand( cmd )
         elif saveAll:
            cmd = 'no vxlan vlan %d learn-restrict' % vlan
            cmds.addCommand( cmd )
   elif saveAll:
      cmd = 'no vxlan vlan learn-restrict vtep'
      cmds.addCommand( cmd )

   for vni in sorted( entity.vniToIpAclMap ):
      aclName = entity.vniToIpAclMap[ vni ]
      cmds.addCommand( 'vxlan vni %d routed ip access-list %s in' % (
         vni, aclName ) )

   for vni in sorted( entity.vniToIpv6AclMap ):
      aclName = entity.vniToIpv6AclMap[ vni ]
      cmds.addCommand( 'vxlan vni %d routed ipv6 access-list %s in' % (
         vni, aclName ) )

   for vni in sorted( entity.vniToEgressIpAclMap ):
      aclName = entity.vniToEgressIpAclMap[ vni ]
      cmds.addCommand( 'vxlan vni %d routed ip access-list %s out' % (
         vni, aclName ) )

   for vni in sorted( entity.vniToEgressIpv6AclMap ):
      aclName = entity.vniToEgressIpv6AclMap[ vni ]
      cmds.addCommand( 'vxlan vni %d routed ipv6 access-list %s out' % (
         vni, aclName ) )

   for vni in sorted( entity.vniQosInfo ):
      if entity.vniQosInfo[ vni ].iPolicer:
         if entity.vniQosInfo[ vni ].iPolicer.policerMode:
            cmds.addCommand( 'vxlan vni %d policer group %s input' % (
            vni, entity.vniQosInfo[ vni ].iPolicer.policerName ) )
         else:
            cmds.addCommand( 'vxlan vni %d policer profile %s input' % (
            vni, entity.vniQosInfo[ vni ].iPolicer.policerName ) )

      if entity.vniQosInfo[ vni ].oPolicer:
         if entity.vniQosInfo[ vni ].oPolicer.policerMode:
            cmds.addCommand( 'vxlan vni %d policer group %s output' % (
            vni, entity.vniQosInfo[ vni ].oPolicer.policerName ) )
         else:
            cmds.addCommand( 'vxlan vni %d policer profile %s output' % (
            vni, entity.vniQosInfo[ vni ].oPolicer.policerName ) )

   for vniAndRemoteVtep in sorted( entity.vniVtepSrcIp ):
      cmds.addCommand(
            'vxlan vni %d vtep %s encapsulation ipv4 source address %s' % (
               vniAndRemoteVtep.vni, vniAndRemoteVtep.remoteVtep,
               entity.vniVtepSrcIp[ vniAndRemoteVtep ] ) )

@CliSave.saver( 'Vxlan::EcnConfig', 'vxlan/ecnConfig',
                requireMounts=( 'interface/config/eth/vxlan',
                                'vxlan/config' ) )
def saveVxlanEcnConfig( entity, root, requireMounts, options ):
   vtiConfigDir = requireMounts[ 'interface/config/eth/vxlan' ]
   vxlanConfigDir = requireMounts[ 'vxlan/config' ]
   saveAll = options.saveAll

   for intfId in vxlanConfigDir.vxlanConfig:
      if intfId not in vtiConfigDir.vtiConfig:
         continue
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
      cmds = mode[ 'Vxlan.ecnConfig' ]

      if entity.ecnPropagation:
         cmds.addCommand( 'vxlan qos ecn propagation' )
      elif options.saveAll:
         cmds.addCommand( 'no vxlan qos ecn propagation' )

      if entity.dscpEcnPropagation:
         if options.saveAll:
            cmds.addCommand(
                  'no vxlan qos dscp ecn propagation decapsulation disabled' )
      else:
         cmds.addCommand( 'vxlan qos dscp ecn propagation decapsulation disabled' )

      if entity.dscpEcnRewriteBridged:
         cmds.addCommand( 'vxlan qos dscp ecn rewrite bridged enabled' )
      elif options.saveAll:
         cmds.addCommand( 'no vxlan qos dscp ecn rewrite bridged enabled' )

@CliSave.saver( 'Vxlan::VxlanConfigDir', 'vxlan/config',
                requireMounts=( 'interface/config/eth/vxlan', ) )
def saveVxlanOuterDscpFromInnerDscpConfig( entity, root, requireMounts, options ):
   vtiConfigDir = requireMounts[ 'interface/config/eth/vxlan' ]
   saveAll = options.saveAll

   for intfId in entity.vxlanConfig:
      if intfId not in vtiConfigDir.vtiConfig:
         continue

      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
      cmds = mode[ 'Vxlan.outerDscpFromInnerDscpConfig' ]

      if Toggles.VxlanToggleLib.toggleVxlanOuterDscpFromInnerDscpEnabled():
         if entity.outerDscpFromInnerDscp:
            cmds.addCommand( 'vxlan qos dscp propagation encapsulation' )
         elif options.saveAll:
            cmds.addCommand( 'no vxlan qos dscp propagation encapsulation' )

         if entity.trafficClassFromOuterDscp:
            cmds.addCommand( 'vxlan qos map dscp to traffic-class decapsulation' )
         elif options.saveAll:
            cmds.addCommand( 'no vxlan qos map dscp to traffic-class decapsulation' )

@CliSave.saver( 'Vxlan::VxlanFloodsetConfig', 'vxlan/floodsetConfig',
                requireMounts=( 'interface/config/eth/vxlan',
                                'vxlan/config' ) )
def saveVxlanFloodsetConfig( entity, root, requireMounts, options ):
   vtiConfigDir = requireMounts[ 'interface/config/eth/vxlan' ]
   vxlanConfigDir = requireMounts[ 'vxlan/config' ]
   saveAll = options.saveAll

   for intfId in vxlanConfigDir.vxlanConfig:
      if intfId not in vtiConfigDir.vtiConfig:
         continue

      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
      cmds = mode[ 'Vxlan.floodsetConfig' ]
      if entity.decapUcastMcastFloodsetExcludeCpu:
         cmd = 'vxlan decap floodset unknown-multicast unknown-unicast exclude cpu'
         cmds.addCommand( cmd )
      elif options.saveAll:
         cmd = 'no vxlan decap floodset ' + \
               'unknown-multicast unknown-unicast exclude cpu'
         cmds.addCommand( cmd )

def formCmd( vrfName, ipAddr, macAddr ):
   cmd = 'vxlan forwarding match vrf ' + vrfName + ' src-ip ' + ipAddr +\
         ' action rewrite src-mac ' + macAddr + ' out'
   return cmd

@CliSave.saver( 'Vxlan::Input::MacRewriteConfig',
                'vxlan/hardware/srcMacRewrite/config' )
def saveVxlanMacRewriteCliConfig( entity, root, requireMounts, options ):

   cmds = root[ 'VxlanGlobal.config' ]
   unsortedCmds = []
   for vrfNameAndIp in entity.overlaySrcMacRewriteEntry:
      vrfIpAndMac = entity.overlaySrcMacRewriteEntry[ vrfNameAndIp ]
      cmd = formCmd( vrfNameAndIp.vrfName, vrfNameAndIp.ipAddr, vrfIpAndMac.macAddr )
      unsortedCmds.append( cmd )
   for cmd in sorted( unsortedCmds ):
      cmds.addCommand( cmd )

def formIntfSrcMacRewriteCmd( vti, vni, macAddr ):
   cmd = 'vxlan forwarding match local-interface ' + vti + ' vni ' + str( vni ) +\
         ' action rewrite src-mac ' + macAddr + ' out'
   return cmd

@CliSave.saver( 'Vxlan::Input::MacRewriteConfig',
                'vxlan/hardware/srcMacRewrite/config' )
def saveIntfSrcMacRewriteCliConfig( entity, root, requireMounts, options ):

   cmds = root[ 'VxlanGlobal.config' ]
   unsortedCmds = []
   for vniVtiPair in entity.intfOverlaySrcMacRewriteEntry:
      vniVtiAndMac = entity.intfOverlaySrcMacRewriteEntry[ vniVtiPair ]
      cmd = formIntfSrcMacRewriteCmd(
            vniVtiPair.vti, vniVtiPair.vni, vniVtiAndMac.macAddr )
      unsortedCmds.append( cmd )
   for cmd in sorted( unsortedCmds ):
      cmds.addCommand( cmd )

# -------------------------------------------------------------------------------
# Saves the state of an PolicyMap::IntfConfig objecti vni field.
# -------------------------------------------------------------------------------
@CliSave.saver( 'PolicyMap::IntfConfig', 'pbr/input/intf/config/vxlan',
                requireMounts=( 'pbr/input/pmap/config/vxlan',
                                'interface/config/eth/vxlan',
                                'routing/hardware/statuscommon',
                                'routing/hardware/status' ) )
def savePbrOnVniConfig( entity, root, requireMounts, options ):

   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   if not routingHwStatus.pbrOnVniSupported:
      return

   vtiConfigDir = requireMounts[ 'interface/config/eth/vxlan' ]

   for vti in vtiConfigDir.vtiConfig:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( vti )
      cmds = mode[ 'VxlanIntf.config' ]
      for vni in sorted( entity.vni ):
         pmapName = entity.vni[ vni ]
         cmds.addCommand( 'vxlan vni %s service-policy type pbr input %s' %
                          ( vni, pmapName ) )
