# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, CliSave
import Arnet
from CliSavePlugin.Controllerdb import ( CvxConfigMode,
                                         controllerConfigPath,
                                         getClusterName )
from VxlanControllerAgent import vxlanControllerConfigSysdbPath
from CliMode.Vxlan import VxlanMode
import Toggles.VxlanControllerToggleLib
 
class VxlanConfigSaveMode( VxlanMode, CliSave.Mode ):
   def __init__( self, param ):
      VxlanMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CvxConfigMode.addChildMode( VxlanConfigSaveMode,
                            after=[ 'cvx' ] )
VxlanConfigSaveMode.addCommandSequence( 'VxlanControllerCVX.config' )
CliSave.GlobalConfigMode.addCommandSequence( 'VxlanControllerGlobal.config',
                                             after=[ CvxConfigMode ] )

def saveVxlcConfigVniNotation( root, entity, options ):
   cmds = root[ 'VxlanControllerGlobal.config' ]
   if entity.vniInDottedNotation:
      cmds.addCommand( 'vxlan vni notation dotted' )
   elif options.saveAll:
      cmds.addCommand( 'no vxlan vni notation dotted' )

def saveVxlcConfigEnable( cmds, entity, options ):
   if entity.enable:
      cmds.addCommand( "no shutdown" )
   elif options.saveAll:
      cmds.addCommand( "shutdown" )

def saveVxlcConfigDataPathLearning( cmds, entity, options ):
   if entity.dataPathLearning:
      cmds.addCommand( 'vtep mac-learning data-plane' )
   elif options.saveAll:
      cmds.addCommand( 'vtep mac-learning control-plane' )

def saveVxlcConfigPortDot1qVniMapping( cmds, entity, options ):
   if entity.portDot1qVniMapping:
      cmds.addCommand( 'vtep vni mapping port-dot1q-tag' )
   elif options.saveAll:
      cmds.addCommand( 'vtep vni mapping vlan' )

def saveVxlcConfigRestartSyncPeriod( cmds, entity, options ):
   if entity.restartSyncPeriod == 0:
      cmds.addCommand( "no resync-period" )
   elif ( entity.restartSyncPeriod != entity.defaultRestartSyncPeriod or
          options.saveAll ):
      cmds.addCommand( f'resync-period {entity.restartSyncPeriod}' )

def saveVxlcConfigBgpEvpnRedist( cmds, entity, options ):
   if entity.bgpEvpnRedist:
      cmds.addCommand( "redistribute bgp evpn vxlan" )
   elif options.saveAll:
      cmds.addCommand( "no redistribute bgp evpn vxlan" )

def saveVxlcConfigMulticastMembershipList( cmds, entity, options ):
   if entity.multicastMembershipList:
      cmds.addCommand( 'multicast membership-list' )
   elif options.saveAll:
      cmds.addCommand( 'no multicast membership-list' )

def saveVxlcConfigDefaultFloodList( cmds, entity, options ):
   if entity.floodVtepList:
      vteps = ' '.join( sorted( entity.floodVtepList, 
                                key=lambda x: Arnet.IpAddress( x ).value ) )
      cmds.addCommand( f'flood vtep {vteps}' )
   elif options.saveAll:
      cmds.addCommand( 'no flood vtep' )

def saveVxlcConfigVniFloodList( cmds, entity, options ):
   macAddr = Tac.Type( "Arnet::EthAddr" ).ethAddrZero
   if entity.vniToVtepList:
      for ( vni, vtepList ) in sorted( entity.vniToVtepList.items(),
                                       key=lambda k__: k__[ 0 ] ):
         bumMacVtepList = vtepList.remoteVtep[ macAddr ]
         vteps = ' '.join( sorted( bumMacVtepList.vtepIpFloodList,
                                   key=lambda x: Arnet.IpAddress( x ).value ) )
         cmds.addCommand( f'vni {vni} flood vtep {vteps}' )
   elif options.saveAll:
      cmds.addCommand( 'no vni flood vtep' )

def saveArpReplyRelayVtepList( cmds, entity, options ):
   cmd = "arp reply relay vtep "
   if entity.arpReplyRelayVtep:
      cmd += " ".join( [ v.stringValue
                         for v in sorted( entity.arpReplyRelayVtep,
                                          key=lambda x: x.sortKey ) ] )
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( "no " + cmd )

# pylint: disable-msg=C0322 # pylint: disable=bad-option-value
@CliSave.saver( 'VxlanController::Config', vxlanControllerConfigSysdbPath,
                requireMounts=( controllerConfigPath, ) )
def saveVxlanControllerConfig( entity, root, requireMounts, options ):
   cvxClusterName = getClusterName( requireMounts[ controllerConfigPath ] )
   if not cvxClusterName:
      return
   cvxMode = root[ CvxConfigMode ].getOrCreateModeInstance(
      CvxConfigMode.modeName( cvxClusterName ) )
   mode = cvxMode[ VxlanConfigSaveMode ].getSingletonInstance()
   cmds = mode[ 'VxlanControllerCVX.config' ]

   saveVxlcConfigVniNotation( root, entity, options )
   saveVxlcConfigEnable( cmds, entity, options )
   saveVxlcConfigDataPathLearning( cmds, entity, options )
   saveVxlcConfigPortDot1qVniMapping( cmds, entity, options )
   saveVxlcConfigRestartSyncPeriod( cmds, entity, options )
   saveVxlcConfigBgpEvpnRedist( cmds, entity, options )
   if Toggles.VxlanControllerToggleLib.toggleVCSMulticastKnobEnabled():
      saveVxlcConfigMulticastMembershipList( cmds, entity, options )
   saveVxlcConfigDefaultFloodList( cmds, entity, options )
   saveVxlcConfigVniFloodList( cmds, entity, options )
   saveArpReplyRelayVtepList( cmds, entity, options )

