# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=consider-using-in

import os
import threading
import traceback

import Ark
import CliSessionDataStore
import EntityManager
import Plugins
from Syscall import gettid
import TableOutput
import Tac
import Tracing
import QuickTrace

qt0 = QuickTrace.trace0

t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2
t5 = Tracing.trace5
t6 = Tracing.trace6
t8 = Tracing.trace8
t9 = Tracing.trace9

WARN_WAIT_TIME = 30

ancestorDir = None
cliConfig = None
pendingSession = None
sessionConfig = None

sessionStatus = None

sessionLock = threading.RLock()

__sessionNameCounter__ = 1

sessionEm = None

redundancyReactor = None

# threadLocalData object needs to be created just once and not per thread. This
# object grabs the locks when attributes are accessed. The attributes are local
# to the thread.
threadLocalData = threading.local()

# This implements a registration mechanism that allows the plugins to register
# a callback function when the current session is committed.
#
# The typical usage is to use BasicCli.maybeCallConfigSessionOnCommitHandler()
# which will either call the handler directly or register it as an onCommit
# handler based on whether it's inside a session right now.
#
# The handler takes one 'onCommit' parameter - it's True if the handler
# is invoked after a CLI session commit, and False if it's called without
# a session being involved (it's really a convenience for using the
# BasicCli.maybeCallConfigSessionOnCommitHandler() API).

# Maps each session to a dictionary of post-commit handlers.
class OnCommitHandler:
   def __init__( self ):
      self.orderList = []
      self.handlerMap = {}

sessionOnCommitHandlers = {}

def registerSessionOnCommitHandler( em, key, handler ):
   """ Adds the handler to the list for the current session.

   A handler must be a callable which takes mode as a parameter. It also
   takes an optional second parameter onSessionCommit=True which tells
   from which context the handler is invoked. This is useful in case the
   handler is invoked from both config session and other cases by using
   BasicCli.maybeCallConfigSessionOnCommitHandler().

   See CliSessionOnCommit.invokeSessionOnCommitHandlers() for how the handler
   is invoked.

   Only the last handler of the same key will be registered. This is useful
   since we only need to have one callback no matter how many times a
   particular configuration is changed.
   """

   sessionName = currentSession( em )
   assert sessionName, "Not in Cli session"
   handlers = sessionOnCommitHandlers.setdefault( sessionName,
                                                  OnCommitHandler() )
   newKey = key not in handlers.handlerMap
   handlers.handlerMap[ key ] = handler
   if newKey:
      # so we can invoke handlers based on registration time
      handlers.orderList.append( key )

_preCommitHandlers = []
registerPreCommitHandler = _preCommitHandlers.append

def runPreCommitHandlers( em, sessionName ):
   t5( 'Run preCommitHandlers' )
   for handler in _preCommitHandlers:
      handler( em, sessionName )

_postCommitHandlers = []
registerPostCommitHandler = _postCommitHandlers.append

def runPostCommitHandlers( mode, sessionName ):
   t5( 'Run postCommitHandlers' )
   for handler in _postCommitHandlers:
      t6( 'Running postCommitHandler', handler )
      handler( mode, sessionName )
      t6( 'Finished running postCommitHandler', handler )

# Error raised by preCommitHandler.  The userMsg is returned to user.
# The debugMsg is written to quicktrace.
class PreCommitHandlerError( Exception ):
   def __init__( self, userMsg, debugMsg ):
      self.userMsg_ = userMsg
      self.debugMsg_ = debugMsg
      Exception.__init__( self, userMsg )

   # Record debugMsg to quicktrace and return response
   def recordAndResponse( self ):
      qt0( "preCommitHandlerError: " + self.debugMsg_ )
      return self.userMsg_

def discardSessionOnCommitHandlers( sessionName ):
   """ Releases the registered session commit handlers. """
   sessionOnCommitHandlers.pop( sessionName, None )
   CliSessionDataStore.cleanup( sessionName )

def configRootInitialized( em ):
   h = handlerDir( em, create=False )
   return h and h.configRoot and h.configRoot.rootsComplete

def waitForConfigRootInitialized( em ):

   sleep = not Tac.activityManager.inExecTime.isZero
   Tac.waitFor( lambda: configRootInitialized( em ),
                sleep=sleep, warnAfter=0,
                description="configRoot to be initialized" )

def sessionStatusInitialized( em ):
   # this test is needed for tests using Simple EntityManager
   if not sessionStatus:
      return False
   h = handlerDir( em, create=False )
   return ( h and h.status and
            h.status.initialized and
            configRootInitialized( em ) )

def waitForSessionStatusInitialized( em ):
   sleep = not Tac.activityManager.inExecTime.isZero
   Tac.waitFor( lambda: sessionStatusInitialized( em or sessionEm ),
                sleep=sleep, warnAfter=0,
                description="session status to be initialized" )

def inputDir( em, create=True ):
   # /ar/CliSessionMgr/inputDir
   agentDir = getAgentDir( em, create )
   if not agentDir:
      return None
   h = agentDir.get( 'inputDir' )
   if h or not create:
      return h
   return agentDir.newEntity( "Tac::Dir", "inputDir" )

def registerCopyHandlerDir( em, name, typeName ):
   # register an entry uner /ar/CliSessionMgr/inputDir/<name>
   return inputDir( em ).newEntity( typeName, name )

def handlerDir( em, create=True ):
   # /ar/CliSessionMgr/handlerDir
   agentDir = getAgentDir( em, create )
   if not agentDir:
      return None
   h = agentDir.get( 'handlerDir' )
   if h or not create:
      return h
   return agentDir.newEntity( "Cli::Session::HandlerDir", "handlerDir" )

def configRoot( em ):
   return handlerDir( em ).configRoot

def attributeEditBypassDir( em ):
   return handlerDir( em ).attributeEditBypassTable.defaultEntityDir

def cachedSessionName():
   return getattr( threadLocalData, 'cachedSessionName', None )

def cachedSessionNameIs( name ):
   threadLocalData.cachedSessionName = name

def currentSession( em ):
   """Returns name of current session, or None"""

   # cached currentSession is based on the assumption that *nothing* will
   # change our session without going through either CliSession.enterSession or
   # CliSession.exitSession.  This assumption will break if we add any cases
   # that enter or exit sessions through, say, tacc.
   return cachedSessionName() or None

def currentSessionState( em ):
   '''
   Return the state of the current session or `None`.
   The state is one of `Cli::Session::SessionState`.
   '''
   sessionName = currentSession( em )
   if sessionName is None:
      return None
   return sessionStatus.sessionStateDir.state.get( sessionName )

def prefixIs( entityManager, path, flags, typename ):
   t9( "prefixIs: path", path, "flags", flags, "type", typename )
   waitForConfigRootInitialized( entityManager )
   cleanPath = EntityManager.cleanPath( path )
   rootTrie = configRoot( entityManager ).rootTrie
   assert rootTrie.isPrefix( cleanPath ), \
          "cannot ConfigMount path %s: " \
          "must also be marked CONFIG: in the preinit profile" \
          % cleanPath
   return rootTrie.prefixFlags( cleanPath )

class RedundancyReactor( Tac.Notifiee ):
   ''' The CliSessionMgr state machines should be created only on the active.
   '''

   notifierTypeName = 'Redundancy::RedundancyStatus'

   def __init__( self, redundancyStatus, entityManager, callback ):
      super().__init__( redundancyStatus )
      self.redundancyStatus_ = redundancyStatus
      self.entityManager_ = entityManager
      self.callback_ = callback
      self.callbackInvoked_ = False

      if ( redundancyStatus.protocol == "sso" and
           redundancyStatus.mode != "active" ):
         t0( "Not creating creating CliSessionMgr on standby" )
         self.handleStage( '' )
      else:
         self.callback_()
         self.callbackInvoked_ = True

   @Tac.handler( 'switchoverStage' )
   def handleStage( self, stage ):
      switchoverReady = 'SwitchoverReady'
      if ( not self.callbackInvoked_ and
           switchoverReady in self.redundancyStatus_.switchoverStage and
           self.redundancyStatus_.switchoverStage[ switchoverReady ] ):
         t8( "Creating CliSessionMgr state machines" )
         self.callback_()
         self.callbackInvoked_ = True


# register for ConfigSession deletion notification
sessionCleanupHandlers_ = []

def registerSessionCleanupHandler( h ):
   sessionCleanupHandlers_.append( h )

class PendingChangeConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Cli::Session::PendingChangeConfigDir'

   @Tac.handler( 'config' )
   def handleConfig( self, name ):
      if name not in self.notifier_.config:
         t0( "notifying session cleanup handlers for", name )
         # config session deleted, notify
         for h in sessionCleanupHandlers_:
            try:
               h( name )
            except:
               t0( "exception occured in session cleanup handler:" )
               t0( traceback.format_exc() )
               raise

def getAgentDir( em, create=True ):
   parent = em.root().parent
   mgr = parent.get( 'CliSessionMgr' )
   if mgr or not create:
      return mgr
   return parent.newEntity( "Tac::Dir", "CliSessionMgr" )

def getAgent( em, create=True ):
   agentDir = getAgentDir( em, create )
   if not agentDir:
      return None
   agent = agentDir.get( "root" )
   if agent or not create:
      return agent
   return agentDir.newEntity( "Cli::Session::Agent::SessionSmCreator",
                              "root" )

def _registerBootstrapHandlers( h, em ):
   dirHandler = h.entityFilterDir.dirHandler
   registerCustomCopyHandler( em,
                              "", "Tac::Dir",
                              dirHandler,
                              direction="both" )

def doMounts( em, block=True, callback=None, bootstrap=True ):
   global ancestorDir, cliConfig, pendingSession, sessionConfig
   global sessionStatus, sessionEm

   t0( "doMounts(", em.sysname(), ", block", block, ")" )
   assert not ( block and not bootstrap ), "block=True requires bootstrap=True"

   h = handlerDir( em )
   if sessionEm == em and sessionStatusInitialized( em ):
      t0( "sysname", em.sysname(), "already mounted for sysname" )
      if bootstrap:
         _registerBootstrapHandlers( h, em )
      if callback:
         callback()
      return

   mg = em.mountGroup()
   # clear previous data
   h.status = h.configRoot = None

   _cleanConfig = mg.mount( "session/cleanConfig", "Tac::Dir", "wi" )
   _cleanConfigStatus = mg.mount( "cli/session/cleanConfigStatus",
                                 "Cli::Session::CleanConfigStatus", "w" )
   _configRoot = mg.mount( "Sysdb/configRoot", "Sysdb::ConfigRoot", "ri" )

   ancestorDir = mg.mount( "session/ancestorDir", "Tac::Dir", "wi" )
   cliConfig = mg.mount( "cli/session/input/config",
                         "Cli::Session::CliConfig", "wi" )
   pendingSession = mg.mount( "session/sessionDir", "Tac::Dir", "wi" )
   sessionConfig = mg.mount( "cli/session/config",
                             "Cli::Session::Config", "wi" )
   sessionStatus = mg.mount( "cli/session/status",
                             "Cli::Session::Status", "wi" )
   sessionStateDir = mg.mount( "cli/session/state",
                               "Cli::Session::SessionStateDir", "wi" )
   sessionEm = em

   def _init():

      h.status = sessionStatus
      h.configRoot = _configRoot

      if bootstrap:
         t0( "initialize sessionConfig and sessionStatus" )
         sessionConfig.pendingChangeConfigDir = ( "pendingChangeConfig", )
         sessionStatus.ancestorDir = ancestorDir
         sessionStatus.cleanConfig = _cleanConfig
         sessionStatus.cleanConfigStatus = _cleanConfigStatus
         sessionStatus.sessionDir = pendingSession
         sessionStatus.sessionStateDir = sessionStateDir
         sessionStatus.pendingChangeStatusDir = ( "pendingChangeStatus", )

         _registerBootstrapHandlers( h, em )
         sessionStatus.initialized = True

      if callback:
         callback()

   mg.close( callback=_init, blocking=False )
   if block:
      waitForSessionStatusInitialized( em )

pendingChangeConfigReactor = None

def startCohabitingAgent( em, redundancyStatus=None, callback=None,
                          block=True, noPlugins=False ):
   t0( "startCohabitingAgent( sysname", em.sysname(), ")" )
   def createSm():
      t0( "Agent mounts complete" )
      # Load all ConfigSessionPlugins
      if not noPlugins:
         loadPlugins( em )

      global pendingChangeConfigReactor
      pendingChangeConfigReactor = PendingChangeConfigReactor(
         sessionConfig.pendingChangeConfigDir )

      t0( "Cohabiting CliSessionMgr starting for", em.sysname() )
      sessionConfig.userConfig = cliConfig
      h = handlerDir( em )

      t1( "Maybe create a cleanConfig" )
      if not sessionStatus.cleanConfigStatus.cleanConfigComplete:
         t0( "Creating clean config" )
         configRootClean = em.root().entity[ 'Sysdb/configRoot' ]
         t0( "rootsComplete:", configRootClean.rootsComplete )
         assert configRootClean.rootsComplete

         t0( "Creating clean config" )
         h.doCreateCleanConfig( em.root(),
                                em.root() )
         t0( "Created clean config successfully" )

      agent = getAgent( em )
      agent.cliSessionSm = ( sessionConfig,
                             sessionStatus,
                             h.openSessionConfigDir,
                             h.openSessionStatusDir,
                             h,
                             em.root(),
                             pendingSession,
                             em.cEntityManager(),
                             # Won't work if we do a cohabiting switchover
                             # breadth test ...
                             None, None )

   def mountsComplete():
      global redundancyReactor
      if redundancyStatus:
         redundancyReactor = RedundancyReactor( redundancyStatus, em,
                                                createSm )
      else:
         createSm()

      if callback:
         callback()

   doMounts( em, callback=mountsComplete, block=block )

def cohabitingAgentReady( em ):
   agent = getAgent( em, create=False )
   return bool( agent and agent.cliSessionSm )

# In a breadth test, if you are killing Sysdb, too, pass in cleanupSysdb=True
# in order to be able to start up a cohabitingAgent using the new Sysdb.
# It would be best if you used a distinct sysname, too.  Even so, this
# is tricky to try.
def stopCohabitingAgent( em, cleanupSysdb=False ):
   agentDir = getAgentDir( em )
   agent = agentDir.get( 'root' )
   if not agent:
      return
   agent.cliSessionSm.sessionConfigReactor.clear()
   agent.cliSessionSm.openSessionDirReactor = None
   agent.cliSessionSm.sessionConfigDirReactor = None
   if agent.cliSessionSm.sessionCleanupReactor:
      agent.cliSessionSm.sessionCleanupReactor.idleSessionReactor = None

   agent.cliSessionSm.sessionCleanupReactor = None
   agent.cliSessionSm = None
   agent = None
   agentDir.deleteEntity( "root" )
   if cleanupSysdb:
      cleanupSessionMgrState()

def cleanupSessionMgrState():
   t0( "cleanupSessionMgrState" )
   global sessionEm
   global ancestorDir, cliConfig, pendingSession, sessionConfig, sessionStatus
   global sessionOnCommitHandlers

   sessionEm = None

   ancestorDir = None
   cliConfig = None
   pendingSession = None
   sessionConfig = None
   sessionStatus = None

   sessionOnCommitHandlers = {}

def cleanupCohabitingAgentIfNeeded( em ):
   agent = getAgent( em )
   if not agent:
      return
   stopCohabitingAgent( em, cleanupSysdb=True )

def internalSessionName( sessionName ):
   """'internal' sessions are sessions that are not exposed to the Cli, and are
   not managed (creation/deletion) by the standard session management of open
   and committed sections."""
   # This function assumes that sessionName is otherwise valid
   return sessionName.startswith( '@' )

def validInternalSessionName( sessionName ):
   """'internal' sessions are sessions that are not exposed to the Cli, and are
   not managed (creation/deletion) by the standard session management of open
   and committed sections.
   Returns True iff sessionName has the structure of an internal name"""
   return validSessionName( sessionName ) and internalSessionName( sessionName )

def sessionNames( entityManager=None, allowInternalSessions=False ):
   waitForSessionStatusInitialized( entityManager )
   return [ name for name, state in sessionStatus.sessionStateDir.state.items()
            if ( state != "aborted" and
                 ( allowInternalSessions or not internalSessionName( name ) ) ) ]

@Ark.synchronized()
def uniqueSessionNameCounter():
   global __sessionNameCounter__
   counter = __sessionNameCounter__
   __sessionNameCounter__ += 1
   return counter

def uniqueSessionName( entityManager=None, prefix="s", okToReuse=False ):
   waitForSessionStatusInitialized( entityManager )
   assert validSessionName( prefix )
   while True:
      if okToReuse:
         counter = __sessionNameCounter__
      else:
         counter = uniqueSessionNameCounter()
      candidate = "%s%x" % ( prefix, counter )
      if ( sessionStatus.pendingChangeStatusDir and
           sessionConfig.pendingChangeConfigDir and
           not ( sessionStatus.pendingChangeStatusDir.status.get( candidate ) or
                 sessionConfig.pendingChangeConfigDir.config.get( candidate ) ) ):
         return candidate
      else:
         okToReuse = False

def getSessionUserName():
   username = ( os.environ.get( "LOGNAME" ) or
                os.environ.get( "USER", "[unknown]" ))
   return username

def getPrettyTtyName():
   terminal = os.getenv( "REALTTY" )
   if not terminal:
      for fd in ( 0, 1, 2 ):
         try:
            terminal = os.ttyname( fd )
            break
         except OSError:
            pass
   if not terminal:
      terminal = ( os.environ.get( "SSH_TTY" ) or
                   os.environ.get( "STY" ) or
                   os.ctermid() )
   # make terminal name prettier (patterned after Cli/UtmpDump)
   if terminal:
      # Strip off a leading '/dev/', if any
      terminal = terminal.replace( '/dev/', '' )
      terminal = terminal.replace( 'pts/', 'vty' ).replace( 'ttyS', 'con' )
   return terminal

def validSessionName( sessionName ):
   def validChar( c ):
      return c.isalnum() or c in "_-"
   if not sessionName or not isinstance( sessionName, str ):
      return False
   s0 = sessionName[ 0 ]
   return ( all( validChar( c ) for c in sessionName[ 1 : ] ) and
            # @<name> is used to encode internal sessions
            ( validChar( s0 ) or s0 == "@" ) )

class SessionAlreadyCompletedError( Exception ):
   def __init__( self, sessionName ):
      # NOTE: this message is displayed by the CLI.
      msg = "Session %s is already completed" % sessionName
      Exception.__init__( self, msg )

def isSessionPendingCommitTimer( sessionName ):
   return sessionStatus.sessionStateDir.commitTimerSessionName == sessionName

def commitTimerSessionName():
   return sessionStatus.sessionStateDir.commitTimerSessionName

def isCommitTimerInProgress():
   return sessionStatus.commitTimerInProgress

def isSessionPresent( sessionName ):
   """Return True if the specified session is present, False otherwise. """
   return bool( sessionConfig.pendingChangeConfigDir.config.get( sessionName ) or
                sessionStatus.pendingChangeStatusDir.status.get( sessionName ) )

def canCommitSession( em, sessionName ):
   """Return "" if we can successfully commit a  session, or return an
      error message."""
   if sessionName is None:
      # We are already in config session mode.
      sessionName = currentSession( em )
   if not isSessionPresent( sessionName ):
      return "Cannot commit non-existent session %s." % sessionName
   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if pcs:
      if pcs.completed:
         if not isSessionPendingCommitTimer( sessionName ):
            return "Cannot commit session %s (already completed)" % sessionName
      elif isCommitTimerInProgress():
         return  "Cannot commit session %s (another session %s is pending commit" \
            " timer)" % ( sessionName,
                          sessionStatus.sessionStateDir.commitTimerSessionName )
      elif pcs.noCommit:
         return  "Cannot commit no-commit session %s" % sessionName
   return ""

def canEnterSession( sessionName, entityManager, noCommit ):
   """Return "" if we can successfully enter the specified session, or
   return an error message otherwise."""
   waitForSessionStatusInitialized( entityManager )
   if sessionName and not validSessionName( sessionName ):
      return "Session name %s is invalid." % sessionName

   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if pcs:
      if pcs.completed:
         if isSessionPendingCommitTimer( sessionName ):
            return "Cannot enter session %s (pending commit timer)" % sessionName
         else:
            return "Cannot enter session %s (already completed)" % sessionName
      if noCommit and not pcs.noCommit:
         return ( "Cannot change an already existing session %s to no-commit"
                  % sessionName )
      if ( pcs.noCommit and
           not internalSessionName( sessionName ) and
           pcs.noCommitUser != getSessionUserName() ):
         return ( "User %s cannot enter no-commit session %s created by %s"
                  % ( getSessionUserName(), sessionName, pcs.noCommitUser ) )
   return ""

def enterSession( sessionName, entityManager=None, transient=False, noCommit=False,
                  description=None, mergeOnCommit=False, dynamicConfigSource=None ):
   """This process will encapsulate all Cli config commands in a session,
   and the commands won't be applied until the session is committed. If
   we are unable to enter the session, this will signal Tac.Timeout, or
   return a non-empty string explaining why we can't.

   If transient is true, the session is exempted from being cleaned up due to
   idleness or max limits (as it'll be deleted automatically).
   """
   t0( 'Entering session:', sessionName )
   if not entityManager:
      entityManager = sessionEm
   response = canEnterSession( sessionName, entityManager, noCommit )
   if response != "":
      return response

   if not sessionName:
      sessionName = uniqueSessionName( entityManager )

   configSet = sessionConfig.pendingChangeConfigDir.config
   statusSet = sessionStatus.pendingChangeStatusDir.status
   if not sessionName in configSet:
      t0( "Creating session config for", sessionName )
      pcc = configSet.newMember( sessionName, transient, Tac.now() )
      t0( "session config created" )
      # only set the mergeOnCommit when the session is first created
      # ie: mergedOnCommit cannot be modified later
      pcc.mergeOnCommit = mergeOnCommit
   sleep = not Tac.activityManager.inExecTime.isZero
   Tac.waitFor( lambda: sessionName in statusSet, sleep=sleep,
                warnAfter=WARN_WAIT_TIME,
                description="session to be created [pcs]" )

   # exit in case we were already in one
   exitSession( entityManager )

   pcc = configSet[ sessionName ]
   if noCommit and not pcc.noCommit:
      pcc.noCommit = noCommit
      pcc.noCommitUser = getSessionUserName()
   if description is not None:
      pcc.userString = description
   if dynamicConfigSource is not None:
      pcc.dynamicConfigSource = dynamicConfigSource
   pid = gettid()
   h = handlerDir( entityManager or sessionEm )
   osc = h.openSessionConfigDir.openSession.get( pid )
   if osc and osc.sessionName != sessionName:
      # clean up previous entry
      del h.openSessionConfigDir.openSession[ pid ]
   osc = h.openSessionConfigDir.openSession.newMember( pid, sessionName )
   osc.user = getSessionUserName()
   osc.terminal = getPrettyTtyName()
   Tac.waitFor( lambda: ( pid in h.openSessionStatusDir.openSession and
                          h.openSessionStatusDir.openSession[ pid ].session == pcc ),
                sleep=sleep,
                warnAfter=WARN_WAIT_TIME,
                description="session to be created [sessionStatus]" )
   cachedSessionNameIs( sessionName )
   t0( "Entered CliSession", sessionName )
   return ""

def _cleanSessionConfig( em, sessionName ):
   assert em
   pid = gettid()
   h = handlerDir( em )
   osc = h.openSessionConfigDir.openSession.get( pid )
   if osc and osc.sessionName == sessionName:
      del h.openSessionConfigDir.openSession[ pid ]
      Tac.waitFor( lambda: pid not in h.openSessionStatusDir.openSession,
                   sleep=not Tac.activityManager.inExecTime.isZero,
                   warnAfter=WARN_WAIT_TIME,
                   description="oss.session to be None" )

def exitSession( entityManager=None ):
   sessionName = cachedSessionName()
   t5( "exitSession", sessionName if sessionName else "(not in session)" )
   if not sessionName:
      # nothing to do
      return
   waitForSessionStatusInitialized( entityManager )
   pid = gettid()
   h = handlerDir( entityManager or sessionEm )
   osc = h.openSessionConfigDir.openSession.get( pid )
   cachedSessionNameIs( "" )
   if osc:
      del h.openSessionConfigDir.openSession[ pid ]
   Tac.waitFor( lambda: pid not in h.openSessionStatusDir.openSession,
                sleep=not Tac.activityManager.inExecTime.isZero,
                warnAfter=WARN_WAIT_TIME,
                description="oss.session to be None" )

def addSessionRoot( em, root, sessionName=None ):
   if sessionName is None:
      sessionName = currentSession( em )
   assert sessionName, "Cannot addRoot when not in Cli session"
   waitForSessionStatusInitialized( em )

   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if not pcs or pcs.completed:
      raise SessionAlreadyCompletedError( sessionName )

   # Note that currentSession already called ensureSessionMounts()
   root = EntityManager.cleanPath( root )
   if root and root[ -1 ] == "/":
      root = root[ 0: -1 ]

   t0( "addSessionRoot sessionName", sessionName, "root", root )
   response = sessionCommand( em, sessionName, "addRoot", cmdArg=root )
   if response:
      print( response )
   return response

def sessionConfigRoots( em, sessionName=None ):
   if sessionName is None:
      sessionName = currentSession( em )
   assert sessionName, "No relevant session"
   waitForSessionStatusInitialized( em )

   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if not pcs:
      assert sessionName, "Session %s is no longer available" % sessionName

   return list( pcs.localRoot )

def sessionCommand( em, sessionName, command,
                    warnAfter=WARN_WAIT_TIME,
                    # for debugging only
                    requestOnly=False, responseOnly=False,
                    cmdArg="", cmdFlags=0 ):
   """For debugging we allow caller to split sessionCommand into two parts.
   The value returned by requestOnly=True must be passed as the value of
   the responseOnly keyword arg in order to complete the original command.
   The caller can tamper with state between the 2 calls.
   If requestOnly is True, we initiate a new call, and return after we
   see that the mgr agent has gotten the request."""
   assert not ( requestOnly and responseOnly )
   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   pcc = sessionConfig.pendingChangeConfigDir.config.get( sessionName )
   pid = gettid()
   responseErrMsg = 'response for PID %d to appear in PendingChangeStatus'
   # Note, we never cleanup commandRequestDir until the session is deleted.
   commandRequestDir = pcc.commandDir.newMember( pid )

   if responseOnly:
      cid, sleep, oldCommand = responseOnly
      assert oldCommand == command
   else:
      commandRequestDir.commandId += 1
      cid = commandRequestDir.commandId
      t2( "sessionCommand: session", sessionName, "command", command, "id", cid )
      commandRequestDir.command = Tac.Value( "Cli::Session::CommandRequest",
                                             id=cid,
                                             cmd=command,
                                             cmdArg=cmdArg,
                                             cmdFlags=cmdFlags )
      sleep = not Tac.activityManager.inExecTime.isZero
      if requestOnly:
         return ( cid, sleep, command )

   commandResponseDir = Tac.waitFor( lambda : pcs.commandResponse.get( pid ),
                                     sleep=sleep, warnAfter=warnAfter,
                                     description=( responseErrMsg % pid ) )
   Tac.waitFor( lambda : commandResponseDir.command.id == cid,
                sleep=sleep, warnAfter=warnAfter,
                description='command id %d to be processed' % cid )
   response = commandResponseDir.response
   if response:
      # first char of response is "+" for success or "-" for failure.
      # If "+", then rest of string should be ""
      response = response[ 1: ]
   return response

def deleteSession( em, sessionName ):
   t0( 'Deleting session:', sessionName )
   configSet = sessionConfig.pendingChangeConfigDir.config
   statusSet = sessionStatus.pendingChangeStatusDir.status
   pcc = configSet.get( sessionName )
   if not pcc:
      t0( "Session", sessionName, "not removed: config not found" )
      return "Cannot delete non-existent session " + sessionName
   # Make sure we're not in the session, and clean any command queue we may
   # have put there.
   response = sessionCommand( em, sessionName, "remove" )
   if response:
      t0( "Session", sessionName, "not removed:", response )
      return response
   sleep = not Tac.activityManager.inExecTime.isZero
   Tac.waitFor( lambda: sessionName not in statusSet, sleep=sleep,
                warnAfter=WARN_WAIT_TIME,
                description="session to be marked for deletion" )
   discardSessionOnCommitHandlers( sessionName )
   _cleanSessionConfig( em, sessionName )
   return ""

def abortSession( em, sessionName=None ):
   if sessionName is None:
      sessionName = currentSession( em )
   t0( 'Aborting session:', sessionName )
   if not sessionName:
      return "Not in Cli session"

   discardSessionOnCommitHandlers( sessionName )
   pcsDir = sessionStatus.pendingChangeStatusDir.status
   # If pcc deleted, then pcs deleted.
   pcc = sessionConfig.pendingChangeConfigDir.config.get( sessionName )
   if not pcc:
      return ""
   sleep = not Tac.activityManager.inExecTime.isZero
   # should never wait for very long
   pcs = Tac.waitFor( lambda : pcsDir.get( sessionName ), sleep=sleep,
                      warnAfter=WARN_WAIT_TIME,
                      description="CliSessionMgr to create the session.")

   if ( pcs.completed and pcs.success and
        not isSessionPendingCommitTimer( sessionName ) ):
      return "Too late to abort; session already committed."

   response = sessionCommand( em, sessionName, "abort" )
   if response:
      print( response )
      return response
   try:
      sleep = not Tac.activityManager.inExecTime.isCurrent

      def _sessionDeletedOrAborted():
         pcs_ = pcsDir.get( sessionName )
         # the session might be deleted immediately
         return not pcs_ or pcs_.sessionState == "aborted"

      # The state can be deleted when the session is aborted
      Tac.waitFor( _sessionDeletedOrAborted,
                   sleep=sleep,
                   warnAfter=WARN_WAIT_TIME,
                   description="session to abort" )
   except Tac.Timeout:
      return "Abort timed out"

   return ""

def commitSession( em, debugAttrChanges=None, timerValue=None, sessionName=None,
                   mode=None ):
   '''Returns an empty string on success, an error string if the session
   could not be committed, or None if the session no longer exists.'''
   t0( "commitSession()", 'debugAttrChanges', debugAttrChanges,
       'sessionName', sessionName, 'timerValue', timerValue )
   if sessionName is None:
      sessionName = currentSession( em )
   assert sessionName, "Not in Cli session"
   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   pcc = sessionConfig.pendingChangeConfigDir.config.get( sessionName )
   if not pcc:
      t0( "Session", sessionName, "not committed: config not found" )
      discardSessionOnCommitHandlers( sessionName )
      return None
   pcs.commitUser = getSessionUserName()
   if debugAttrChanges is None:
      # there is a test command that sets this flag
      debugAttrChanges = pcc.logAttributeEdits
   else:
      pcc.logAttributeEdits = bool( debugAttrChanges )
   pcc.commitTimeout = timerValue or 0

   # If timerValue is None, this is due to a "commit".
   # If timerValue is not None, this is due to a "commit timer". In that case,
   # do not commit if it is only a timerValue change(not the first commit
   # timer command).
   if pcs.sessionState != "committed":
      try:
         runPreCommitHandlers( em, sessionName )
      except PreCommitHandlerError as e:
         return e.recordAndResponse()
   t0( 'Committing session:', sessionName )
   response = sessionCommand( em, sessionName, "commit" )
   if response:
      pcc.logAttributeEdits = False
      return response
   try: # pylint: disable=too-many-nested-blocks
      if isCommitTimerInProgress():
         Tac.waitFor( lambda: ( pcs.success ),
                      warnAfter=WARN_WAIT_TIME,
                      description="session to commit" )
      if debugAttrChanges is not None:
         output = []
         for attrEditEntry in pcs.attributeEditLog.attributeEditEntry.values():
            attrEdit = attrEditEntry.attributeEdit
            for path in attrEditEntry.pathCounter:
               if not output:
                  tableHeadings = ( 'ParentName', 'AttrName', 'AttrType',
                                    'Op' )
                  # set tableWidth to unlimited so that we do not wrap around.
                  # this is easier to create logres.
                  table = TableOutput.createTable( tableHeadings, tableWidth=9999 )
                  f1 = TableOutput.Format( justify='left' )
                  f1.noPadLeftIs( True )
                  f2 = TableOutput.Format( justify='left' )
                  f2.noPadLeftIs( True )
                  f3 = TableOutput.Format( justify='left' )
                  table.formatColumns( f1, f2, f2, f3 )

               line = ( path, attrEdit.attributeName,
                        attrEdit.attributeType.split( '::' )[ -1 ],
                        # We cannot use 'set' in tac model, so we translate
                        # 'createOrModify' to 'set'
                        'set' if attrEdit.editType == 'createOrModify' else
                        attrEdit.editType )
               if line not in output:
                  table.newRow( *line )
                  output.append( line )
         if output:
            print( table.output() )
   except Tac.Timeout:
      return "Commit timed out"
   finally:
      pcc.logAttributeEdits = False

   return ""

def closeSession( em, sessionName=None, mode=None ):
   '''Returns an empty string on success, an error string if the session
   could not be closed, or None if the session no longer exists.'''
   t0( "commitSession()", 'sessionName', sessionName )
   if sessionName is None:
      sessionName = currentSession( em )
   assert sessionName, "Not in Cli session"
   pcc = sessionConfig.pendingChangeConfigDir.config.get( sessionName )
   if not pcc:
      # At the moment we don't expect general user-visible sessions to get
      # closed: only committed or aborted. But, just in case:
      reason = f"Session {sessionName} not closed: config not found"
      t0( reason )
      discardSessionOnCommitHandlers( sessionName )
      return reason
   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if pcs.completed and not pcs.sessionState == "closed":
      return f"Cannot close session; state is already {pcs.sessionState}."

   # This, also, should never have been set. But defensively:
   pcc.commitTimeout = Tac.endOfTime
   # What's the point of closing, without committing, if you don't want to
   # keep it around?
   pcc.cleanupAfterCommit = False

   response = ""
   t0( 'Closing session:', sessionName )
   response = sessionCommand( em, sessionName, "close" )
   return response

def rollbackSession( em, sessionName=None, localRootsOnly=False ):
   t0( "rollbackSession sessionName", sessionName, "localRootsOnly", localRootsOnly )
   sessionName = sessionName if sessionName is not None else currentSession( em )
   assert sessionName, "Not in Cli session"
   # Should only happen during breadth tests:
   if not sessionStatus.cleanConfigStatus.cleanConfigComplete:
      return "Cannot rollback because no clean-config exists."

   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   command = "clearConfig"
   flagsType = 'Cli::Session::SessionCommandFlags'
   cmdFlags = Tac.enumValue( flagsType, 'noFlags' )
   if localRootsOnly:
      cmdFlags = Tac.enumValue( flagsType, 'localRootsOnly' )
   clearCount = pcs.clears
   response = sessionCommand( em, sessionName, command, cmdArg="",
                              cmdFlags=cmdFlags )
   if response:
      print( response )
      return response
   try:
      Tac.waitFor( lambda: pcs.clears != clearCount, warnAfter=WARN_WAIT_TIME,
                   description="session config state to be cleared" )
   except Tac.Timeout:
      return "rollback timed out"
   return ""

# You can rollback/copy from a saved, completed, session.
# It is more efficient than using a saved config file because
# there is no Cli parsing needed. On the downside, the granularity is
# coarser: you copy on a per-root basis, not per-command.
def copyFromSession( em, fromSession ):
   sessionName = currentSession( em )
   if not sessionName:
      return "Not in Cli session"
   pcs = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if not pcs:
      msg = "Cannot copy from session %s (session does not exist)"
      return msg % sessionName
   # pcs exists
   if pcs.completed:
      msg = "Cannot rollback session %s (already completed)"
      return msg % sessionName
   # This is not really a limitation if we wanted to support it. Real
   # constraint (assuming we don't want to lose any existing content) is that
   # we can't allow pcs.localPrefix() and fromPcs.localPrefix to share any
   # roots. But for now, we only allow empty pcs.localPrefix
   if not pcs.localPrefix.empty():
      msg = "Cannot rollback session %s: "
      msg += "rollback can only be performed at the start of session."
      return msg % sessionName

   fromPcs = sessionStatus.pendingChangeStatusDir.status.get( fromSession )
   if not ( fromPcs and fromPcs.completed and fromPcs.success and
            ( fromPcs.sessionState == "committed" or
              fromPcs.sessionState == "closed" ) ):
      msg = "Cannot copy from session %s"
      if not fromPcs:
         msg += " (session does not exist)"
      elif not ( fromPcs.completed and fromPcs.success and
                 ( fromPcs.sessionState == "committed" or
                   fromPcs.sessionState == 'closed' ) ):
         msg += " (not successfully completed)"
      return  msg % fromSession
   fromRoot = pendingSession.get( fromSession )
   if not list( fromRoot ):
      msg = "configuration state of session %s has been reclaimed"
      return msg % fromSession

   h = handlerDir( em )
   if h.doCopySession( fromSession, sessionName, em.root() ):
      pcs.rollbackSession = fromSession
      # pcs.clears += 1  # how do we mark it complete?
      return ""
   else:
      return "replace config did not succeed."

def sessionSize( em, sessionName=None, warnAfter=WARN_WAIT_TIME ):
   """Return the amount of memory used by the given sessionName or current session
   if sessionName is None.
   Note that it's computed when this function is called, so it is an expensive
   operation."""
   if not sessionName:
      sessionName = currentSession( em )
   before = Tac.now()
   size = sessionCommand( em, sessionName, "size", warnAfter )
   t0( 'Computing size of session', sessionName, 'took', Tac.now() - before,
       'seconds' )
   return int( size )

def validFilterAttribute( typeName, attr ):
   """True if attribute 'attr' is a logging attribute of type 'typeName'"""

   t = Tac.typeNode( typeName )
   return t.attr( attr ) is not None and t.attr( attr ).isLogging

def validAttribute( typeName, attr ):
   """True if attribute 'attr' is an attribute of type 'typeName'"""

   t = Tac.typeNode( typeName )
   return t.attr( attr ) is not None

def registerCustomCopyHandler( em, path, typeName, handler, direction="out" ):
   """
   We can register a special handler for entityCopy either by path, or
   by type, or by both.

   path-only is most specific --- it applies only to the specific entity at
   that path.

   If you pass path and typename, then it applies to all entities of
   exactly type == typename that are in the subtree rooted at path.
   Finally, if you pass in only the typename, then the handler applies
   to entities of type typename, anywhere they are encountered (assuming
   not overridden by a path).

   So a handler for type Foo::Bar is overridden, for entities under the
   x/y subtree by a handler for type Foo::Bar scoped to x/y, and *that*
   is overridden, for the entity x/y/a/b/c by a handler registered
   specifically for x/y/a/b/c

   (From comments in CliSession.tac)

   direction can be either "in", "out", or "both"
   """
   assert handler
   h = handlerDir( em )

   if direction == "in" or direction == "both":
      h.inCopyHandlerTable.registerCustomCopyHandler( path, typeName, handler )
      if path:
         # if a path has in customer copy handler, during rollback we need to
         # copy from clean-config into the session, as it may have side effects.
         h.rollbackForcedRoot.prefixIs( path,
                                        Tac.newInstance( "Ark::PathTrieState" ) )
   if direction == "out" or direction == "both":
      h.copyHandlerTable.registerCustomCopyHandler( path, typeName, handler )

def registerEntityCopyFilter( em, path, typeName, filteredAttrs,
                              orderedAttrsBefore=None,
                              orderedAttrsAfter=None,
                              direction="out" ):
   """Skip copying the attributes in filteredAttrs.  Optionally impose an order
   on the copying: first copy orderedAttrsBefore, in the order of the list.
   Then copy all other attrs not in filteredAttrs, orderedAttrsBefore, or
   orderedAttrsAfter.  And, finally, optionally copy (in the order of the list)
   copy the attributes named in the list orderedAttrsAfter

   See help string from registerCustomCopyHandler for more info"""
   h = handlerDir( em )
   if not ( orderedAttrsBefore or orderedAttrsAfter ):
      attrFilter = Tac.newInstance( "Cli::Session::AttributeFilter" )
      for attr in filteredAttrs:
         assert validFilterAttribute( typeName, attr )
         attrFilter.attributeName[ attr ] = True
      if direction == "in" or direction == "both":
         h.inCopyHandlerTable.registerEntityCopyFilter( h.entityFilterDir, path,
                                                        typeName, attrFilter )
      if direction == "out" or direction == "both":
         h.copyHandlerTable.registerEntityCopyFilter( h.entityFilterDir, path,
                                                      typeName, attrFilter )

   else:
      def buildFilter( ecFilterDir ):
         key = Tac.Value( "Cli::Session::HandlerContext", path=path,
                          typeName=typeName )
         ecf = ecFilterDir.ecf.newMember( key )
         ecf.filteredAttribute = ()
         for attr in filteredAttrs:
            assert validFilterAttribute( typeName, attr )
            ecf.filteredAttribute.attributeName[ attr ] = True

         if orderedAttrsBefore:
            ecf.preAttribute = ()
            for i, attr in enumerate( orderedAttrsBefore ):
               assert validFilterAttribute( typeName, attr )
               ecf.preAttribute.attributeName[ i ] = attr
               ecf.filteredAttribute.attributeName[ attr ] = True

         if orderedAttrsAfter:
            ecf.postAttribute = ()
            for i, attr in enumerate( orderedAttrsAfter ):
               assert validFilterAttribute( typeName, attr )
               ecf.postAttribute.attributeName[ i ] = attr
               ecf.filteredAttribute.attributeName[ attr ] = True
         return ecf

      ecf = buildFilter( h.entityFilterDir )
      if direction == "in" or direction == "both":
         h.inCopyHandlerTable.registerCustomCopyHandler( path, typeName, ecf )
      if direction == "out" or direction == "both":
         h.copyHandlerTable.registerCustomCopyHandler( path, typeName, ecf )

def registerAttributeEditLogBypass( em, parentPath, attrName, attrType,
                                    attrEditType, defaultEnt=None ):
   """ Ideally, no attributes should be changed during "config replace
   running-config force". Some exceptions are, however, found to be acceptable.
   Agents can use this API to register exceptions. Registered attributes will
   not be shown in "debug-attribute".
   We find that some agents do not delete a config entity even when all its
   configurable fields are default, but it is legal for config replace to delete
   this entity. To ensure that the entity deleted by config replace has default
   setting, agents can further provide a defaultEnt to which all deleted entities
   (of the same type) are compared during entity copy. Note that the comparison
   excludes construtor parameters and hence the defaultEnt can be instantiated
   using dummy construtors.

   parentPath: the path to the bypassed entity relative to the root. If an
               absolute path is passed in (with a leading '/'), it will be
               automatically converted to a relative path.
   attrName: the attribute name as reported in debug-attribute output. "" can
             be specified to apply to all attribute names.
   attrType: the attribute type as reported in debug-attribute output. "" can
             be specified to apply to all attribute types.
   attrEditType: 'set' or 'del'.
   defaultEnt: when specified, EntityCopy compares it against the entity that
               is in the destination but not the source. The entity is bypassed
               by debug-attribute if all non-constructor attributes are the same.
   """
   h = handlerDir( em )
   assert attrEditType in ( 'set', 'del' )
   ae = Tac.Value( "Cli::Session::AttributeEdit", attrName, attrType,
                   'createOrModify' if attrEditType == 'set' else 'del' )
   if parentPath.startswith( '/' ):
      # strip the first two components
      parentPath = '/'.join( parentPath.split( '/' )[ 3: ] )
   aeBypassEntry = h.attributeEditBypassTable.attributeEditBypassEntry.\
                   newMember( ae )
   aeBypassEntry.parentPath[ parentPath ] = True
   if defaultEnt:
      aeBypassEntry.defaultEnt = defaultEnt

def registerEntityCopyDebug( entityManager, attr, entityType="", pathname="",
                             editType="set", action=None ):
   assert sessionConfig.userConfig
   assert editType in [ "set", "del" ]
   assert attr
   assert action in [ None, "actionAssert", "actionTrace" ]
   assert cliConfig == sessionConfig.userConfig

   key = Tac.Value( "Cli::Session::AttributeEdit",
                    attributeName=attr, attributeType=entityType,
                    editType=editType )
   if action: # add an entry
      if entityType:
         ecDebugConfig = cliConfig.debugPerTypeEntityCopy.get( key )
         if not ecDebugConfig:
            ecDebugConfig = cliConfig.debugPerTypeEntityCopy.newMember( key )
         ecDebugConfig.action[ pathname ] = action
      else:
         ecDebugConfig = cliConfig.debugEntityCopy.get( pathname )
         if not ecDebugConfig:
            ecDebugConfig = cliConfig.debugEntityCopy.newMember( pathname )
         entry = ecDebugConfig.debugConfig.get( key )
         if not entry:
            entry = ecDebugConfig.debugConfig.newMember( key )
         entry.action = action
   else: # delete the entry for this description
      if entityType:
         ecDebugConfig = cliConfig.debugPerTypeEntityCopy.get( key )
         if ecDebugConfig:
            if ecDebugConfig.action.get( pathname ):
               del ecDebugConfig.action[ pathname ]
            if not ecDebugConfig.action:
               del cliConfig.debugPerTypeEntityCopy[ key ]
      else:
         ecDebugConfig = cliConfig.debugEntityCopy.get( pathname )
         if ecDebugConfig:
            entry = ecDebugConfig.debugConfig.get( key )
            if entry:
               del ecDebugConfig.debugConfig[ key ]
            if not entityType and not ecDebugConfig.debugConfig:
               del cliConfig.debugEntityCopy[ pathname ]

def registerConfigGroup( em, cfgGroupName, mountPath ):
   """ Register 'mountPath' as a member of config-group 'cfgGroupName'. Adds to
   any previously registered mounts
   """
   assert cfgGroupName == "airstream-cmv"
   h = handlerDir( em )
   assert mountPath in h.configRoot.root, 'Not a config root: %s' % mountPath

   mountPath = EntityManager.cleanPath( mountPath )
   h.airstreamConfigGroup.add( mountPath )

def registerConfigRootDependency( em, path, dependent ):
   """Register that dependent depends on path (the former having a Ptr to
   the latter). This causes EntityCopy to copy them together.

   To keep things simple:
   1. No chain dependency is allowed. A path being dependent on cannot depend
      on another path.
   2. A path can at most depend on one other path.
   """
   assert '*' not in path
   assert '*' not in dependent
   h = handlerDir( em )
   assert dependent not in h.rootDependencyTable.root
   for p in h.rootDependencyTable.root.values():
      assert path not in p.dependent
      if p.name != path:
         assert dependent not in p.dependent

   root = h.rootDependencyTable.newRoot( path )
   root.dependent.add( dependent )

def unregisterConfigRootDependency( em, path, dependent ):
   # This is just for testing so we can try different dependencies
   h = handlerDir( em )
   root = h.rootDependencyTable.root.get( path )
   if root:
      del root.dependent[ dependent ]
      if not root.dependent:
         del h.rootDependencyTable.root[ path ]

def registerConfigRootFilter( em, name, mountPath ):
   """ Register 'mountPath' to be filtered from config root.
   """
   t0( "registerConfigRootFilter", name, mountPath )
   h = handlerDir( em )
   assert mountPath in h.configRoot.root, 'Not a config root: %s' % mountPath
   mountPath = EntityManager.cleanPath( mountPath )
   t9( "Adding mountpoint:", mountPath, "to filter", name )
   table = h.configRootFilterTable.newFilter( name )
   table.root[ mountPath ] = True
   state = Tac.newInstance( "Ark::PathTrieState" )
   table.rootTrie.prefixIs( mountPath, state )

def activeConfigRootFilterIs( em, name ):
   """ Set the active config root filter table. Note, the API currently doesn't
   work (very well) for the existence of multiple filters. This should be addressed
   in case the need arises. """
   h = handlerDir( em )
   if name:
      t0( "set active config root filter to", name )
      h.activeConfigRootFilter = h.configRootFilterTable.filter.get( name )
      # Only allow filtering of dependees if all dependents are already filtered.
      # This means that dependents must be registered before dependees
      for root in h.activeConfigRootFilter.root:
         if root in h.rootDependencyTable.root:
            assert( all( dependent in h.activeConfigRootFilter.root for dependent in
                         h.rootDependencyTable.root[ root ].dependent ) ), \
                     "Root %s cannot be filtered as it's depended on " % root + \
                     "by others that are not members of the filter"
   else:
      t8( "restore active config root filter to default" )
      h.activeConfigRootFilter = None

def activeConfigRootFilter( em ):
   h = handlerDir( em )
   return h.activeConfigRootFilter

def registerDynamicConfigPath( em, staticPath, dynamicPath ):
   """ Map a static CLI config path onto a dynamic path Tac::Dir path.

   A dynamic path will be created in the dynamicPath Tac::Dir with 
   the suffix -dyn, and will contain the combined static CLI configuration
   along with the dynamic configuration (ex from dot1x VSA).
   """
   t0( "registerDyanmicConfigPath", staticPath, dynamicPath )
   h = handlerDir( em )
   static = EntityManager.cleanPath( staticPath )
   dynamic = EntityManager.cleanPath( dynamicPath )
   h.dynamicConfigPathTable.dynamicPath[ static ] = dynamic

class TemporaryConfigSession:
   """A context manager to do things in a config session.
   It exits from current session, creates and enters a new sessio (if a
   sessionName is passed), then aborts/exits, and re-enters the old session.

   If sessionName is None, do not create a temporary config session and only
   temporarily get out of the current session.
   """
   def __init__( self, em, sessionName,
                 action="abort",
                 description=None ):
      self.em_ = em
      self.oldSessionName_ = None
      self.oldPcc_ = None
      self.sessionName_ = sessionName
      self.action_ = action
      self.description_ = description
      assert action in ( "abort", "commit" )

   def __enter__( self ):
      sessionLock.acquire()
      oldSessionName = currentSession( self.em_ )
      if oldSessionName:
         self.oldPcc_ = sessionConfig.pendingChangeConfigDir.config.get(
            oldSessionName )
      if self.oldPcc_:
         # This has to happen before exiting the session
         self.oldPcc_.protectedCount += 1
         exitSession( self.em_ )
      if self.sessionName_:
         noCommit = ( self.action_ == 'abort' )
         enterSession( self.sessionName_, transient=True,
                       noCommit=noCommit,
                       description=self.description_ )
      return self

   def __exit__( self, typeArg, value, tb ):
      if self.sessionName_:
         if self.action_ == 'abort':
            abortSession( self.em_ )
         else:
            commitSession( self.em_ )
         exitSession( self.em_ )
      if self.oldPcc_:
         enterSession( self.oldPcc_.name, entityManager=self.em_ )
         # This has to happen after entering the session
         self.oldPcc_.protectedCount -= 1
      sessionLock.release()
      return False

def loadPlugins( entityManager ):
   t0( "load ConfigSessionPlugins" )
   Plugins.loadPlugins( 'ConfigSessionPlugin', entityManager )
