#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing

t0 = Tracing.trace0

dataStore_ = {}

class DataStore:
   """This class provides APIs to implement a parallel data store for config
   session. It's used by certain special commands to emulates config command
   behavior in a config session.

   WARNING: This is a very special mechanism. Do not use it without serious
   design justification.
   """
   def __init__( self ):
      pass

   def backup( self, mode, sessionName ):
      """Called before a config session is commited to backup
      current data for restore.

      Returns True for success, False for failure.
      """
      raise NotImplementedError

   def restore( self, mode, sessionName ):
      """Called when config session fails to commit and needs to
      be restored. Cannot fail."""
      raise NotImplementedError

   def commit( self, mode, sessionName ):
      """Called when a session is committed.

      Returns True for success, False for failure."""
      raise NotImplementedError

   def cleanup( self, mode, sessionName ):
      """Called when a session is aborted or deleted. Cannot fail.
      Note the mode parameter is unused (always None).
      """
      raise NotImplementedError

def registerDataStore( name, dataStore ):
   t0( "registering data store", name )
   dataStore_[ name ] = dataStore

def _invoke( mode, sessionName, functionName ):
   for name, store in dataStore_.items():
      t0( "invoking data store:", name, functionName )
      err = getattr( store, functionName )( mode, sessionName )
      if not err:
         t0( "data store function", functionName, "failed" )
         return False
   return True

def backup( mode, sessionName ):
   return _invoke( mode, sessionName, "backup" )

def restore( mode, sessionName ):
   err = _invoke( mode, sessionName, "restore" )
   assert err is True, "Cannot return error"

def commit( mode, sessionName ):
   return _invoke( mode, sessionName, "commit" )

def cleanup( sessionName ):
   err = _invoke( None, sessionName, "cleanup" )
   assert err is True, "Cannot return error"

